package pl.fabrykagier.eduFarma.gameData 
{
	import flashx.textLayout.property.BooleanProperty;
	import pl.fabrykagier.eduFarma.main.MainGameClass;
	import pl.fabrykagier.eduFarma.main.UserData;
	
	/**
	 * Holds information about products stored in a warehouse.
	 * @author Andrzej Kaczor
	 */
	public class Warehouse extends StructureData
	{
		private var _storedProducts:Vector.<Product> = new Vector.<Product>();
		private var _freeSpace:int = 0;
		private var _warehouseCapacity:int = 0;
		private var _storageType:String;
		private var structureLevelArray:Array = [];
		private var upgradeCostArray:Array = [];
		private var databaseIdArray:Array = [];
		private var _currentID:int = 0;
		
		/**
		 * Constructor. Creates the Warehouse object.
		 * @param	type - sets the type of the warehouse.
		 */
		public function Warehouse(type:String) 
		{
			storageType = type;
		}
		
		/**
		 * Used to expand the global storage space of warehouses when a new one is added.
		 * @param	additionalSpace - the amount of space to add to the global amount
		 */
		public function expandFreeSpace(additionalSpace:int):void
		{
			_freeSpace += additionalSpace;
		}
		
		/**
		 * Adds a product to the warehouse.
		 * @param	product - @see pl.fabrykagier.edufarma.gameData.Product
		 */
		override public function storeProduct(product:Product):void
		{
			// Call the warehouse extension on server to perform the same operation on server side
			if (MainGameClass.FLAG_dataRestored && !product.onMarket)
			{
				MainGameClass.getInstance.communicator.callExtension("buildingsExt", "updateBuildingProduct", {dbId:product.buildingId, userId:UserData.getInstance.userDbID, amount:0});
				MainGameClass.getInstance.communicator.callExtension("warehouseExt", "addProduct", {userDbId:UserData.getInstance.userDbID, productId:product.productId, amount:product.amount, type:product.type, buildingId:product.buildingId});
			}
			
			if (product.onMarket)
			{
				product.onMarket = false;
				product.price = 0;
			}
			
			// Cycle through stored products in search of the provided product
			for each (var object:Product in storedProducts)
			{
				// If that product is in the warehouse already
				if (object.product == product.product)
				{
					// increase it's amount
					object.amount += product.amount;
					
					// and decrease available free storage space
					_freeSpace -= product.amount;
					
					product = null;
					
					return;
				}
			}			
			
			// Add the new product to the stored products vector.
			storedProducts.push( product );
			// Reduce the available free storage space
			_freeSpace -= product.amount;
		}
		
		/**
		 * Removes a product from warehouse
		 * @param	product - the name of the product to remove
		 * @param	amount - the amount to remove
		 */
		public function removeProduct(product:String, amount:int = 0, buyingRound:Boolean=false):void
		{
						// Cycle through stored products in search of the provided product
			for each (var object:Product in storedProducts)
			{
				trace("PRODUCTS: ", object.product, product);
				// The product has been found
				if (object.product == product)
				{					
					trace("ON MARKET: ", object.onMarket, buyingRound);
					
					// If the product has been put on market, it cannot be removed. Terminate the action.
					if (object.onMarket && !buyingRound) return;
					
					if (buyingRound && !object.onMarket) return;
					
					// Call the warehouse extension on server to perform the same operation on server side
					MainGameClass.getInstance.communicator.callExtension("warehouseExt", "removeProduct", {userDbId:UserData.getInstance.userDbID, productId:object.productId, amount:amount});
										
					// Decrease the amount of product by the amount to remove.
					object.amount -= amount;
										
					// If the all pieces of the product have been removed
					if (object.amount <= 0)
					{
						// remove this product from the stored products vector
						storedProducts.splice(storedProducts.indexOf(object), 1);
					}
					
					// Increase the available free storage space.
					_freeSpace += amount;
					
					return;
				}
			}
		}
		
		/**
		 * Get the available free storage space.
		 * @return
		 */
		public function getFreeSpace():int
		{
			return freeSpace;
		}
		
		/**
		 * Get the amount of specified product stored in the warehouse.
		 * @param	product - the name of the product.
		 * @return
		 */
		public function getProductAmount(product:String):int
		{
			// Cycle through stored products
			for each (var object:Product in storedProducts)
			{
				// if there is a product matching the specified name
				if (object.product == product)
				{
					// return it's amount in warehouse
					return object.amount;
				}
			}
			
			// If there was no such product found - return an amount of 0.
			return 0;
		}
		
		public function getProductByName(productName:String):Product
		{
			for each (var object:Product in storedProducts)
			{
				// If that product is in the warehouse already
				if (object.product == productName)
				{					
					return storedProducts[storedProducts.indexOf(object)];
				}
			}
			
			return null;
		}
		
		public function isProductAvailable(productName:String, demand:int):Boolean
		{
			var product:Product = getProductByName(productName);
			
			if (product && product.amount >= demand) return true;
			
			return false;
		}
		
		/**
		 * Get the warehouse storage type.
		 */
		public function get storageType():String { return _storageType; }
		
		/**
		 * Set the warehouse storage type.
		 */
		public function set storageType(value:String):void 
		{
			_storageType = value;
		}
		
		/**
		 * Get the vector of products stored in the warehouse.
		 */
		public function get storedProducts():Vector.<Product> { return _storedProducts; }
		
		/**
		 * Set the vector of products stored in the warehouse.
		 */
		public function set storedProducts(value:Vector.<Product>):void 
		{
			_storedProducts = value;
		}
		
		public function get freeSpace():int
		{
			return _freeSpace - UserData.getInstance.marketplace.getProductTypeAmount(storageType);
		}
		
		public function set freeSpace(value:int):void
		{
			_freeSpace = value;
		}
		
		override public function get structureLevel():int { return structureLevelArray[currentID]; }
		
		override public function set structureLevel(value:int):void 
		{			
			structureLevelArray[currentID] = value;
		}
		
		override public function get upgradeCost():int { return upgradeCostArray[currentID]; }
		
		override public function set upgradeCost(value:int):void 
		{
			upgradeCostArray[currentID] = value;
		}
		
		override public function get dbID():int { return databaseIdArray[currentID]; }
		
		override public function set dbID(value:int):void
		{
			databaseIdArray[currentID] = value;
		}
		
		override public function get capacity():int { return _warehouseCapacity; }
		
		override public function set capacity(value:int):void 
		{
			_warehouseCapacity = value;
		}
		
		public function get currentID():int { return _currentID; }
		
		public function set currentID(value:int):void 
		{
			_currentID = value;
		}
	}

}