<?php

abstract class BaseModel
{
    private static $connection = null;
    private static $loadedModels = array();
    private static $modelsPath;
    private static $debugMode;


    public static function setup($host, $username, $password, $db, $modelsPath, $debugMode)
    {
        if (!BaseModel::$connection) {
            BaseModel::$connection = new PDO('mysql:dbname=' . $db . ';host=' . $host, $username, $password);
            BaseModel::$connection->query('SET NAMES "utf8"');
        }

        if (empty($modelsPath) || !is_dir($modelsPath) || !is_readable($modelsPath))
            throw new RuntimeException('BaseModel::setup(): models path must be a readable directory (specified: ' . $modelsPath . ')');

        BaseModel::$modelsPath  = $modelsPath;
        BaseModel::$debugMode   = $debugMode;
    }

    
    public static function load($modelName)
    {
        if (!BaseModel::$connection)
            throw new RuntimeException('$connection not set, call BaseModel::setup() first');

        $modelName = ucfirst($modelName);
        $modelPath = BaseModel::$modelsPath . $modelName . '.php';
        if (key_exists($modelPath, BaseModel::$loadedModels))
            return BaseModel::$loadedModels[$modelPath];
        
        if (!file_exists($modelPath) || !is_readable($modelPath))
            throw new RuntimeException('BaseModel::get(): ' . $modelPath . ' must exists and be readable');

        include_once $modelPath;

        $obj = new $modelName();

        BaseModel::$loadedModels[$modelPath] = &$obj;

        return $obj;
    }


    public function lastInsertId() {
        return BaseModel::$connection->lastInsertId ();
    }


	/**
	 * get query action :: return array of rows and cols
	 *
	 * @param string $syntax				MySql syntax
	 * @param array $params					array od params to bind
	 * @return array
	 */
	public function execute ($syntax = '', $params = array ())
	{
		$this->getData ($syntax, $params);

		return $this->query ? $this->query->fetchAll (PDO::FETCH_ASSOC) : false;
	} // end method query


	/**
	 * get single row action :: return array cols
	 *
	 * @param string $syntax				MySql syntax
	 * @param array $params					array od params to bind
	 * @return array
	 */
	public function executeGetRow ($syntax = '', $params = array ())
	{
		$this->getData ($syntax, $params);
		return $this->query ? $this->query->fetch (PDO::FETCH_ASSOC) : false;
	} // end method single


	/**
	 * get single row action :: return value of single row
	 *
	 * @param string $syntax				MySql syntax
	 * @param array $params					array od params to bind
	 * @return array
	 */
	public function executeGetValue ($syntax = '', $params = array ())
	{
		$this->getData ($syntax, $params);
		return $this->query && is_array ($data = $this->query->fetch (PDO::FETCH_ASSOC)) ? end ($data) : false;
	} // end method row


	/**
	 * main request for query/single/one action
	 *
	 * @param string $syntax				MySql syntax
	 * @param array $params					array od params to bind
	 * @return array
	 */
	private function getData ($syntax, $params)
	{
		if (empty ($syntax))
			return null;

		// query prepare
		$this->query = BaseModel::$connection->prepare ($syntax);

		// bind params
		if (!empty ($params))
		{
			foreach ($params AS $field => $value)
                $this->query->bindValue ($field, $value, PDO::PARAM_STR);
		}

		$success = $this->query->execute ();

        if (!$success && BaseModel::$debugMode === true)
        {
            echo $syntax . "\r\n" . implode(', ', $this->query->errorInfo()) . "\r\n";
            echo Api::getStacktrace();
            exit;
        }

		return (!$success || !(int)$this->query->rowCount ()) ? false : true;
	}
}
