<?php
    $basePath = getcwd() . '/../protected';

    include_once $basePath . '/config/global.php';
    include_once $basePath . '/system/GRequest.php';
    include_once $basePath . '/system/GResponse.php';
    include_once $basePath . '/system/Error.php';
    include_once $basePath . '/system/BaseModel.php';
    include_once $basePath . '/system/BaseExtension.php';


    class Api {
        protected $extensions = array();


        public function handle() {
            global $basePath;
            global $config;
            
            $methodsPath        = $basePath . '/methods/';
            $hostsConfigPath    = $basePath . '/config/hosts/';

            $hostConfigPath = $hostsConfigPath . $_SERVER['SERVER_NAME'] . '.php';

            if (!file_exists($hostConfigPath) || !is_readable($hostConfigPath))
                throw new RuntimeException($msg);

            //load host config
            include_once $hostConfigPath;

            if (!empty($hostConfig['environment']) && $hostConfig['environment'] == 'dev') {
                error_reporting(E_ALL);
                ini_set('display_errors', '1');
            }

            //merge config into single array
            $config = array_merge($hostConfig, $config);

            //prepare response
            $response = new GResponse();
            $response->data = $_POST;

            //JSON headers
            header('Cache-Control: no-cache, must-revalidate');
            header('Content-type: application/json');
                        
            //get method name
            $methodName = $_SERVER['REQUEST_URI'];
            $pos = strrpos($methodName, '/api/');
            if ($pos !== false)
                $methodName = substr($methodName, $pos + 5);
            $pos = strrpos($methodName, '?');
            if ($pos !== false)
                $methodName = substr($methodName, 0, $pos);

            $methodName = preg_replace("/[^a-zA-Z0-9\s]/", '', $methodName);
            $methodPath = $methodsPath . $methodName . '.php';

            //load extensions
            $this->loadExtensions($basePath . '/extensions/');

            //pre extensions
            $ret = $this->authMethod($methodName);
            $authPassed = false;
            if ($ret !== null) {
                //ok we can pass authorization to api
                $authPassed = true;
            }
            else {
                //try to authorize api

                //parse request
                $ret = $this->parseRequest($config['api_version'], $config['api_allowed_users']);
                if ($ret instanceof Error) {
                    $response->errorList[] = $ret;

                    return json_encode($response);
                }
            }

            //save request
            $request = $ret;

            //overwrite requestData from parsed request
            $response->requestData = $request->data;

            //validate token
            if (!$authPassed && !$this->isValidToken($request, $methodName, $config['api_token'])) {
                $response->errorList[] = new Error(Error::ERR_AUTH);

                return json_encode($response);
            }

            if (!file_exists($methodPath) || !is_readable($methodPath)) {
                $response->errorList[] = new Error(Error::ERR_METHOD);

                return json_encode($response);
            }

            try {
                //prepare BaseModel and connection
                BaseModel::setup(
                    $config['database']['host'],
                    $config['database']['username'],
                    $config['database']['password'],
                    $config['database']['db'],
                    $basePath . '/models/',
                    !empty($hostConfig['environment']) && $config['environment'] == 'dev'
                );
            }
            catch (Exception $x) {
                echo $x->getMessage();
                exit;
            }

            //pre extensions
            $error = $this->preMethod($methodName, $request);
            if ($error) {
                $response->errorList[] = $error;

                return json_encode($response);
            }

            //include and execute
            include_once $methodPath;
            if (!function_exists($methodName)) {
                $response->errorList[] = new Error(Error::ERR_METHOD);

                return json_encode($response);
            }

            $response->responseData = call_user_func($methodName, $request);
            if ($response->responseData instanceof Error)
            {
                //capture error from method
                $response->errorList[] = $response->responseData;

                //clean response data
                $response->responseData = null;
            }


            //post extensions
            $error = $this->postMethod($methodName, $request, $response);
            if ($error) {
                $response->errorList[] = $error;

                return json_encode($response);
            }

            return json_encode($response);
        }


        protected function isValidToken($request, $methodName, $api_token) {
            return ($request->api_token == md5($methodName . $api_token));
        }

        
        protected function parseRequest($api_version, $api_allowed_users) {
            if (empty($_POST['api_user']) ||
                empty($_POST['api_token']) ||
                empty($_POST['api_version']) ||
                empty($_POST['data']))
                return new Error(Error::ERR_REQUEST);

            if ($_POST['api_version'] != $api_version)
                return new Error(Error::ERR_VERSION);

            if (!in_array($_POST['api_user'], $api_allowed_users))
                return new Error(Error::ERR_AUTH);

            $_POST['data'] = stripslashes($_POST['data']);

            $request = new GRequest();

            $request->api_user      = $_POST['api_user'];
            $request->api_token     = $_POST['api_token'];
            $request->api_version   = $_POST['api_version'];
            $request->data          = json_decode($_POST['data'], 1);

            if ($request->data === null)
                return new Error(Error::ERR_JSON);

            return $request;
        }


        protected function loadExtensions($path)
        {
            if (!is_dir($path) || !is_readable($path))
                return;

            $needlePhp = '.php';
            $needleExtension = 'Extension';

            foreach (scandir($path) as $item)
            {
                if (substr($item, -strlen($needlePhp)) == $needlePhp)
                {
                    //php file, include
                    include_once $path . $item;

                    $className = str_replace('.php', '', $item);

                    //extension
                    if (substr($className, -strlen($needleExtension)) == $needleExtension)
                        $this->extensions[] = &new $className();
                }
            }
        }


        protected function authMethod($method)
        {
            foreach ($this->extensions as $ext)
            {
                $ret = $ext->authMethod($method);
                if ($ret !== null)
                    return $ret;
            }

            return null;
        }


        protected function preMethod($method, &$request)
        {
            foreach ($this->extensions as $ext)
            {
                $error = $ext->preMethod($method, $request);
                if ($error)
                    return $error;
            }
        }

        
        protected function postMethod($method, &$request, &$response)
        {
            foreach ($this->extensions as $ext)
            {
                $error = $ext->postMethod($method, $request, $response);
                if ($error)
                    return $error;
            }
        }


        public static function getStacktrace() {
            $backtracel = '';

            foreach (debug_backtrace() as $k => $v) {
                if ($v['function'] == "include" || $v['function'] == "include_once" || $v['function'] == "require_once" || $v['function'] == "require") {
                    $backtracel .= "#" . $k . " " . $v['function'] . "(" . $v['args'][0] . ") called at [" . (!empty($v['file']) ? $v['file'] : '') . ":" . $v['line'] . "]\r\n";
                } else {
                    $backtracel .= "#" . $k . " " . $v['function'] . "() called at [" . $v['file'] . ":" . $v['line'] . "]\r\n";
                }
            }
        }
    }


    //service
    $api = new Api();
    echo $api->handle();
