<?php

class AclException extends Exception { }

/**
 * Action control list
 * 
 * @package system
 * @author dawid kraczkowski <lunereaper>
 */
class Acl
{
    /**
     * @param array
     */
    private $aclData;
    /**
     * @param array
     */
    private $aclFileRoles = array();
    
    private $userRoles = array();
    
    private $cachedQueries = array();

    /**
     * initialize Acl
     * @param string $aclFile acl configuration file
     */
    public function __construct ($aclData)
    {
        $this->aclData = $aclData;
        $this->parseAclData($aclData);
        $this->addRole('default');
    }
    
    protected function parseAclData ($aclData)
    {
        //register roles
        foreach($aclData as $roleData => $access)
        {
            $roleData = explode('.', $roleData);
            $roleName = $roleData[0];
            array_shift($roleData);
            if(!key_exists($roleName, $this->aclFileRoles))
                $this->aclFileRoles[$roleName] = new Role($roleName);

            call_user_func_array
            (
                array($this->aclFileRoles[$roleName], 'addRestriction'), 
                array_merge(array($access), $roleData)
            );
        }
    }

    /**
     * Add role
     * @param string $roleName
     */
    public function addRole ($roleName)
    {
        if (key_exists($roleName, $this->aclFileRoles) && !key_exists($roleName, $this->userRoles))
        {
            $this->userRoles[$roleName] = $this->aclFileRoles[$roleName];
            $this->flushCache();
        }
        return $this;
    }

    /**
     * Remove role
     * @param string $roleName
     */
    public function removeRole ($roleName)
    {
       if (key_exists($roleName, $this->userRoles))
       {
           unset($this->userRoles[$roleName]);
           $this->flushCache();
       }
       return $this;
    }
    
    public function flushCache ()
    {
        $this->cachedQueries = array();
        return $this;
    }

    /**
     * Check if restrictions to given resource exists
     * @param string $controller controller name
     * [@param string $action action name]
     * @return bool
     */
    public function hasRole ($roleName)
    {
        return key_exists($roleName, $this->userRoles);
    }

    /**
     * Check if user has access to given resource
     * @param string $controller
     * @param string $action
     * @return bool
     */
    public function hasAccess ($resourcePoll, $resource = null)
    {
        $searchKey = $resourcePoll . '.' . $resource;
        
        if (key_exists($searchKey, $this->cachedQueries))
            return $this->cachedQueries[$searchKey];
        
        foreach ($this->userRoles as $roleName => $role)
        {
            if ($role->hasAccess($resourcePoll, $resource))
            {
                $this->cachedQueries[$searchKey] = true;
                return true;
            }
        }
        
        $this->cachedQueries[$searchKey] = false;
        return false;
    }
}