<?php

class ProfilerException extends Exception { }

/**
 * Profiler class
 * 
 * This class process profilling logs. Initialize this class at
 * start of your application. And enclosure profiled code between
 * two methods Profiler::start() and Profiler::stop()
 * 
 * @exmaple
 * <code>
 * <?php
 * Profiler::initialize();
 * 
 * Profiler::start($forkId,'sample tag');
 * //do something here
 * Profiler::stop($forkId);
 * ?>
 * </code>
 *
 */
class Profiler
{
    /**
     * Log array
     * @var array
     */
    private static $log;
    /**
     * Temporary log stack
     * @var array
     */
    private static $stack;
    /**
     * Init data
     * @var array
     */
    private static $init;

    private function __construct ()
    {
        
    }

    /**
     * Initialize profiler
     * @return null
     */
    public static function initialize ()
    {
        self::$init['mem_used'] = memory_get_peak_usage();
        self::$init['time'] = microtime(true);
    }

    /**
     * Log current status
     * @param string $tag status' tag
     * @return null
     */
    public static function logStatus ($tag = null)
    {
        $status = self::getStatus();
        $tag ? $status['tag'] = $tag : false;
        self::$log[] = $status;
    }

    /**
     * Get current status
     * @return array
     */
    public static function getStatus ()
    {
        $status['overall_mem_used'] = self::getReadableSize(memory_get_peak_usage() - self::$init['mem_used']);
        $status['overall_exec_time'] = self::getReadableTime(microtime(true) - self::$init['time']);
        $status['at_exec_time'] = self::getReadableTime(microtime(true) - self::$init['time']);
        $includedFiles = get_included_files();
        foreach ($includedFiles as $file)
        {
            $status['included_files'][] = array('file' => $file, 'size' => self::getReadableSize(filesize($file)));
        }
        $backtrace = debug_backtrace();
        $struct = '';
        if ($backtrace[1])
        {
            if ($backtrace[1]['class'] == __CLASS__)
            {
                if ($backtrace[2])
                {
                    if (!$backtrace[2]['class'])
                        $struct = 'function: ';
                    $status['called_in'] = $struct . $backtrace[2]['class'] . $backtrace[2]['type'] . $backtrace[2]['function'];
                    $status['args'] = $backtrace[2]['args'];
                }
                else
                {
                    $status['called_in'] = 'file: ' . $backtrace[0]['file'] . ' at line ' . $backtrace[1]['line'];
                }
            }
            else
            {
                if (!$backtrace[1]['class'])
                    $struct = 'function: ';
                $status['called_in'] = $struct . $backtrace[1]['class'] . $backtrace[1]['type'] . $backtrace[1]['function'];
                $status['args'] = isset($backtrace[2]['args']) ? $backtrace[2]['args'] : null;
            }
        }
        else
        {
            $status['called_in'] = 'file: ' . $backtrace[0]['file'] . ' at line ' . $backtrace[0]['line'];
        }
        return $status;
    }

    /**
     * Start profiling the fork
     * @example
     * <code>
     * <?php
     * Profiler::start(1,'start');
     * //put here code to be profiled
     * Profiler::stop(1);
     * </code>
     * 
     * @param mixed $forkId unique id for start and stop fork
     * @param string $tag tag name for fork
     * @return null
     */
    public function start ($forkId, $tag = null)
    {
        self::$stack[$forkId]['backtrace'] = debug_backtrace();
        self::$stack[$forkId]['mem_used'] = memory_get_peak_usage();
        self::$stack[$forkId]['time'] = microtime(true);
        self::$stack[$forkId]['tag'] = $tag;
    }

    /**
     * Profile the closure x times
     */
    public static function test (Closure $closure, $times = null)
    {
        $backtrace = debug_backtrace();
        $memory = memory_get_peak_usage();
        $microtime = microtime(true);
        if ($times && $times > 1)
        {
            for ($i = 0; $i < $times; $i++)
                $closure();
        }else
        {
            $closure();
        }
        $tmp['mem_used'] = self::getReadableSize(memory_get_peak_usage() - $memory);
        $tmp['exec_time'] = self::getReadableTime(microtime(true) - $microtime);
        $tmp['at_exec_time'] = self::getReadableTime(microtime(true) - self::$init['time']);
        $tmp['backtrace'] = $backtrace;
        //add data to log
        self::$log[] = $tmp;
        return $tmp;
    }

    /**
     * Stop profiling the fork
     * @param mixed $forkId started fork's id
     * @return array fork info
     */
    public static function stop ($forkId)
    {
        $currentBackTrace = debug_backtrace();
        $startBackTrace = self::$stack[$forkId]['backtrace'];
        $tmp['mem_used'] = self::getReadableSize(memory_get_peak_usage() - self::$stack[$forkId]['mem_used']);
        $tmp['exec_time'] = self::getReadableTime(microtime(true) - self::$stack[$forkId]['time']);
        $tmp['at_exec_time'] = self::getReadableTime(microtime(true) - self::$init['time']);

        if ($currentBackTrace[0]['file'] == $startBackTrace[0]['file'])
        {
            $tmp['file'] = $currentBackTrace[0]['file'];
            $tmp['start_line'] = $startBackTrace[0]['line'];
            $tmp['end_line'] = $currentBackTrace[0]['line'];
        }
        if ($startBackTrace[1])
        {
            $struct = '';
            if (!$startBackTrace[1]['class'])
                $struct = 'function: ';
            $tmp['called_in'] = $struct . $startBackTrace[1]['class'] . $startBackTrace[1]['type'] . $startBackTrace[1]['function'];
        }
        else
        {
            $tmp['called_in'] = 'file: ' . $startBackTrace[0]['file'] . ' at line ' . $startBackTrace[0]['line'];
        }

        //add data to log
        self::$log[] = $tmp;

        return $tmp;
    }

    /**
     * Get information about included file
     */
    public static function getIncludedFileInfo ()
    {
        $files = get_included_files();
        $fileList = array();
        $total = array
          (
          "count" => count($files),
          "size" => 0,
          "largest" => 0,
        );
        if (!$files)
            return;
        foreach ($files as $file)
        {
            $size = filesize($file);
            $fileList[] = array
              (
              'name' => $file,
              'size' => self::getReadableSize($size)
            );
            $total['size'] += $size;
            if ($size > $largest)
            {
                $total['largest'] = $file;
                $largest = $size;
            }
        }
        $total['size'] = self::getReadableSize($total['size']);
        return array('files_info' => $fileList, 'total' => $total);
    }

    /**
     * Get file/memory size in readable format
     * @param int $size file size
     * @param string $format
     * @return string 
     */
    protected static function getReadableSize ($size, $format = '%01.2f %s')
    {
        $sizes = array('bytes', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
        $lastsizestring = end($sizes);

        foreach ($sizes as $sizestring)
        {
            if ($size < 1024)
                break;
            if ($sizestring != $lastsizestring)
                $size /= 1024;
        }
        if ($sizestring == $sizes[0])
            $format = '%01d %s'; // Bytes aren't normally fractional
        return sprintf($format, $size, $sizestring);
    }

    /**
     * Get microtime in readable format
     * @param int $microtime microtime
     * @param string $format
     * @return string 
     */
    protected static function getReadableTime ($microtime, $format = '%.5f %s')
    {
        $formats = array('ms', 's', 'm');
        $si = 0;
        $time = $microtime;

        if ($microtime >= 1 && $microtime < 6000)
        {
            $si = 1;
            $time = $microtime;
        }
        if ($microtime >= 6000)
        {
            $si = 2;
            $time = $microtime / 60;
        }
        return sprintf($format, $time, $formats[$si]);
    }

    /**
     * Get profiler log
     * @return array
     */
    public function getLog ()
    {
        return self::$log;
    }

    /**
     * Append results of profiling as json file
     * @param string $filename filename
     * @param bool $append append profiler data to file
     */
    public function saveResults ($filename, $append = true)
    {
        if (!is_writable($filename))
            throw new ProfilerException('File: ' . $filename . ' is not writeable');
    }
}