<?php

/**
 * Session Class
 * Handle sessions
 * 
 * @package system
 * @author dawid kraczkowski <lunereaper>
 */
class SessionException extends ApplicationException { }

class Session
{
    /**
     * Is session started
     * @param boolean
     */
    private static $sessionStarted = false;
    /**
     * Session storage handler
     * @param system\session\IStorage
     */
    private static $sessionStorage;

    /**
     * Prevent from instancinizig
     */
    protected function __construct ()
    {
        
    }

    /**
     * Destroy current session
     */
    public static function destroy ()
    {
        self::$sessionStarted = false;
        session_destroy();
    }

    /**
     * Start session
     * @throws system\SessionException
     */
    public static function start ()
    {
        if (self::$sessionStarted)
            throw new SessionException('The session has already been started. You cannot start session twice.');
        if (headers_sent())
            throw new SessionException('Headers already sent. Call ' . __CLASS__ . '::' . __FUNCTION__ . '() before sending any output to the browser');
        session_start();
        self::$sessionStarted = true;
    }

    /**
     * Set session ID
     * @throw system\SessionException
     * @return int session ID
     */
    public static function setId ($sessionId)
    {
        if (self::$sessionStarted)
            throw new SessionException('The session has already been started. The session id must be set first.');
        if (headers_sent())
            throw new SessionException('Headers already sent. Call' . __CLASS__ . '::' . __FUNCTION__ . '() before any output has been sent to the browser');
        session_id($sessionId);
    }

    /**
     * Set session storage
     * @param system\session\IStorage session storage object
     */
    public static function setStorage (IStorage $sessionStorage)
    {
        self::$sessionStorage = $sessionStorage;
    }

    /**
     * Get session storage
     * @return system\session\IStorage session storage object
     */
    public static function getStorage ()
    {
        return self::$sessionStorage;
    }

    /**
     * Register new session namesapce
     * @param string $namespaceName
     * @return system\session\SessionNamespace
     */
    public static function registerNamespace ($namespaceName)
    {
        if (!key_exists($namespaceName, (array) @$_SESSION[SESSION_NAMESPACE]))
        {
            $_SESSION[SESSION_NAMESPACE][$namespaceName] = new SessionNamespace();
        }
        return $_SESSION[SESSION_NAMESPACE][$namespaceName];
    }

    /**
     * Check if namespace exists
     * @return boolean
     */
    public static function namespaceExists ($namespaceName)
    {
        return $_SESSION[SESSION_NAMESPACE][$namespaceName] instanceof SessionNamespace;
    }

    /**
     * Get existing session namespace
     * @param string $namespaceName
     * @return system\session\SessionNamespace
     */
    public static function getNamespace ($namespaceName)
    {
        return $_SESSION[SESSION_NAMESPACE][$namespaceName];
    }

    /**
     * Set session lifetime
     * @see Session::rememberUntill
     */
    public static function setExpirationTime ($seconds = 0)
    {
        self::rememberUntill($seconds);
    }

    /**
     * Set session lifetime
     */
    protected static function rememberUntill ($seconds)
    {
        $cookie = session_get_cookie_params();
        session_set_cookie_params($seconds, $cookie['path'], $cookie['domain'], $cookie['secure']);
    }
}