<?php

sfContext::getInstance()->getConfiguration()->loadHelpers(array('View'));

/**
 * Klasa zarzadzania widokiem dla gry Hotel
 */
class HotelGameView extends BaseGameView {

  const STATUS_NONE = 0;
  const STATUS_NEW = 1;
  const STATUS_INACTIVE = 2;
  
  /**
   * Tablica zawierajaca wszystkie zakladki dla wszystkich budynkow
   * z podzialem typu: Budynek -> Pomieszczenie -> Zakladka.
   */
  public $allSquares;
  public $buildings;
  public $squares;

  public function __construct(BaseGameController $gameController) {
    parent::__construct($gameController);
    $this->initialize();
  }
  
  public static function getLogoImageAbsolutePath(){
    return sfConfig::get('sf_upload_dir') . '/' . sfConfig::get('app_game_logo_path', 'game/logo/');
  }
  
  public static function getLogoImagePath(){
    return '/uploads/' . sfConfig::get('app_game_logo_path', 'game/logo/');
  }

  protected function initialize() {
    $this->buildings = sfConfig::get('squares_buildings', array());
    $this->squares = sfConfig::get('squares_squares', array());
  }
  
  public function prepareBuildingsAndSquares() {
    $list = array();

    foreach ($this->buildings as $building => $data) {
      $squares = $this->prepareSquaresList($building);

      // Budynek nie jest pusty wiec jest dostepny
      $list[$building] = $data;
      $list[$building]['squares'] = $squares;
    }

    $round = $this->getAll('round');
    $round['view'] = $list;
    $this->setAll($round, 'round');
  }
  
  /**
   * Funkcja zwaraca dostepne pomieszczenia dla podanego w parametrze budynku;
   *
   * @param $edifice, nazwa budynku, dla którego zostaną zwrócone pomieszczenia.
   * @return array, lista pomieszczeń
   * @throws sfException, niepoprawny parametr (pomieszczenie)
   */
  public function prepareSquaresList($edifice) {

    if (isset($this->squares[$edifice])) {
      $squares = array();

      foreach ($this->squares[$edifice] as $name => $square) {
        $prepareFunction = 'prepareTab' . sfInflector::camelize($square['uniqueName']);
        $widgets = $this->gameController->$prepareFunction();
        $squares[$name] = $square;
        
        if (count($widgets) > 0) {
          // Zakladka nie jest pusta wiec jest dostepna
          // Pamietamy ze z R1 -> R2 jestesmy w R2
          if ($this->rGT(1)) {
            $oldWidgets = $this->g('view|'.$edifice.'|squares|'.$name.'|widgets', 'last_round', array());
          } else {
            $oldWidgets = array();
          }
          
          if ($name == 'historia_projektow') {
            // W przypadku Historii Projektow nie powinismy dostawac informacji o nowym widgetcie
            $squares[$name]['widgets'] = $this->prepareWidgetsList($widgets, $oldWidgets, self::STATUS_NONE);
          } else {
            $squares[$name]['widgets'] = $this->prepareWidgetsList($widgets, $oldWidgets);
          }
        } else {
          $squares[$name]['widgets'] = array();
        }
      }
      
      return $squares;
    } else {
      throw new sfException('Nie zdefiniowano pomieszczeń dla budynku : ' . $edifice);
    }
  }
  
  public function prepareWidgetsList($widgets, $oldWidgets, $newStatus = self::STATUS_NEW) {
    $list = array();
    
    foreach ($widgets as $widget) {
      foreach ($this->getGameData()->getTeam()->getUserHasTeams() as $record) {
        if (isset($oldWidgets[$widget]['status'][$record->getUserId()]) &&
                $oldWidgets[$widget]['status'][$record->getUserId()] == self::STATUS_NONE) {
          $list[$widget]['status'][$record->getUserId()] = self::STATUS_NONE;
        } else {
          if ($widget == 'badanie_marketingowe_popyt_standard') {
            // Pierwsze badania juz podjelismy nie trzeba ich ponownie podswietlac
            $list[$widget]['status'][$record->getUserId()] = self::STATUS_NONE;
          } else {
            $list[$widget]['status'][$record->getUserId()] = $newStatus;
          }
        }
      }
    }
    
    return $list;
  }
  
  public static function getSquareStatus($data, $userId) {
    $status = self::STATUS_NONE;
    if (count($data['widgets']) == 0) {
      $status = self::STATUS_INACTIVE;
    } else {
      foreach ($data['widgets'] as $widget) {
        
        if (isset($widget['status'][$userId]) && $widget['status'][$userId] == self::STATUS_NEW) {
          $status = self::STATUS_NEW;
          break;
        }
      }
    }

    return $status;
  }
  
  public static function getBuildingStatus($data, $userId) {
    $status = self::STATUS_INACTIVE;
    
    foreach ($data['squares'] as $square) {
      $squareStatus = self::getSquareStatus($square, $userId);
      
      if ($squareStatus == self::STATUS_NONE) {
        $status = self::STATUS_NONE;
      }
      
      if ($squareStatus == self::STATUS_NEW) {
        $status = self::STATUS_NEW;
        break;
      }
    }
    
    return $status;
  }
  
  public function setWidgetsStatus($widgets, $tabUniqueName, $userId) {
    $view = $this->g('view', 'round', array());
    $building = false;
    $square = false;
    
    foreach ($view as $buildingName => $buildingData) {
      
      foreach ($buildingData['squares'] as $squareName => $squareData) {
        if ($buildingName.'_'.$squareName ==  $tabUniqueName) {
          // Szukamy budynku zakladki
          $building = $buildingName;
          $square = $squareName;
          break;
        }
      }
    }
    
    if ($building && $square) {
      foreach ($widgets as $widget) {
        $view[$building]['squares'][$square]['widgets'][$widget]['status'][$userId] = self::STATUS_NONE;
      }
    }
    
    $this->s('view', $view, 'round');
    $this->getGameData()->save(true);
  }
  
  /**
   * Zwraca liste budynkow
   * @return array 
   */
  public function getBuildingsList() {
    return $this->g('view', 'round', array());
  }
  
  /**
   * Zwraca liste kafelkow
   * @return array 
   */
  public function getSquaresList($edifice) {
    return $this->g('view|'.$edifice, 'round', array());
  }
  
  /**
   * Sprawdza czy dany widget jest dostepny w tej rundzie
   * Uzywany przy generowaniu raportow
   * @param string $widget
   * @return boolean 
   */
  public function isWidgetAvailable($widget) {
    $view = $this->g('view', 'round', array());
    $available = false;
    
    foreach ($view as $elements) {
      foreach ($elements['squares'] as $square) {
        if (isset($square['widgets'][$widget])) {
          $available = true;
          break;
        }
      }
    }
    
    return $available;
  }

################  FUNKCJE ZWRACAJĄCE INFORMACJĘ DO POLA FORMULARZA ################

  public static function getCostDiv($value, $id = false) {

    if ($id) {
      return '<span class="hiden" widgetCost="' . $value . '"  id="' . $id . '" ></span>';
    }

    return '<span class="hiden" widgetCost="' . $value . '"></span>';
  }

  public static function getCostRoundDiv($value, $id = false) {

    if ($id) {
      return '<span class="hiden" widgetCost="' . $value . '"  id="' . $id . '" ></span>';
    }

    return '<span class="hiden" widgetCostRound="' . $value . '"></span>';
  }

  /** Funckje ogolne do zwracania kosztow */

  /**
   * Funkcja zwraca zawsze jedna translacje z kosztem stalym 
   * oraz dorzuca diva z kosztem decyzji.
   * Wykorzystywany glownie do widgetow TAK/NIE
   * 
   * @param HotelGameController $gc
   * @param type $value
   * @param string $name
   * @param float $koszt
   * @return string 
   */
  public static function kosztJednorazowyStaly($gc, $value, $name, $koszt) {
    $doZaplaty = $value * $koszt;

    return self::getCostDiv($doZaplaty)
            . __t($name . '_info', array('%koszt%' => format_money_value($koszt)), 'game_info');
  }

  /**
   * Funkcja zwraca translacje w zaleznosci od poniesionego kosztu (trzy przypadki)
   * oraz dorzuca diva z kosztem decyzji.
   * Wykorzystywany glownie do prostych widgetow wyboru (event sadzenia, panele, kierowniczka)
   * 
   * @param HotelGameController $gc
   * @param type $value
   * @param string $name
   * @param float $koszt
   * @return string 
   */
  public static function kosztJednorazowyZmienny($gc, $value, $name, $koszt) {
    if ($koszt == 0) {
      return self::getCostDiv($koszt) . __t($name . '_info_stan', array('%koszt%' => self::format_money_value($koszt)), 'game_info');
    } elseif ($koszt < 0) {
      return self::getCostDiv($koszt) . __t($name . '_info_zysk', array('%koszt%' => self::format_money_value(-$koszt)), 'game_info');
    } else {
      return self::getCostDiv($koszt) . __t($name . '_info_koszt', array('%koszt%' => self::format_money_value($koszt)), 'game_info');
    }
  }

  /**
   * 
   * @param HotelGameController $gc
   * @param type $value
   * @return string
   */
  public static function getInfoPranieOutsourcingStandardKwota(HotelGameController $gc, $value, $name) {

    $kss = $gc->g('koszty_bazowe|pokoje_nieoblozone|pranie', 'game');
    $min_koszt = $kss[$gc->gL()] * $gc->gLPS();

    $ksso = $gc->g('koszty_bazowe|pokoje_oblozone|pranie', 'game');
    $max_koszt = ($min_koszt + $ksso[$gc->gL()] * $gc->gLPS()) * $gc->g('pranie_mnoznik', 'game');

    $infoTranslation = $name;
    $koszt = $value;
    if ($gc->getGameProcess()->wspEventPralniaEkologiczna() == 1) {
      $infoTranslation = $name.'_disabled';
      $koszt = 0;
    }

    return self::getCostRoundDiv($koszt) . __t($infoTranslation . '_info', array('%min_koszt%' => self::format_money_value($min_koszt), '%max_koszt%' => self::format_money_value($max_koszt), '%koszt%' => self::format_money_value($value)), 'game_info');
  }

  public static function getInfoPranieOutsourcingLuxKwota(HotelGameController $gc, $value, $name) {
    $kss = $gc->g('koszty_bazowe|pokoje_nieoblozone|pranie', 'game');
    $min_koszt = $kss[$gc->gL()] * $gc->gLPL() * $gc->getGameProcess()->getWspLux();

    $ksso = $gc->g('koszty_bazowe|pokoje_oblozone|pranie', 'game');
    $max_koszt = ($min_koszt + $ksso[$gc->gL()] * $gc->gLPL()) * $gc->getGameProcess()->getWspLux() * $gc->g('pranie_mnoznik', 'game');
    
    $infoTranslation = $name;
    $koszt = $value;
    if ($gc->getGameProcess()->wspEventPralniaEkologiczna() == 1) {
      $infoTranslation = $name.'_disabled';
      $koszt = 0;
    }

    return self::getCostRoundDiv($koszt) . __t($infoTranslation . '_info', array('%min_koszt%' => self::format_money_value($min_koszt), '%max_koszt%' => self::format_money_value($max_koszt), '%koszt%' => self::format_money_value($value)), 'game_info');
  }

  public static function getInfoSprzatanieOutsourcingStandardKwota(HotelGameController $gc, $value, $name) {
    $kss = $gc->g('koszty_bazowe|pokoje_nieoblozone|sprzatanie', 'game');
    $min_koszt = $kss[$gc->gL()] * $gc->gLPS();

    $ksso = $gc->g('koszty_bazowe|pokoje_oblozone|sprzatanie', 'game');
    $max_koszt = ($min_koszt + $ksso[$gc->gL()] * $gc->gLPS()) * $gc->g('sprzatanie_mnoznik', 'game');
    
    $infoTranslation = $name;
    $koszt = $value;
    if ($gc->gPO() > 0) {
      $infoTranslation = $name.'_disabled';
      $koszt = 0;
    }

    return self::getCostRoundDiv($koszt) . __t($infoTranslation . '_info', array('%min_koszt%' => self::format_money_value($min_koszt), '%max_koszt%' => self::format_money_value($max_koszt), '%koszt%' => self::format_money_value($value)), 'game_info');
  }

  public static function getInfoSprzatanieOutsourcingLuxKwota(HotelGameController $gc, $value, $name) {
    $kss = $gc->g('koszty_bazowe|pokoje_nieoblozone|sprzatanie', 'game');
    $min_koszt = $kss[$gc->gL()] * $gc->gLPL();

    $ksso = $gc->g('koszty_bazowe|pokoje_oblozone|sprzatanie', 'game');
    $max_koszt = ($min_koszt + $ksso[$gc->gL()] * $gc->gLPL()) * $gc->g('sprzatanie_mnoznik', 'game');
    
    $infoTranslation = $name;
    $koszt = $value;
    if ($gc->gPO() > 0) {
      $infoTranslation = $name.'_disabled';
      $koszt = 0;
    }

    return self::getCostRoundDiv($koszt) . __t($infoTranslation . '_info', array('%min_koszt%' => self::format_money_value($min_koszt), '%max_koszt%' => self::format_money_value($max_koszt), '%koszt%' => self::format_money_value($value)), 'game_info');
  }

  public static function getInfoSniadaniaKwota(HotelGameController $gc, $value, $name) {
    $koszt = 0;
    if ($gc->rGT(1) && $value > 0) {
      $kosztZaPokoj = $gc->g('sniadania|koszt_za_pokoj', 'game');
      $iloscPokoi = $gc->gLPS() + $gc->gLPL();
      $koszt = (($gc->g('pokoje_oblozone_standard', 'last_round') + $gc->g('pokoje_oblozone_lux', 'last_round')) * $value + ($iloscPokoi) * $kosztZaPokoj) * sfConfig::get('game_hotel_dni_w_kwartale', 90);
    }

    return __t($name . '_info', array('%koszt%' => self::format_money_value($koszt), '%cena%' => $value), 'game_info');
  }
//
//  public static function getInfoCenaPokojeStandard(HotelGameController $gc, $value, $name) {
//    return __t($name . '_info_' . $gc->gL(), array(), 'game_info');
//  }
//
//  public static function getInfoCenaPokojeLux(HotelGameController $gc, $value, $name) {
//    return __t($name . '_info_' . $gc->gL(), array(), 'game_info');
//  }

  /** Bank - Lokata */
  public static function getInfoLokataKwota(HotelGameController $gc, $value, $name) {
    $kwota = $gc->g('lokata_kwota', 'round');
    $translacja = $name;

    if ($value == 0) {
      $translacja .= '_stan';
    } else {
      $translacja .= '_koszt';
    }

    $lokata = $kwota + $value;
    $rata = $gc->getGameProcess()->getLokataOdsetki($lokata);

    return self::getCostDiv($value) . 
            __t($translacja . '_info', array(
        '%kwota%' => self::format_money_value($lokata)
        , '%rata%' => self::format_money_value($rata)
        , '%oprocentowanie%' => $gc->getGameProcess()->getLokataOdsetkiRoczne($lokata) * 100
        , '%value%' => self::format_money_value($value))
            , 'game_info');
  }

  public static function getInfoLokataSplata(HotelGameController $gc, $value, $name) {

    $kwota = $gc->g('lokata_kwota', 'round');
    $translacja = $name;

    if ($value == 0) {
      $translacja .= '_stan';
    } else {
      $translacja .= '_zysk';
    }

    $lokata = max(0, $kwota - $value);
    $rata = $gc->getGameProcess()->getLokataOdsetki($lokata);
    
    return self::getCostDiv(-$value) . 
            __t($translacja . '_info', array(
        '%kwota%' => self::format_money_value($lokata)
        , '%rata%' => self::format_money_value($rata)
        , '%oprocentowanie%' => $gc->getGameProcess()->getLokataOdsetkiRoczne($lokata) * 100
        , '%value%' => self::format_money_value($value))
            , 'game_info');
  }

  /** Bank - Kredyty */
//  public static function getInfoKredytKarnySplata(HotelGameController $gc, $value, $name) {
//    $kredytKarny = $gc->g('kredyt_karny_kwota', 'round');
//    $doSplaty = $value * ($gc->g('kredyty|karny|odsetki', 'game') / 4);
//
//    if ($value > $kredytKarny) {
//      $pozostaleOdsetki = 0;
//      $resztaDoSplaty = 0;
//      $doSplaty = $kredytKarny * ($gc->g('kredyty|karny|odsetki', 'game') / 4);
//    } else {
//      $pozostaleOdsetki = ($kredytKarny - $value) * ($gc->g('kredyty|karny|odsetki', 'game') / 4);
//      $resztaDoSplaty = $kredytKarny - $value;
//    }
//    return self::getCostDiv($kredytKarny - $value)
//            . __t($name . '_info', array('%pozostaleOdsetki%' => self::format_money_value($pozostaleOdsetki), '%calkowitaKwota%' => self::format_money_value($kredytKarny), '%splata%' => self::format_money_value($resztaDoSplaty), '%doSplaty%' => self::format_money_value($doSplaty), '%pozostalaKwota%' => self::format_money_value($value)), 'game_info');
//  }

  public static function getInfoKredytObrotowyKwota(HotelGameController $gc, $value, $name) {
    $kwota = $gc->g('kredyt_obrotowy_kwota', 'round');
    $translacja = $name;

    if ($value == 0) {
      $translacja .= '_stan';
    } else {
      $translacja .= '_koszt';
    }

    $zdolnosc = $gc->getGameProcess()->getZdolnoscKredytObrotowy();
    $kredyt = $kwota + $value;
    $rata = $gc->getGameProcess()->getKredytObrotowyRata($kredyt);

    return self::getCostDiv(-$value) .
            __t($translacja . '_info', array(
                '%zdolnosc%' => self::format_money_value($zdolnosc)
                , '%kwota%' => self::format_money_value($kredyt)
                , '%rata%' => self::format_money_value($rata)
                , '%oprocentowanie%' => $gc->getGameProcess()->getKredytObrotowyOprocentowanie($kredyt) * 100
                , '%value%' => self::format_money_value($value))
                    , 'game_info');
  }

  public static function getInfoKredytObrotowySplata(HotelGameController $gc, $value, $name) {

    $kwota = $gc->g('kredyt_obrotowy_kwota', 'round');
    $translacja = $name;

    if ($value == 0) {
      $translacja .= '_stan';
    } else {
      $translacja .= '_zysk';
    }

    $kredyt = max(0, $kwota - $value);
    $rata = $gc->getGameProcess()->getKredytObrotowyRata($kredyt);

    return self::getCostDiv($value) .
            __t($translacja . '_info', array(
                '%kwota%' => self::format_money_value($kredyt)
                , '%rata%' => self::format_money_value($rata)
                , '%oprocentowanie%' => $gc->getGameProcess()->getKredytObrotowyOprocentowanie($kredyt) * 100
                , '%value%' => self::format_money_value($value))
                    , 'game_info');
  }

  public static function getInfoKredytInwestycyjnyKwota(HotelGameController $gc, $value, $name) {
    $translacja = $name;

    $zdolnosc = $gc->getGameProcess()->getZdolnoscKredytInwestycyjny();
    $rata = $gc->getGameProcess()->getKredytInwestycyjnyRata($value);
    $roznica = $gc->g('kredyt_inwestycyjny_kwota', 'decision') - $value;

    if ($roznica == 0) {
      $translacja .= '_stan';
    } elseif ($roznica > 0) {
      $translacja .= '_zysk';
    } else {
      $translacja .= '_koszt';
    }

    return self::getCostDiv($roznica) .
            __t($translacja . '_info'
                    , array('%zdolnosc%' => self::format_money_value($zdolnosc)
                , '%kwota%' => self::format_money_value($value)
                , '%roznica%' => self::format_money_value(abs($roznica))
                , '%okres_kredytu%' => $gc->g('kredyty|inwestycyjny|okres', 'game')
                , '%oprocentowanie%' => $gc->g('kredyty|inwestycyjny|odsetki', 'game') * 100
                , '%rata%' => self::format_money_value($rata)), 'game_info');
  }

  public static function getInfoStronaInternetowa(HotelGameController $gc, $value, $name) {
    $koszt = $gc->getGameProcess()->getKosztStronaInternetowa($value);
    $infoName = $name;
    
    $kosztJednorazowyStaly = 0;
    if (!($gc->g('kupiona_strona_internetowa', 'round')) && $value > 0) {
      $kosztJednorazowyStaly = $gc->g('strona_internetowa|koszt_jednorazowy', 'game');      
    } else {
      $infoName .= '_bez_stalego';
    }

    if ($gc->rLT(4) || $gc->rGT(5)) {
      $info = __t($infoName . '_info', array('%koszt%' => self::format_money_value($kosztJednorazowyStaly), '%koszt_kwartalny%' => self::format_money_value($koszt)), 'game_info');
    } else {
      $info = self::getCostDiv($kosztJednorazowyStaly) . self::getCostRoundDiv($koszt) . __t($infoName . '_info', array('%koszt%' => self::format_money_value($kosztJednorazowyStaly), '%koszt_kwartalny%' => self::format_money_value($koszt)), 'game_info');
    }

    return $info;
  }

  public static function getInfoPokojeStandard(HotelGameController $gc, $value, $name) {
    $staraDecyzja = $gc->g('pokoje_standard', 'decision', 0);
    $translacja = $name;
    $koszt = 0;
    if ($value == $staraDecyzja) {
      if ($value == 0) {
        // Brak zmiany
        $translacja = $name . '_stan';
      } elseif ($value > 0) {
        // Brak zmiany, ale mamy kupione pokoje
        $translacja = $name . '_stan_kupno';
      } else {
        // Brak zmiany, ale mamy sprzedane pokoje
        $translacja = $name . '_stan_sprzedaz';
      }
    } else {

      if ($staraDecyzja != 0) {
        // Sprowadzamy do stanu 0
        if ($staraDecyzja > 0) {
          // Mielismy kupione pokoje w tej rundzie, sprowadzamy do 0
          $koszt -= $gc->getGameProcess()->getKosztPokoiStandard($staraDecyzja);
        } else {
          // Sprzedalismy pokoje w tej rundzie, sprowadzamy do 0
          $koszt += $gc->getGameProcess()->getZyskSprzedazyPokoiStandard(-$staraDecyzja);
        }
      }

      // Zmieniamy z stanu 0
      if ($value > 0) {
        // Kupujemy X pokoi
        $koszt += $gc->getGameProcess()->getKosztPokoiStandard($value);
      } else {
        // Sprzedajemy X pokoi
        $koszt -= $gc->getGameProcess()->getZyskSprzedazyPokoiStandard(-$value);
      }

      if ($staraDecyzja > 0) {
        if ($value == 0) {
          $translacja = $name . '_kupno_sprzedaz_stan';
        } elseif ($value < 0) {
          $translacja = $name . '_kupno_sprzedaz_sprzedaz';
        } else {
          $translacja = $name . '_kupno_sprzedaz';
        }
      } else {
        if ($value == 0) {
          $translacja = $name . '_sprzedaz_kupno_stan';
        } elseif ($value < 0) {
          $translacja = $name . '_sprzedaz_kupno';
        } else {
          $translacja = $name . '_sprzedaz_kupno_kupno';
        }
      }

      if ($staraDecyzja >= 0 && $value > $staraDecyzja) {
        $translacja = $name . '_kupno';
      }
      if ($staraDecyzja <= 0 && $value < $staraDecyzja) {
        $translacja = $name . '_sprzedaz';
      }
    }

    $pokojeZmiana = abs($value - $staraDecyzja);
    $pokojeDecyzja = abs($value);
    $pokojeDecyzjaStara = abs($staraDecyzja);
    
    if (($gc->gLPS() + $value) < 0 && $value != 0) {
        return self::getCostDiv(0)
                . __t('brak_mozliwosci_sprzedazy_info', array('%koszt%' => format_money_value(abs($koszt)), '%pokoje%' => $gc->gLPS(), '%pokoje_decyzja_stara%' => $pokojeDecyzjaStara, '%pokoje_decyzja%' => $pokojeDecyzja, '%pokoje_zmiana%' => $pokojeZmiana), 'game_info');
    } else {
        return self::getCostDiv($koszt)
            . __t($translacja . '_info', array('%koszt%' => format_money_value(abs($koszt)), '%pokoje%' => $gc->gLPS(), '%pokoje_decyzja_stara%' => $pokojeDecyzjaStara, '%pokoje_decyzja%' => $pokojeDecyzja, '%pokoje_zmiana%' => $pokojeZmiana), 'game_info');
    }

  }

  public static function getInfoPokojeStandardWyposazenie(HotelGameController $gc, $value, $name) {
    $staraDecyzja = $gc->gWS();
    // Decyzja z ostatniej rundy
    if ($gc->rGT(1)) {
      $decyzjaLR = $gc->g('pokoje_standard_wyposazenie', 'last_decision');
    } else {
      $decyzjaLR = 0;
    }
    $koszt = 0;
    $translacja = $name;
    $liczbaPokoiStandard = $gc->gLPS() + max(0, $gc->g('pokoje_standard', 'decision'));
    if ($value == $staraDecyzja) {
      // Brak zmiany
      $translacja .= '_stan';
    } else {
      // Sprowadzamy wyposazenie do stanu z poprzedniej rundy
      if ($staraDecyzja > $decyzjaLR) {
        // Zwracamy zainwestowane pieniadze
        $koszt -= $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $staraDecyzja)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $decyzjaLR);
      } else {
        // Odejmujemy zysk ze sprzedazy pokoi
        $koszt += ($gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $decyzjaLR)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $staraDecyzja))
                * $gc->g('wspolczynnik|likwidacji_wyposazenia', 'game');
      }

      // Zmieniamy wyposazenie
      if ($value > $decyzjaLR) {
        // Kupujemy wyposazenie na wyzszy poziom
        $koszt += $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $value)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $decyzjaLR);
      } else {
        // Sprzedajemy wyposazenie
        // Kwote mnozymy przez wspolczynnik likwidacji wyposazenia
        $koszt -= ($gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $decyzjaLR)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiStandard($liczbaPokoiStandard, $value))
                * $gc->g('wspolczynnik|likwidacji_wyposazenia', 'game');
      }

      if ($value > $staraDecyzja) {
        // Inwestujemy
        $translacja .= '_kupno';
      } else {
        // Sprzedajemy
        $translacja .= '_sprzedaz';
      }
    }
    
    return self::getCostDiv($koszt)
            . __t($translacja . '_info', array('%koszt%' => format_money_value(abs($koszt))), 'game_info');
  }

  public static function getInfoPokojeLux(HotelGameController $gc, $value, $name) {
    $staraDecyzja = $gc->g('pokoje_lux', 'decision', 0);
    $translacja = $name;
    $koszt = 0;
    if ($value == $staraDecyzja) {
      if ($value == 0) {
        // Brak zmiany
        $translacja = $name . '_stan';
      } elseif ($value > 0) {
        // Brak zmiany, ale mamy kupione pokoje
        $translacja = $name . '_stan_kupno';
      } else {
        // Brak zmiany, ale mamy sprzedane pokoje
        $translacja = $name . '_stan_sprzedaz';
      }
    } else {
      if ($staraDecyzja != 0) {
        // Sprowadzamy do stanu 0
        if ($staraDecyzja > 0) {
          // Mielismy kupione pokoje w tej rundzie, sprowadzamy do 0
          $koszt -= $gc->getGameProcess()->getKosztPokoiLux($staraDecyzja);
        } else {
          // Sprzedalismy pokoje w tej rundzie, sprowadzamy do 0
          $koszt += $gc->getGameProcess()->getZyskSprzedazyPokoiLux(-$staraDecyzja);
        }
      }

      // Zmieniamy z stanu 0
      if ($value > 0) {
        // Kupujemy X pokoi
        $koszt += $gc->getGameProcess()->getKosztPokoiLux($value);
      } else {
        // Sprzedajemy X pokoi
        $koszt -= $gc->getGameProcess()->getZyskSprzedazyPokoiLux(-$value);
      }

      if ($staraDecyzja > 0) {
        if ($value == 0) {
          $translacja = $name . '_kupno_sprzedaz_stan';
        } elseif ($value < 0) {
          $translacja = $name . '_kupno_sprzedaz_sprzedaz';
        } else {
          $translacja = $name . '_kupno_sprzedaz';
        }
      } else {
        if ($value == 0) {
          $translacja = $name . '_sprzedaz_kupno_stan';
        } elseif ($value < 0) {
          $translacja = $name . '_sprzedaz_kupno';
        } else {
          $translacja = $name . '_sprzedaz_kupno_kupno';
        }
      }

      if ($staraDecyzja >= 0 && $value > $staraDecyzja) {
        $translacja = $name . '_kupno';
      }
      if ($staraDecyzja <= 0 && $value < $staraDecyzja) {
        $translacja = $name . '_sprzedaz';
      }
    }

    $pokojeZmiana = abs($value - $staraDecyzja);
    $pokojeDecyzja = abs($value);
    $pokojeDecyzjaStara = abs($staraDecyzja);
    
    if (($gc->gLPS() + $value) < 0 && $value != 0) {
        return self::getCostDiv($koszt)
                . __t('brak_mozliwosci_sprzedazy_info', array('%koszt%' => format_money_value(abs($koszt)), '%pokoje%' => $gc->gLPL(), '%pokoje_decyzja_stara%' => $pokojeDecyzjaStara, '%pokoje_decyzja%' => $pokojeDecyzja, '%pokoje_zmiana%' => $pokojeZmiana), 'game_info');
    } else {
        return self::getCostDiv($koszt)
            . __t($translacja . '_info', array('%koszt%' => format_money_value(abs($koszt)), '%pokoje%' => $gc->gLPL(), '%pokoje_decyzja_stara%' => $pokojeDecyzjaStara, '%pokoje_decyzja%' => $pokojeDecyzja, '%pokoje_zmiana%' => $pokojeZmiana), 'game_info');
    }
  }

  public static function getInfoPokojeLuxWyposazenie(HotelGameController $gc, $value, $name) {
    $staraDecyzja = $gc->gWL();
    // Decyzja z ostatniej rundy
    if ($gc->rGT(1)) {
      $decyzjaLR = $gc->g('pokoje_lux_wyposazenie', 'last_decision');
    } else {
      $decyzjaLR = 0;
    }
    $koszt = 0;
    $translacja = $name;
    $liczbaPokoiLux = $gc->gLPL() + max(0, $gc->g('pokoje_lux', 'decision'));
    if ($value == $staraDecyzja) {
      // Brak zmiany
      $translacja .= '_stan';
    } else {
      // Sprowadzamy wyposazenie do stanu z poprzedniej rundy
      if ($staraDecyzja > $decyzjaLR) {
        // Zwracamy zainwestowane pieniadze
        $koszt -= $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $staraDecyzja)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $decyzjaLR);
      } else {
        // Odejmujemy zysk ze sprzedazy pokoi
        $koszt += ($gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $decyzjaLR)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $staraDecyzja))
                * $gc->g('wspolczynnik|likwidacji_wyposazenia', 'game');
      }

      // Zmieniamy wyposazenie
      if ($value > $decyzjaLR) {
        // Kupujemy wyposazenie na wyzszy poziom
        $koszt += $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $value)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $decyzjaLR);
      } else {
        // Sprzedajemy wyposazenie
        // Kwote mnozymy przez wspolczynnik likwidacji wyposazenia
        $koszt -= ($gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $decyzjaLR)
                - $gc->getGameProcess()->getKosztWyposazeniaPokoiLux($liczbaPokoiLux, $value))
                * $gc->g('wspolczynnik|likwidacji_wyposazenia', 'game');
      }

      if ($value > $staraDecyzja) {
        // Inwestujemy
        $translacja .= '_kupno';
      } else {
        // Sprzedajemy
        $translacja .= '_sprzedaz';
      }
    }

    return self::getCostDiv($koszt)
            . __t($translacja . '_info', array('%koszt%' => format_money_value(abs($koszt))), 'game_info');
  }
  
  public static function getInfoUslugaGolf(HotelGameController $gc, $value, $name) {
    $usluga = 'golf';
    $ceny = $gc->g('sezonowosc|koszt|usluga_'.$usluga, 'game');
    if ($gc->g($usluga, 'decision', false)) {
      $cena = $ceny[1];
    } else {
      $cena = $ceny[0];
    }
    if (!$gc->getGameProcess()->isUslugaEventActive('usluga_'.$usluga)) {
      $cena = 0;
    }
    $doZaplaty = $value * $cena;

    return self::getCostRoundDiv($doZaplaty)
            . __t($name . '_info', array('%koszt%' => format_money_value($doZaplaty)), 'game_info');
  }
  
  public static function getInfoUslugaWyciagNarciarski(HotelGameController $gc, $value, $name) {
    $usluga = 'wyciag_narciarski';
    $ceny = $gc->g('sezonowosc|koszt|usluga_'.$usluga, 'game');
    if ($gc->g($usluga, 'decision', false)) {
      $cena = $ceny[1];
    } else {
      $cena = $ceny[0];
    }
    if (!$gc->getGameProcess()->isUslugaEventActive('usluga_'.$usluga)) {
      $cena = 0;
    }
    $doZaplaty = $value * $cena;

    return self::getCostRoundDiv($doZaplaty)
            . __t($name . '_info', array('%koszt%' => format_money_value($doZaplaty)), 'game_info');
  }
  
  public static function getInfoUslugaPlaza(HotelGameController $gc, $value, $name) {
    $usluga = 'plaza';
    $ceny = $gc->g('sezonowosc|koszt|usluga_'.$usluga, 'game');
    if ($gc->g($usluga, 'decision', false)) {
      $cena = $ceny[1];
    } else {
      $cena = $ceny[0];
    }
    if (!$gc->getGameProcess()->isUslugaEventActive('usluga_'.$usluga)) {
      $cena = 0;
    }
    $doZaplaty = $value * $cena;

    return self::getCostRoundDiv($doZaplaty)
            . __t($name . '_info', array('%koszt%' => format_money_value($doZaplaty)), 'game_info');
  }

  public static function kosztUslugi($value, $name, $gc) {
    $cena = $gc->g('uslugi|' . $name, 'game');
    $doZaplaty = $value * $cena;

    return self::getCostRoundDiv($doZaplaty)
            . __t($name . '_info', array('%koszt%' => format_money_value($cena)), 'game_info');
  }

  public static function getInfoRejestracjaNaStronieWww(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoTaxi(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoSejf(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoBudzenie(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoOchronkaDlaDzieci(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoPokojZabawDlaDzieci(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoSalaKonferencyjna(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoWspolpracaUrzadPracy(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoWspolpracaBiuramiPodrozy(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoMozliwoscOdebraniaGosci(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoBusinessLounge(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoEkspresyCzajniki(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoAutomatyVendingowe(HotelGameController $gc, $value, $name) {
    return self::kosztUslugi($value, $name, $gc);
  }

  public static function getInfoGolf(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('sezonowosc|koszt|' . $name, 'game'));
  }

  public static function getInfoWyciagNarciarski(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('sezonowosc|koszt|' . $name, 'game'));
  }

  public static function getInfoPlaza(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('sezonowosc|koszt|' . $name, 'game'));
  }

  public static function getInfoMedia(HotelGameController $gc, $value, $name, $koszt) {
    $doZaplaty = $value * $koszt;
    $infoTranslation = $name;
//    if ($gc->getGameProcess()->hasEventKonkurencja()) {
//        $doZaplaty *= $gc->g('eventy|konkurencja_reklama_mnoznik', 'game');
//    }
    
    return self::getCostRoundDiv($doZaplaty) . __t($infoTranslation . '_info', array('%koszt_sztuka%' => self::format_money_value($koszt), '%koszt%' => self::format_money_value($doZaplaty)), 'game_info');
  }

  public static function getInfoMediaLokalneUlotki(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|ulotki', 'game'));
  }

  public static function getInfoMediaLokalnePlakaty(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|plakaty', 'game'));
  }

  public static function getInfoMediaLokalneBilboardy(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|bilboardy', 'game'));
  }

  public static function getInfoMediaLokalnePrasa(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|prasa', 'game'));
  }

  public static function getInfoMediaLokalneRadio(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|radio', 'game'));
  }

  public static function getInfoMediaRegionalneRadio(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|radio', 'game'));
  }

  public static function getInfoMediaRegionalneTv(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|tv', 'game'));
  }

  public static function getInfoMediaRegionalnePrasa(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|prasa', 'game'));
  }

  public static function getInfoMediaKrajoweRadio(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|radio', 'game'));
  }

  public static function getInfoMediaKrajoweTv(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|tv', 'game'));
  }

  public static function getInfoMediaKrajowePrasa(HotelGameController $gc, $value, $name) {
    return self::getInfoMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|prasa', 'game'));
  }

  public static function getInfoResearch(HotelGameController $gc, $value, $name) {
    $koszt = $gc->getGameProcess()->getResearchCost($name);

    if ($gc->getGameProcess()->hasResearch($name)) {
      return __t($name . '_kupione_info', array('%koszt%' => format_money_value($koszt)), 'game_info');
    } else {
      return self::kosztJednorazowyStaly($gc, $value, $name, $koszt);
    }
  }

  public static function getInfoBadanieMarketingowePopytStandard(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }

  public static function getInfoBadanieMarketingoweUslugiPodstawowe(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }

  public static function getInfoBadanieMarketingoweSezonowosc(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }

  public static function getInfoBadanieMarketingowePopytLux(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }

  public static function getInfoBadanieMarketingoweReklamaZaawansowana(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }
  
  public static function getInfoBadanieMarketingoweKonkurencja(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }
  
  public static function getInfoBadanieMarketingoweUslugiZaawansowaneSegmentacja(HotelGameController $gc, $value, $name) {
    return self::getInfoResearch($gc, $value, $name);
  }

  public static function getInfoEventPanele(HotelGameController $gc, $value, $name) {
    $koszt = ($value - $gc->g('event_panele', 'decision', 0)) * $gc->g('eventy|panele|koszt', 'game');
    
    if ($koszt == 0) {
      return self::getCostDiv($koszt) . __t($name . '_info_stan', array('%koszt%' => self::format_money_value($koszt), '%max_panele%' => $gc->getGameProcess()->getEventPaneleMax()), 'game_info');
    } elseif ($koszt < 0) {
      return self::getCostDiv($koszt) . __t($name . '_info_zysk', array('%koszt%' => self::format_money_value(-$koszt), '%max_panele%' => $gc->getGameProcess()->getEventPaneleMax()), 'game_info');
    } else {
      return self::getCostDiv($koszt) . __t($name . '_info_koszt', array('%koszt%' => self::format_money_value($koszt), '%max_panele%' => $gc->getGameProcess()->getEventPaneleMax()), 'game_info');
    }
  }

  public static function getInfoEventSadzenieDrzew(HotelGameController $gc, $value, $name) {
    $koszt = $gc->getGameProcess()->getKosztSadzenieDrzew($value) - $gc->getGameProcess()->getKosztSadzenieDrzew($gc->g('event_sadzenie_drzew', 'decision'));
    return self::kosztJednorazowyZmienny($gc, $value, $name, $koszt);
  }

  public static function getInfoEventFestiwalFilmowy(HotelGameController $gc, $value, $name) {
    $koszt = 0;
    $doZaplaty = 0;

    if ($gc->rEQ(2)) {
      $koszt = $gc->g('eventy|festiwal_filmowy|koszt', 'game') * $gc->getGameProcess()->getKosztPokoiStandard($gc->gLPS());
      $gc->s('koszt_event_festiwal_filmowy', $koszt, 'round');
      $doZaplaty = $koszt * $value;
    } elseif ($gc->rGT(2)) {
      $koszt = $gc->g('koszt_event_festiwal_filmowy', 'round');
    }

    return self::getCostRoundDiv($doZaplaty) . __t($name . '_info', array('%koszt%' => self::format_money_value($koszt)), 'game_info');
  }

  public static function getInfoEventZimowisko(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|zimowisko', 'game'));
  }

  public static function getInfoEventKierowniczka(HotelGameController $gc, $value, $name) {
    $koszt = $gc->getGameProcess()->getKosztKierowniczka($value);
    return self::getCostRoundDiv($koszt)
        . __t($name . '_info', array('%koszt%' => format_money_value($koszt)), 'game_info');
  }

  public static function getInfoEventAkcjaCharytatywna(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|akcja_charytatywna', 'game'));
  }

  public static function getInfoEventPralniaEkologiczna(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|pralnia_ekologiczna|koszt', 'game'));
  }

  public static function getInfoEventSzkolenie(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|szkolenie', 'game'));
  }

  public static function getInfoEventMikolajkowyEkspres(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|mikolajkowy_ekspres|koszt', 'game'));
  }

  public static function getInfoEventPodjazdDlaNiepelnosprawnych(HotelGameController $gc, $value, $name) {
    return self::kosztJednorazowyStaly($gc, $value, $name, $gc->g('eventy|podjazd_dla_niepelnosprawnych', 'game'));
  }

  public static function getInfoEventStopUmowomSmieciowym(HotelGameController $gc, $value, $name) {
    return __t($name . '_info', array(), 'game_info');
  }

  public static function getInfoKosztZatrudnieniaObslugiStale(HotelGameController $gc) {

    if ($gc->g('pracownicy_obslugi_stale_liczba', 'decision', 0) == 0) {
      return __t('koszt_obslugi_stale_nie_ma_pracownikow_info', array(), 'game_info');
    }

    if ($gc->g('pracownicy_obslugi_stale_wynagrodzenie', 'decision', 0) == 0) {
      return __t('koszt_obslugi_stale_ustal_wynagrodzenie_info', array(), 'game_info');
    }

    $koszt = $gc->getGameProcess()->kosztPracownicyObslugiStale();
    $kosztMotywacji = $gc->getGameProcess()->kosztPracownicyObslugiStaleMotywacjaProcent();

    return __t('koszt_obslugi_stale_info', array('%koszt%' => self::format_money_value($koszt), '%kosztMotywacji%' => self::format_money_value($kosztMotywacji), '%placa_minimalna%' => self::format_money_value($gc->getGameProcess()->getPlacaMinimalnaPracownicyObslugi())), 'game_info');
  }
  
  public static function getInfoKosztZatrudnieniaObslugiZlecenie(HotelGameController $gc) {

    if ($gc->g('pracownicy_obslugi_zlecenie_liczba', 'decision', 0) == 0) {
      return __t('koszt_obslugi_zlecenie_nie_ma_pracownikow_info', array(), 'game_info');
    }

    if ($gc->g('pracownicy_obslugi_zlecenie_wynagrodzenie', 'decision', 0) == 0) {
      return __t('koszt_obslugi_zlecenie_ustal_wynagrodzenie_info', array(), 'game_info');
    }

    $koszt = $gc->getGameProcess()->kosztPracownicyObslugiZlecenie();
    $kosztMotywacji = $gc->getGameProcess()->kosztPracownicyObslugiZlecenieMotywacjaProcent();

    return __t('koszt_obslugi_zlecenie_info', array('%koszt%' => self::format_money_value($koszt), '%kosztMotywacji%' => self::format_money_value($kosztMotywacji), '%placa_minimalna%' => self::format_money_value($gc->getGameProcess()->getPlacaMinimalnaPracownicyObslugi())), 'game_info');
  }

  public static function getInfoKosztZatrudnieniaAdministracjiStale(HotelGameController $gc) {

    if ($gc->g('pracownicy_administracji_stale_liczba', 'decision', 0) == 0) {
      return __t('koszt_administracji_stale_nie_ma_pracownikow_info', array(), 'game_info');
    }

    if ($gc->g('pracownicy_administracji_stale_wynagrodzenie', 'decision', 0) == 0) {
      return __t('koszt_administracji_stale_ustal_wynagrodzenie_info', array(), 'game_info');
    }

    $koszt = $gc->getGameProcess()->kosztPracownicyAdministracjiStale();
    $kosztMotywacji = $gc->getGameProcess()->kosztPracownicyAdministracjiStaleMotywacjaProcent();

    return __t('koszt_administracji_stale_info', array('%koszt%' => self::format_money_value($koszt), '%kosztMotywacji%' => self::format_money_value($kosztMotywacji), '%placa_minimalna%' => self::format_money_value($gc->getGameProcess()->getPlacaMinimalnaPracownicyAdministracji())), 'game_info');
  }
  
  public static function getInfoKosztZatrudnieniaAdministracjiZlecenie(HotelGameController $gc) {

    if ($gc->g('pracownicy_administracji_zlecenie_liczba', 'decision', 0) == 0) {
      return __t('koszt_administracji_zlecenie_nie_ma_pracownikow_info', array(), 'game_info');
    }

    if ($gc->g('pracownicy_administracji_zlecenie_wynagrodzenie', 'decision', 0) == 0) {
      return __t('koszt_administracji_zlecenie_ustal_wynagrodzenie_info', array(), 'game_info');
    }

    $koszt = $gc->getGameProcess()->kosztPracownicyAdministracjiZlecenie();
    $kosztMotywacji = $gc->getGameProcess()->kosztPracownicyAdministracjiZlecenieMotywacjaProcent();

    return __t('koszt_administracji_zlecenie_info', array('%koszt%' => self::format_money_value($koszt), '%kosztMotywacji%' => self::format_money_value($kosztMotywacji), '%placa_minimalna%' => self::format_money_value($gc->getGameProcess()->getPlacaMinimalnaPracownicyAdministracji())), 'game_info');
  }
  
  /** Obsluga decyzji absurdalnych */
  public static function getAbsurditySprzatanieOutsourcingStandardKwota(HotelGameController $gc, $value, $field) {
    if ($gc->getGameProcess()->isCostUnreasonable($value)) {
      return 'Nierozsadny';
    } else {
      return;
    }
  }
  
  public static function getAbsurditySprzatanieOutsourcingLuxKwota(HotelGameController $gc, $value, $field) {
    if ($gc->getGameProcess()->isCostUnreasonable($value)) {
      return 'Nierozsadny';
    } else {
      return;
    }
  }
  
  public static function getAbsurdityPranieOutsourcingStandardKwota(HotelGameController $gc, $value, $field) {
    if ($gc->getGameProcess()->isCostUnreasonable($value)) {
      return 'Nierozsadny';
    } else {
      return;
    }
  }
  
  public static function getAbsurdityPranieOutsourcingLuxKwota(HotelGameController $gc, $value, $field) {
    if ($gc->getGameProcess()->isCostUnreasonable($value)) {
      return 'Nierozsadny';
    } else {
      return;
    }
  }
  
  public static function getAbsurdityMedia(HotelGameController $gc, $value, $name, $koszt) {
    $doZaplaty = $value * $koszt;
    if ($gc->getGameProcess()->isCostUnreasonable($doZaplaty)) {
      return 'Nierozsadny';
    } else {
      return;
    }
  }

  public static function getAbsurdityMediaLokalneUlotki(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|ulotki', 'game'));
  }

  public static function getAbsurdityMediaLokalnePlakaty(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|plakaty', 'game'));
  }

  public static function getAbsurdityMediaLokalneBilboardy(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|bilboardy', 'game'));
  }

  public static function getAbsurdityMediaLokalnePrasa(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|prasa', 'game'));
  }

  public static function getAbsurdityMediaLokalneRadio(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|lokalne|radio', 'game'));
  }

  public static function getAbsurdityMediaRegionalneRadio(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|radio', 'game'));
  }

  public static function getAbsurdityMediaRegionalneTv(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|tv', 'game'));
  }

  public static function getAbsurdityMediaRegionalnePrasa(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|regionalne|prasa', 'game'));
  }

  public static function getAbsurdityMediaKrajoweRadio(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|radio', 'game'));
  }

  public static function getAbsurdityMediaKrajoweTv(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|tv', 'game'));
  }

  public static function getAbsurdityMediaKrajowePrasa(HotelGameController $gc, $value, $name) {
    return self::getAbsurdityMedia($gc, $value, $name, $gc->g('media|koszty|krajowe|prasa', 'game'));
  }

}
