<?php

/**
 * Klasa bazowa do zarzadzania gra
 */
abstract class BaseGameController extends HotelGameFunctions {

  /** Prefiks dla klas dziedziczacych */
  protected static $prefix = 'Base';

  /** Klasy obslugujace gre */
  protected $decisionFormClass;
  protected $gameValidatorClass;
  protected $gameDecisionClass;
  protected $gameProcessClass;
  protected $gameViewClass;
  protected $gameMessageClass;
  protected $gameDataClass;
  protected $gameLogClass;
  protected $gameValidator = null;
  protected $gameDecision = null;
  protected $gameProcess = null;
  protected $gameView = null;
  protected $gameMessage = null;
  protected $gameLog = null;
  protected $gameData = null;
  protected $backupGameData = null;
  /** Klasy obslugujace gre - KONIEC */

  /** Zmienna zawierajaca partiale dla widoku zakladki */
  protected $partials = array();

  /** ID zespolu */
  protected $teamId = null;

  /** ID uzytkownika */
  protected $userId = null;

  public function __construct($teamId = null, $userId = null) {
    // Konfiguracja klas
    $this->configureClasses();
    // Inicjalizacja gry
    if (!is_null($teamId)) {
      $this->gameData = new $this->gameDataClass($teamId);
    }

    $this->userId = $userId;
    $this->teamId = $teamId;
      }

  public static function getClass($baseClass) {
    return str_replace('Base', static::$prefix, $baseClass);
  }

  public function __destructor() {
        $this->getGameLog()->save();
  }

  protected function configureClasses() {
    $this->decisionFormClass = static::getClass('BaseDecisionForm');
    $this->gameValidatorClass = static::getClass('BaseGameValidator');
    $this->gameDecisionClass = static::getClass('BaseGameDecision');
    $this->gameProcessClass = static::getClass('BaseGameProcess');
    $this->gameViewClass = static::getClass('BaseGameView');
    $this->gameMessageClass = static::getClass('BaseGameMessage');
    $this->gameDataClass = static::getClass('BaseGameData');
    $this->gameLogClass = static::getClass('BaseGameLog');
  }

  /**
   * Funkcja wykorzystywana do zwracania unikalnej nazwy zakladki 
   */
  abstract public function getTabUniqueName();

  public function getRoundNumber() {
    return $this->getGameData()->getRoundNumber();
  }

  /**
   * Funkcja robi backup aktualnego stanu gry. Wykorzystywane m.in. przy dokonywaniu
   * walidacji formularza decyzji
   */
  public function backupGameData() {
    $this->backupGameData = $this->getGameData();
    $this->gameData = clone $this->getGameData();
  }

  /**
   * Funkcja przywraca stan gry z backupu i usuwa aktualny
   */
  public function restoreGameData() {
    $this->gameData = $this->backupGameData;
    $this->backupGameData = null;
  }

  /**
   * 
   * @param type $gameId
   */
  public static function gameReverseRound($gameId) {

    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::gameReverseRound($gameId, static::getClass('BaseGameController'));
  }

  /**
   * Funkcja przechodzi do nastepnej rundy dla wszystkich zespolow w grze
   *
   * @param type $gameId ID Gry
   * @throws sfException
   */
  public static function moveToNextRoundForGame($gameId) {

    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::moveToNextRoundForGame($gameId, static::getClass('BaseGameController'));
  }

  /**
   * Funkcja aktywuje gre. Dodaje runde 0 dla kazdej druzyny. Uzupelnia dane
   * gry o 'game_data_json' i 'game_controller_class' oraz zmienia jej status na 1 (aktywna).
   *
   * @param type $gameId ID Gry
   * @throws sfException
   */
  public static function activateGame($gameId) {
    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::activateGame($gameId, static::getClass('BaseGameController'), static::getClass('BaseGameData'));
  }

  /**
   * Funkcja archiwizuje gre. Ustawia status na 2 (archiwum) i ustawia ostatnia runde na nieaktywna
   *
   * @param type $gameId ID Gry
   * @throws sfException
   */
  public static function archiveGame($gameId) {
    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::archiveGame($gameId, static::getClass('BaseGameController'));
  }

  /**
   * Funkcja przygotowuje gre. Uzupelnia dane
   * gry o 'game_data_json' i 'game_controller_class' oraz zmienia jej status na 3 (przygotowana).
   *
   * @param type $gameId ID Gry
   * @throws sfException
   */
  public static function prepareGame($gameId) {
    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::prepareGame($gameId, static::getClass('BaseGameController'), static::getClass('BaseGameData'));
  }

  /**
   * Funkcja deaktywuje gre. Usuwa wszystkie istniejace rundy zwiazane z druzynami
   * przypisanymi do danej gry. Ustawia status na 0 i 'game_json_data' na NULL.
   * @param type $gameId ID Gry
   * @throws sfException
   */
  public static function deactivateGame($gameId) {
    $gameProcessClass = static::getClass('BaseGameProcess');
    $gameProcessClass::deactivateGame($gameId);
  }

  /**
   * Funkcja zwraca formularz decyzji. Przygotowuje go odpowiednio na podstawie
   * przekazanej nazwy zakladki oraz stanu gry.
   *
   * @param string $tabUniqueName
   * @return HotelDecisionForm
   * @throws sfException
   */
  public function getDecisionForm($tabUniqueName = FALSE, $userId) {
    $formClass = $this->decisionFormClass;
    $form = new $formClass(array(), array('gameController' => $this));

    if (!$tabUniqueName) {
      return $form;
    }

    $prepareFunction = 'prepareTab' . sfInflector::camelize($tabUniqueName);

    if (method_exists($this, $prepareFunction)) {
      
      $widgets = $this->$prepareFunction($form); 
      
      if (empty($widgets) && count($this->partials) == 0) {
        throw new sfException('Próba otwarcia zakładki "' . $tabUniqueName . '" przy nieodpowiednim stanie gry. Funkcja  "' . $prepareFunction . '" w klasie  "' . get_class($this) . '"');
      }
      $form->useFields($widgets);
      
      $this->getGameView()->setWidgetsStatus($widgets, $tabUniqueName, $userId);
      
      $postPrepareFunction = 'postPrepareTab' . sfInflector::camelize($tabUniqueName);
      if (method_exists($this, $postPrepareFunction)) {
        $this->$postPrepareFunction($form, $widgets);
      }
      
    } else {
      throw new sfException('Zakładka "' . $tabUniqueName . '" nie jest zdefiniowana w klasie "' . get_class($this) . '". Brak funkcji "' . $prepareFunction . '".');
    }

    if ($this->gRN() == self::LAST_ROUND) {
            
      foreach ($form->getWidgetSchema()->getInformations()  as $fieldName => $val) {
        
        if (isset($form[$fieldName]) && !in_array($tabUniqueName, array('ankieta_po','test_po','ankieta_przed','test_przed')))
          $form->getWidget($fieldName)->setAttribute('disabled', 'disabled');
      }
    }

    return $form;
  }

  /**
   * Funkcja waliduje decyzje o podanej nazwie na podstawie aktualnego stanu gry.
   * Poczatkowo szuka metody we wlasnej klasie, nastepnie w klasie $gameValidatorClass.
   * W przypadku braku metod w obu tych miejscach zwraca true (brak bledu, walidacja poprawna).
   * Nazwy metod sa tworzone na podstawie nazwy decyzji przy pomocy "sfInflector::camelize" -
   * z "hotel_pokoje_inwestycje" powstanie "validateHotelPokojeDecyzje". Metody walidujace
   * zwracaja true lub klase sfValidatorError ktora jest dopinana do pola w formularzu.
   *
   * @param string $decisionName nazwa decyzji
   * @param mixed $value wartosc decyzji
   * @return boolean or sfValidatorError
   */
  public function validateDecision($decisionName, $value) {
    $validateFunctionName = 'validate' . sfInflector::camelize($decisionName);
    if (method_exists($this, $validateFunctionName)) {
      return $this->$validateFunctionName($value);
    } else {
      $gameValidator = $this->getGameValidator();
      if (method_exists($gameValidator, $validateFunctionName)) {
        $validateResult = $gameValidator->$validateFunctionName($value);
        return $validateResult;
      }
    }

    return true;
  }

  /**
   * Funkcja zapisuje decyzje.
   * Poczatkowo szuka metody we wlasnej klasie, nastepnie w klasie $gameDecisionClass.
   * W przypadku braku metod w obu tych miejscach generuje exception.
   * Nazwy metod sa tworzone na podstawie nazwy decyzji przy pomocy "sfInflector::camelize" -
   * z "hotel_pokoje_inwestycje" powstanie "makeHotelPokojeDecyzje".
   * Dodatkowo loguje podjeta decyzje
   *
   * @param string $decisionName nazwa decyzji
   * @param mixed $value wartosc decyzji
   * @throws sfException
   */
  public function makeDecision($decisionName, $value) {
    $makeFunctionName = 'make' . sfInflector::camelize($decisionName);
    if (method_exists($this, $makeFunctionName)) {
      $oldValue = $this->getGameData()->g($decisionName, 'decision');
      $gameDecision->$makeFunctionName($value);
      $newValue = $this->getGameData()->g($decisionName, 'decision');
      $this->getGameData()->logDecision($decisionName, $oldValue, $newValue, $this->userId);
    } else {
      $gameDecision = $this->getGameDecision();
      if (method_exists($gameDecision, $makeFunctionName)) {
        $oldValue = $this->getGameData()->g($decisionName, 'decision');
        $gameDecision->$makeFunctionName($value);
        $newValue = $this->getGameData()->g($decisionName, 'decision');
        $this->getGameData()->logDecision($decisionName, $oldValue, $newValue, $this->userId);
      } else {
        throw new sfException('Brak funkcji "' . $makeFunctionName . '" wymaganej do podjecia decyzji.');
      }
    }

    $this->getGameData()->save();
  }

  /**
   * Zwraca ID uzytkownika
   * @return int
   */
  public function getUserId() {
    return $this->userId;
  }

  public function getTeamId() {
    return $this->teamId;
  }

  /**
   * Zwraca partiale potrzebne do wyswietlenia zakladki
   * @return array
   */
  public function getPartials() {
    $partials = $this->partials;
    $this->partials = array();
    return $partials;
  }
  
  public function getNoForm() {
    if (isset($this->noForm)) {
      return $this->noForm;
    } else {
      return false;
    }
  }

  /**
   * Funkcja zwraca klase $gameValidatorClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameValidatorClass
   */
  public function getGameValidator() {
    if (is_null($this->gameValidator)) {
      $this->gameValidator = new $this->gameValidatorClass($this);
    }

    return $this->gameValidator;
  }

  /**
   * Funkcja zwraca klase $gameDataClass. Jesli nie byla zainicjowana
   * generuje wyjatek
   *
   * @return object $gameDataClass
   * @throws sfException
   */
  public function getGameData() {
    if (is_null($this->gameData)) {
      throw new sfException('Brak zainicjowanej klasy GameData.');
    }

    return $this->gameData;
  }

  /**
   * Funkcja zwraca klase $gameDecisionClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameDecisionClass
   */
  public function getGameDecision() {
    if (is_null($this->gameDecision)) {
      $this->gameDecision = new $this->gameDecisionClass($this);
    }

    return $this->gameDecision;
  }

  /**
   * Funkcja zwraca klase $gameProcessClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameProcessClass
   */
  public function getGameProcess() {
    if (is_null($this->gameProcess)) {
      $this->gameProcess = new $this->gameProcessClass($this);
    }

    return $this->gameProcess;
  }

  /**
   * Funkcja zwraca klase $gameViewClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameViewClass
   */
  public function getGameView() {
    if (is_null($this->gameView)) {
      $this->gameView = new $this->gameViewClass($this);
    }

    return $this->gameView;
  }

  /**
   * Funkcja zwraca klase $gameMessageClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameMessageClass
   */
  public function getGameMessage() {
    if (is_null($this->gameMessage)) {
      $this->gameMessage = new $this->gameMessageClass($this);
    }

    return $this->gameMessage;
  }

  /**
   * Funkcja zwraca klase $gameLogClass. Inicjuje ja, jesli nie byla wczesniej
   * zainicjowana
   *
   * @return object $gameLogClass
   */
  public function getGameLog() {
    if (is_null($this->gameLog)) {
      $this->gameLog = new $this->gameLogClass($this);
    }

    return $this->gameLog;
  }

}
