<?php

/**
 * Klasa bazowa zwierająca metody służace do obsługi wiadomości w grze.
 *
 */
abstract class BaseGameMessage extends HotelGameFunctions {

  /**
   * @var $component
   */
  protected $component = 'message',
          $articleComponent = 'message',
          $partial = 'message/partial',
          $defaultTitle = 'Wiadomość',
          $defaultCategory = 'Wiadomości';

  const READ = 1;
  const NOT_READ = 0;

  /**
   * Wysyła wiaodmosć z artykułu do całego zespołu
   *
   * @param $message = array('uniqueName' => 'Unikalna nazwa artykułu')
   * @param $params = array z prametrami do podstawiania w treści wiadomosći.
   */
  public function sendArticleMessage($message, $params) {
    $message['type'] = 'article';

    $sentTo = array('type' => 'team');

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Wysyła wiadomość z partiala do całego zespołu
   *
   * @param $message= array('partial' => 'Unikalna nazwa partiala')
   * @param $params
   */
  public function sendPartialMessage($partial, $title = 'Event' , $params = array() , $app = 'frontend') {
    $message['type'] = 'partial';
    $message['application'] = $app; 
    $message['partial'] = $partial;
    $message['title'] = $title;
    
    $sentTo = array('type' => 'team');

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Wysyła wiadomość z artykułu do użytkownika o wskazanym ID
   *
   * @param $message= array('uniqueName' => 'Unikalna nazwa artykułu', '')
   * @param $userId
   * @param $params
   */
  public function sendArticleMessageToUser($message, $userId, $params) {
    $message['type'] = 'article';

    $sentTo = array('type' => 'user',
        'user_id' => $userId);

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Wysyła wiadomość z partiala do użytkownika/
   *
   * @param $message= array('partial' => 'Unikalna nazwa partiala')
   * @param $userId
   * @param $params
   */
  public function sendPartialMessageToUser($message, $userId, $params) {
    $message['type'] = 'partial';

    $sentTo = array('type' => 'user',
        'user_id' => $userId);

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Wysyła wiadomość textową do użytkownika
   *
   * @param $message , String, z treścią wiadomości.
   * @param $userId
   * @param $params
   */
  public function sendTextMessageToUser($title, $body, $userId, $params) {

    $message = array();

    $message['title'] = $title;
    $message['body'] = $body;
    $message['type'] = 'text';

    $sentTo = array('type' => 'user',
        'user_id' => $userId);

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Wysyła wiadomość textową do użytkownika
   *
   * @param $message , String, z treścią wiadomości.
   * @param $userId
   * @param $params
   */
  public function sendTextMessageToTeam($title, $body, $params) {

    $message = array();

    $message['title'] = $title;
    $message['body'] = $body;
    $message['type'] = 'text';

    $sentTo = array('type' => 'team');

    return $this->sendMessage($message, $params, $sentTo);
  }

  /**
   * Metoda wysyła wiadomość jako parametry przyjmuje 3 tablice;
   *
   * @param $message; tablica z danymi dotyczącymi treści wiadomości.
   * @param $params: tablica ze zmiennymi które należy spersonalizować dla każdej z wiadomości.
   * @param $sentTo: ustwianie do kogo wysłać wiadomosć;
   *
   *        $sentTo = array('type' => 'user / team / game',
   *                        'user_id' => 'Id iżytkownika dla typu : user ',
   *                        'team_id' => 'Id zespołu dla typów : user, team ',
   *                  );
   *        // INFO parmetr type, wymagany reszta opcjonalna.
   *
   * @return $mixed, true, false, messages count.
   */
  public function sendMessage($message, $params, $sendTo = false) {
    if ($sendTo) {

      switch ($sendTo['type']) {
        case "user": {
            /**
             * Wyślij wiadomośc do użytkownika:
             */
            return $this->sendUserMessage($message, $params, $sendTo['user_id']);
            break;
          }
        case "team": {
            /**
             * Wyślij wiadomośc do użytkowników z zespołu.
             */
            return $this->sendTeamMessage($message, $params);
            break;
          }
        default: {
            throw new sfException('Niepoprawny typ wiadomośći. Typ :' . $sendTo['type'] . ' nie istnieje.');
          }
      }
    } else {
      throw new sfException('Podaj w parametrze $sendTo, typ wiadomośći oraz ID do kogo wiadomość ma zostać wysłana.
                  Wiadomośc może zostać wysłana do użytkownika, zespołu oraz wszytskich użytkowników z gry.');
    }

    return false;
  }

  /**
   *
   * @param $message
   * @param $params
   * @param $userId
   */
  public function sendUserMessage($message, $params, $userId) {

    $message = $this->getMessageObject($message, $params, $userId);

    return $this->saveDatabaseUserMessage($message);
  }

  /**
   * Metoda wysyła wiadomość do wszytkich osób dodanych do zespołu któprego id jest przekazane
   * .
   * @param $message; tablica z danymi dotyczącymi treści wiadomości.
   * @param $params: tablica ze zmiennymi które należy spersonalizować dla każdej z wiadomości.
   * @param $teamId, id zespołu dla którego członków należy wysłać wiadomość.
   * @return int, ilość wysłąnych wiaodmości;
   */
  public function sendTeamMessage($message, $params) {
    $users = Doctrine::getTable('UserTeam')->findByTeamId($this->getGameData()->getTeam()->getId());
    $counter = 0;
    $messages = array();

    foreach ($users as $user) {
      $object = $this->getMessageObject($message, $params, $user->getUserId());
      array_push($messages, $object);
      $counter++;
    }

    $this->saveDatabaseTeamMessages($messages);

    return $counter;
  }

  /**
   * Wysyła (zapisuje w bazie danych) wiadomość do użytkownika
   *
   * @param $message; tablica z danymi dotyczącymi treści wiadomości.
   * @param $params: tablica ze zmiennymi które należy spersonalizować dla każdej z wiadomości.
   * @param $userId, id użytkownika do którego wysłano wiadomosć.
   * @param $teamId, id zespołu do którego należy użytkownik któremu wysyłano wiadomość;
   * @return object, wiadomość;
   *
   * @see: sendTeamMessage(), sendMessage();
   */
  public function getMessageObject($message, $params, $userId) {
    if (isset($message['title'])) {
      $title = $message['title'];
    } else {
      $title = $this->defaultTitle;
    }

    if (isset($message['category'])) {
      $category = $message['category'];
    } else {
      $category = $this->defaultCategory;
    }
    
    if (isset($message['status'])) {
      $status = $message['status'];
    } else {
      $status = self::NOT_READ;
    }

    $teamId = $this->getGameData()->getTeam()->getId();
    $body = $this->prepareStringWithParams($this->getMessageBody($message), $params);
    $article = Doctrine::getTable('Article')->findOneByUniqueName($message['unique_name']);
    if ($article) {
      $title = $this->prepareStringWithParams($article->getTitle());
    } else {
      $title = 'Brak artykułu o unique_name ="'.$message['unique_name'].'"';
    }

    return $this->setMessageObject($teamId, $userId, $status, $title, $category, $body);
  }

  /**
   * Metoda przygotowuje treść wiaodmości;
   *
   * @param $message, tablica z parametrami dotyczącymi treści wiadomości;
   * @return mixed, html w przypadku sukcesu;
   * @throws sfException, w przypadku niepoprawnych parametrów;
   *
   * @see: sendUserMessage(), sendTeamMessage(),
   */
  public function getMessageBody($message) {

    switch ($message['type']) {
      case 'article':
        /**
         * Wstawianie tekstu wiadomości z artykułu.
         */
        if (!isset($message['component'])) {
          $message['component'] = $this->articleComponent;
        }
        $html = $this->getMessageBodyFromComponent($message);
        break;
      case 'component':
        /**
         * Wstawianie tekstu wiadomości z komponentu.
         */
        $html = $this->getMessageBodyFromComponent($message);
        break;
      case 'text':
        /**
         * Wstawianie textu wiadomości z parametru.
         */
        $html = $message['body'];
        break;
      case 'partial':
        /**
         * Wstawianie textu wiadomości z partiala.
         */
        $html = $this->getMessageBodyFromPartial($message);
        break;
      default:
        throw new sfException('Niepoprawny typ treści wiadomośći. Typ :' . $message['type'] . ' nie istnieje. Wiadomośći są pobierane z artykułów, partiali bezpośrednio z parametru');
    }

    return $html;
  }

  /**
   * Pobiera treść wiadomości z partiala.
   *
   * @param $message
   * @return string
   */
  public function getMessageBodyFromPartial($message) {
    if (isset($message['params'])) {
      $parameters = $message['params'];
    } else {
      $parameters = array();
    }
    
    if (isset($message['application'])){
      $app = $message['application']; 
    }  else {
      $app = null;
    }

    if (isset($message['partial'])) {
      return View::getPartial($message['partial'], $parameters, $app);
    } else {
      return View::getPartial($this->partial, $parameters, $app);
    }
  }

  /**
   * Pobiera treść artykułu i opakowuje go w layout;
   *
   * @param $message, dane dotyczące jaki artykuł pobrać;
   * @return mixed, zwraca html wiadomosći;
   *
   * @see getMessageBody();
   */
  public function getMessageBodyFromComponent($message) {
    if (isset($message['params'])) {

      if (isset($message['uniqueName'])) {
        $parameters = array_merge($message['params'], array('uniqueName' => $message['uniqueName']));
      } else {
        $parameters = $message['params'];
      }
    } else {
      $parameters = array('uniqueName' => $message['uniqueName']);
    }

    if (isset($message['component'])) {
      return View::getComponent('message', $message['component'], $parameters);
    } else {
      return View::getComponent('message', $this->component, $parameters);
    }
  }

  /**
   * Wstawia parametry do treści wiadomości;
   *
   * @param $message, HTML wiadomosci, z wstawkami zamiast parametrów;
   * @param $params, parametry do wstawiania w HTML;
   * @return mixed, ostatecznych HTML zapisywany do bazy danych;
   *
   * @see: sendUserMessage()
   */
  public function prepareStringWithParams($str, $params = array()) {
    // Poprostu genialne
    $params = $this->getMessageParams();
    foreach ($params as $var_name => $replacement) {
      $str = str_replace('%' . $var_name . '%', $replacement, $str);
    }

    return $str;
  }
  
  /**
   * Zwraca parametry gry wykorzystywane w wiadomosciach
   * @return array 
   */
  public function getMessageParams() {
    return array();
  }

  /**
   * Zapisuje wiadomość do bazy danych.
   *
   * @param $teamId
   * @param $userId
   * @param $status
   * @param $title
   * @param $category
   * @param $body
   * @return Message
   */
  public function setMessageObject($teamId, $userId, $status, $title, $category, $body) {
    
    $message = new Message();
    $message->setTeamId($teamId);
    $message->setUserId($userId);
    $message->setStatus($status);
    $message->setTitle($title);
    $message->setBody($body);
    $message->setRound($this->getGameData()->getRound()); 
    $message->setCategory($category);

    return $message;
  }

  /**
   * @param Message $message
   */
  public function saveDatabaseUserMessage(Message $message) {

    $conn = sfContext::getInstance()->getDatabaseManager()->getDatabase('doctrine')->getDoctrineConnection();

    
    try {
      $conn->beginTransaction();
      $this->saveDatabaseMessage($conn, $message);
      $conn->commit();
    } catch (Doctrine_Exception $e) {
      $conn->rollback();
    }
  }

  /**
   * @param Doctrin_Connection $conn
   * @param Message $message
   */
  public function saveDatabaseMessage(Doctrine_Connection_Mysql $conn, Message $message) {
    $conn->beginTransaction();
    $message->save();
    $conn->commit();
  }

  /**
   * @param $messages
   */
  public function saveDatabaseTeamMessages($messages) {

    $conn = sfContext::getInstance()->getDatabaseManager()->getDatabase('doctrine')->getDoctrineConnection();

    try {
      $conn->beginTransaction();
      foreach ($messages as $message) {
        $this->saveDatabaseMessage($conn, $message);
      }

      $conn->commit();
    } catch (Doctrine_Exception $e) {
      $conn->rollback();
    }

    return true;
  }
}