<?php

/**
 * Klasa bazowa zarzadzania procesem dla gry 
 */
abstract class BaseGameProcess extends HotelGameFunctions {

  /**
   * Funkcja przechodzi do poprzendiej rundy dla zespolow w calej grze
   * @param type $gameId
   * @param type $gameControllerClass 
   */
  public static function gameReverseRound($gameId, $gameControllerClass) {

    $teams = Doctrine::getTable('Team')->findByGameId($gameId);

    foreach ($teams as $team) {
      $gameController = new $gameControllerClass($team->getId());
      $gameController->getGameProcess()->reverseRound($team->getId());
    }
  }
  
  /**
   * Funkcja realizuje przejście do poprzedniej rundy zespołu. 
   * 
   * @param type $teamId
   */
  public function reverseRound($teamId){
    
    $currentRoundNumber =  $this->getGameData()->getRoundNumber(); 
    
    $currentRound = Doctrine::getTable('Round')->createQuery('r')
            ->where('r.team_id = ?', $teamId)
            ->andWhere('r.round_number = ?', $currentRoundNumber)
            ->fetchOne(); 
    
    $previousRoundNumber =  $currentRoundNumber - 1; 
    
    $previousRound = Doctrine::getTable('Round')->createQuery('r')
            ->where('r.team_id = ?', $teamId)
            ->andWhere('r.round_number = ?', $previousRoundNumber)
            ->fetchOne(); 
    
    $currentRound->delete(); 
    
    $previousRound->setStatus(Round::ACTIVE);
    $previousRound->save(); 
  }

  /**
   * Funkcja przechodzi do nastepnej rundy dla zespolow w calej grze
   * @param type $gameId
   * @param type $gameControllerClass 
   */
  public static function moveToNextRoundForGame($gameId, $gameControllerClass) {
    $teams = Doctrine::getTable('Team')->findByGameId($gameId);
    foreach ($teams as $team) {
      $gameController = new $gameControllerClass($team->getId());
      $gameController->getGameProcess()->moveToNextRound();
    }
  }

  /** Przejscie do nastepnej rundy i obliczenia */
  abstract public function moveToNextRound();

  /** Funkcja tworzy pierwsza runde */
  public static function createFirstRound(Team $team, $gameDataClass, $gameDataJson) {
    $round = new Round();
    $round->setTeam($team);
    $round->setRoundNumber(1);
    $round->setStatus(Round::ACTIVE);
    $round->setName('Runda 1');

    $round->setRoundDataJson($gameDataClass::arrayToJson(static::getFirstRoundDataJson()));
    $round->setRoundDecisionDataJson($gameDataClass::arrayToJson(static::getFirstRoundDecisionDataJson()));
    $round->setRoundRaportDataJson($gameDataClass::arrayToJson(static::getFirstRoundRaportDataJson()));

    $round->save();
  }

  /**
   * Funkcja aktywuje gre. Dodaje runde 0 dla kazdej druzyny. Uzupelnia dane
   * gry o 'game_data_json' i 'game_controller_class' oraz zmienia jej status na 1 (aktywna).
   *
   * @param int $gameId ID Gry
   * @param string $gameControllerClass Klasa kontrolera grys
   * @throws sfException 
   */
  public static function activateGame($gameId, $gameControllerClass, $gameDataClass) {
    $game = Doctrine::getTable('Game')->findOneById($gameId);

    if (!$game) {
      throw new sfException('Gra o id "' . $gameId . '" nie istnieje.');
    }
    if ($game->isActive() || $game->isArchive()) {
      throw new sfException('Gra o id "' . $gameId . '" została już zainicjalizowana.');
    }

    $teams = Doctrine::getTable('Team')->findByGameId($gameId);
    if ($teams->count() == 0) {
      throw new sfException('Gra o id "' . $gameId . '" nie posiada przypisanych drużyn.');
    }

    $game->setGameControllerClass($gameControllerClass);
    if (!$game->isPrepared()) {
      $game->setGameDataJson($gameDataClass::arrayToJson(static::getGameJsonData()));
    };

    $game->setStatus(Game::ACTIVE);
    $game->save();

    foreach ($teams as $team) {
            self::createFirstRound($team, $gameDataClass, $gameDataClass::jsonToArray($game->getGameDataJson()));
    }
  }

  /**
   * Funkcja przygotowuje gre. Uzupelnia dane
   * gry o 'game_data_json' i 'game_controller_class' oraz zmienia jej status na 3 (przygotowana).
   *
   * @param int $gameId ID Gry
   * @param string $gameControllerClass Klasa kontrolera grys
   * @throws sfException 
   */
  public static function prepareGame($gameId, $gameControllerClass, $gameDataClass) {
    $game = Doctrine::getTable('Game')->findOneById($gameId);

    if (!$game) {
      throw new sfException('Gra o id "' . $gameId . '" nie istnieje.');
    }
    if ($game->isActive() || $game->isArchive()) {
      throw new sfException('Gra o id "' . $gameId . '" została już zainicjalizowana.');
    }

    $game->setGameControllerClass($gameControllerClass);
    $game->setStatus(Game::PREPARED);
    $game->setGameDataJson($gameDataClass::arrayToJson(static::getGameJsonData()));
    $game->save();
  }

  /**
   * Funkcja deaktywuje gre. Usuwa wszystkie istniejace rundy zwiazane z druzynami
   * przypisanymi do danej gry. Ustawia status na 0 i 'game_json_data' na NULL.
   * @param type $gameId ID Gry
   * @throws sfException 
   */
  public static function deactivateGame($gameId) {
    $game = Doctrine::getTable('Game')->findOneById($gameId);
    if (!$game) {
      throw new sfException('Gra o id "' . $gameId . '" nie istnieje.');
    }
    if ($game->getStatus() == 0) {
      throw new sfException('Gra o id "' . $gameId . '" jest juz nie aktywna.');
    }

    Doctrine::getTable('Round')->deleteByGameId($gameId);
    Doctrine::getTable('Message')->deleteByGameId($gameId);

    $game->setStatus(Game::INACTIVE);
    $game->setGameDataJson(NULL);
    $game->save();
  }

  /**
   * Funkcja archiwizuje gre. Ustawia status na 2 (archiwum) i ustawia ostatnia runde na nieaktywna
   * 
   * @param type $gameId ID Gry
   * @throws sfException 
   */
  public static function archiveGame($gameId) {
    $game = Doctrine::getTable('Game')->findOneById($gameId);
    if (!$game) {
      throw new sfException('Gra o id "' . $gameId . '" nie istnieje.');
    }
    if ($game->getStatus() == 0) {
      throw new sfException('Gra o id "' . $gameId . '" jest juz nie aktywna.');
    }

    $round = Doctrine::getTable('Round')->getActiveRoundByGameId($gameId);
    if ($round) {
      $round->setStatus(Round::INACTIVE);
      $round->save();
    }

    $game->setStatus(Game::ARCHIVE);
    $game->save();
  }

    /**
   * Funkcja zwraca poczatkowe wartosci gry dla 'game_data_json' wstawiane podczas aktywacji gry.
   * @return array Tablica PHP z danymi gry
   */
  public static function getGameJsonData() {
    
  }

  /**
   * Funkcja zwraca poczatkowe wartosci rundy dla 'round_data_json' wstawiane podczas tworzenia pierwszej rundy.
   * @return array Tablica PHP z danymi gry
   */
  public static function getFirstRoundDataJson() {
    
  }

  /**
   * Funkcja zwraca poczatkowe wartosci rundy dla 'round_decision_data_json' wstawiane podczas tworzenia pierwszej rundy.
   * @return array Tablica PHP z danymi gry
   */
  public static function getFirstRoundDecisionDataJson() {
    
  }

  /**
   * Funkcja zwraca poczatkowe wartosci rundy dla 'round_raport_data_json' wstawiane podczas tworzenia pierwszej rundy.
   * @return array Tablica PHP z danymi gry
   */
  public static function getFirstRoundRaportDataJson() {
    
  }

}
