<?php

/**
 * Klasa odpowiada za wygenerowanie kodu HTML (oraz PHP) zawierajacego
 * ikonke rozwijania menu akcji i menu akcji z linkami. Klasa zwiazana z 
 * plikiem manageForm.js
 *

 */
abstract class BaseFmActionBar {

  protected $settings = array();
  protected $links = array();
  protected $object = null;

  /**
   * Konstruktor przyjmuje ustawienia oraz nazwe klasy collectora.
   * 
   * @param object $object Obiekt artykulu lub kategorii
   * @param array $settings
   * @param string $dataCollectorClass 
   */
  public function __construct($object, $settings = array()) {

    artEscaper::escape($object);
    $this->initSettings($settings);
    $this->object = $object;

    // Rejestrujemy dostepne linki
    $this->registerLink('article_edit');
    $this->registerLink('article_photo_add');
    $this->registerLink('article_photo_list_order');
    $this->registerLink('article_photo_list');
    $this->registerLink('article_meta_edit');
    
    $this->registerLink('article_add');
    $this->registerLink('category_edit');
    $this->registerLink('comment_edit');
    
    $this->registerLink('article_layout');
    $this->registerLink('category_layout');
    $this->registerLink('article_delete');
    $this->registerLink('comment_delete');
    $this->registerLink('category_delete');
  }

  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow.
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = $settings + $this->getDefaultSettings();
  }

  /**
   * Funkcja zwraca wartosci default dla ustawien.
   * 
   * @return array
   */
  protected function getDefaultSettings() {
    return array(
      'blacklist' => null, // Tablica z wyłaczonymi linkami dod anego FM AcB.
      'whitelist' => null, // Tablica z właczonymi linkami dod anego FM AcB.
      'enabled' => true,
      'empty_url' => 'javascript:void(0);',
      'div' => array(
        'class' => 'options-container pa',
      ),
      'options_icon' => array(
        'class' => 'cp megaicon1-widget frontend-management',
        'onclick' => 'toggleActionBar(this);',
      ),
      'ul' => array(
        'class' => 'action-bar hide actionbar-menu indicator',
      ),        
      'li' => array(
      ),
      'a' => array(
      ),
      'i' => array(
          'class' => '',
      ),
    );
  }

  /**
   * Rejestruje link do wstawiania do menu zarzadzania
   */
  protected function registerLink($link) {
    if (isset($this->settings['blacklist']) && is_array($this->settings['blacklist'])) {
      $blackList = $this->settings['blacklist'];

      if (!in_array($link, $blackList)) {
        $this->links[] = $link;
      }
    } elseif (isset($this->settings['whitelist']) && is_array($this->settings['whitelist'])) {
      $whiteList = $this->settings['whitelist'];

      if (in_array($link, $whiteList)) {
        $this->links[] = $link;
      }
    } else {
      $this->links[] = $link;
    }
  }

  protected function prepareLink() {
    $data = array(
      'li' => $this->settings['li'],
      'a' => $this->settings['a'],
      'i' => $this->settings['i'],
    );

    $data['a']['href'] = $this->settings['empty_url'];

    return $data;
  }

  protected function getArticleEdit() {
    if (get_class($this->object) == 'Article' && $this->object->hasContents()) {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articleEdit(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-pencil";
      $data['text'] = __('Edytuj treść');
      return $data;
    } else {
      return false;
    }
  }

  protected function getCategoryEdit() {
    if (get_class($this->object) == 'Category') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.categoryEdit(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-pencil";
      $data['text'] = __('Edytuj kategorię');
      return $data;
    } else {
      return false;
    }
  }

  protected function getArticleDelete() {
    if (get_class($this->object) == 'Article' && $this->object->getUniqueName() == '') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articleDelete(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-trash";
      $data['text'] = __('Usuń artykuł');
      return $data;
    } else {
      return false;
    }
  }
  
  protected function getCommentEdit() {
    if (get_class($this->object) == 'ArticleComment') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.commentEdit(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-pencil";
      $data['text'] = __('Edytuj komentarz');
      return $data;
    } else {
      return false;
    }
  }

  protected function getCommentDelete() {
    if (get_class($this->object) == 'ArticleComment') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.commentDelete(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-trash";
      $data['text'] = __('Usuń komentarz');
      return $data;
    } else {
      return false;
    }
  }

  protected function getCategoryDelete() {
    if (get_class($this->object) == 'Category') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.categoryDelete(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-trash";
      $data['text'] = __('Usuń kategorię');
      return $data;
    } else {
      return false;
    }
  }

  protected function getArticleAdd() {
    if (get_class($this->object) == 'Category') {
      $data = $this->prepareLink();

      $layout = $this->object->getLayout();
      $data['a']['onclick'] = "ManageForm.articleAdd(this, '" . $layout->getArticleExampleUniqueName() . "', '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-plus";

      $data['text'] = __('Dodaj artykuł');

      $module = $layout->getModule();

      switch ($module) {
        case 'reference_category':
          $data['text'] = __('Dodaj referencję');
          break;
        case 'article_category':
          $data['text'] = __('Dodaj artykuł');
          break;
        case 'blog_category':
          $data['text'] = __('Dodaj post');
          break;
        default:
          $data['text'] = __('Dodaj element');
          break;
      }

      return $data;
    } else {
      return false;
    }
  }

  protected function getArticleLayout() {
    if (get_class($this->object) == 'Article') {
      $data = $this->prepareLink();
      
      $data['a']['onclick'] = "ManageForm.articleLayoutEdit(this, '" . $this->object->getId() . "', '" . $this->object->getLayout()->getModule() . "');";
      $data['i']['class'] .= " megaicon1-list-thumbnails";
      $data['text'] = __('Zmień wygląd');
      return $data;
    } else {
      return false;
    }
  }

  protected function getArticleMetaEdit() {
    if (get_class($this->object) == 'Article') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articleMetaEdit(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-share";
      $data['text'] = __('Edytuj metadane');
      return $data;
    } else {
      return false;
    }
  }

  protected function getArticlePhotoListOrder() {
    if (get_class($this->object) == 'Article'
        && $this->object->hasGallery()) {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articlePhotoListOrder(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-ionicons-86";
      $data['text'] = __('Ustaw kolejność zdjęć');
      return $data;
    } else {
      return false;
    }
  }
  
  protected function getArticlePhotoList() {
    if (get_class($this->object) == 'Article'
        && $this->object->hasGallery()) {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articlePhotoList(this, '" . $this->object->getId() . "', '1');";
      $data['i']['class'] .= " megaicon1-list";
      $data['text'] = __('Zarządzaj zdjęciami');
      return $data;
    } else {
      return false;
    }
  }
  
  protected function getArticlePhotoAdd() {
    if (get_class($this->object) == 'Article'
        && $this->object->hasGallery()) {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.articlePhotoAdd(this, '" . $this->object->getId() . "');";
      $data['i']['class'] .= " megaicon1-plus";
      $data['text'] = __('Dodaj zdjęcie');
      return $data;
    } else {
      return false;
    }
  }

  protected function getCategoryLayout() {
    if (get_class($this->object) == 'Category') {
      $data = $this->prepareLink();
      $data['a']['onclick'] = "ManageForm.categoryLayout(this, '" . $this->object->getId() . "', '" . $this->object->getLayout()->getModule() . "');";
      $data['i']['class'] .= " megaicon1-list-thumbnails";
      $data['text'] = __('Zmień wygląd');
      return $data;
    } else {
      return false;
    }
  }

  /**
   * Funkcja zwraca HTML i kod PHP na podstawie podanego
   * klucza linku do wygenerowania $link i przekazego elementu
   * 
   * @param string $link
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  protected function generateLink($link, $linkData) {
    $content = tag('li', $linkData['li'], true);
  
    $content .= tag('a', $linkData['a'], true);

    $content .= content_tag('i', '', $linkData['i']).' '.$linkData['text'];

    $content .= '</a></li>';

    return $content;
  }

  protected function checkLinks() {
    foreach ($this->links as $link) {
      if ($this->checkLink($link)) {
        return true;
      }
    }

    return false;
  }

  protected function getUser() {
    return sfContext::getInstance()->getUser();
  }

  protected function checkLink($link) {
    $method = 'get' . sfInflector::camelize($link);
    if (method_exists($this, $method)) {
      if (!$this->getUser()->hasFmLink($link)) {
        return false;
      }

      $generate = $this->$method();

      if ($generate === false) {
        return false;
      } else {
        return true;
      }
    } else {
      throw new sfException('Brak metody "' . $method . '" do generowania linka.');
    }
  }

  protected function generateLinks() {
    $content = '';
    foreach ($this->links as $link) {
      // Sprawdzamy czy metoda istnieje i uzytkownik ma prawo do wyswietlenia linku
      if ($this->checkLink($link)) {
        $method = 'get' . sfInflector::camelize($link);
        $linkData = $this->$method();
        $content .= $this->generateLink($link, $linkData);
      }
    }

    return $content;
  }

  /**
   * Funkcja odpowiedzialna za caly proces tworzenia ikonki i menu paska akcji.
   * Jesli 'enabled' = false lub 'app_fm_enabled' = false - nic nam nie zwroci. 
   * W przeciwnym wypadku zwraca kod HTML i PHP zawierajacy ikonke i menu paska
   * akcji.
   * 
   * @return string 
   */
  public function generateActionBar() {
    // Sprawdzamy czy jest wszystko powlaczane i czy wogule uzytkownik ma prawa
    // do wyswietlenia chociaz jednego linku
    if (!$this->settings['enabled'] || !sfConfig::get('app_fm_enabled', false) || !$this->checkLinks() || !$this->getUser()->hasAnyFmLink($this->links)) {
      // Jesli jest nie aktywny lub wylaczony w app.yml to nic nie zwracaj
      return '';
    } else {
      $content = tag('div', $this->settings['div'], true);
      $content .= tag('i', $this->settings['options_icon'], true);

      $content .= content_tag('ul', $this->generateLinks(), $this->settings['ul']);

      $content .= '</i>';
      $content .= '</div>';
      
      return $content;
    }
  }

  public function __toString() {
    try {
      return $this->generateActionBar();
    } catch (Exception $ex) {
      return $ex->getMessage();
    }
  }

}
