<?php

/**
 * Klasa odpowiada za wygenerowanie kodu HTML (oraz PHP) zawierajacego
 * ikonke rozwijania menu akcji i menu akcji z linkami. Klasa zwiazana z 
 * plikiem actionBar.js
 *

 */
abstract class BaseGroupMenuActionBar extends GroupMenuFunctions {

  protected $settings = array();
  protected $links = array();
  protected $culture = null;

  /**
   * Konstruktor przyjmuje ustawienia oraz nazwe klasy collectora.
   * 
   * @param array $settings
   */
  public function __construct($settings, $culture) {
    $this->initSettings($settings);
    $this->culture = $culture;
    // Rejestrujemy dostepne linki
    $this->registerLink('menu_item_add');
    $this->registerLink('menu_item_edit');
    $this->registerLink('menu_item_delete');
    $this->registerLink('menu_item_list_order');
    $this->registerLink('menu_group_add');
    $this->registerLink('menu_group_edit');
    $this->registerLink('menu_group_delete');

    $this->registerLink('order_up');
    $this->registerLink('order_down');
  }

  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow.
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace($this->getDefaultSettings(), $settings);
  }

  /**
   * Funkcja zwraca wartosci default dla ustawien.
   * 
   * @return array
   */
  protected function getDefaultSettings() {
    return array(
        'enabled' => true,
        'empty_url' => 'javascript:void(0);',
        'div' => array(
            'class' => 'options-container container_menu menu pa',
        ),
        'options_icon' => array(
            'class' => 'cp megaicon1-widget frontend-management',
            'onclick' => 'toggleActionBar(this);',
        ),
        'ul' => array(
            'class' => 'action-bar hide actionbar-menu indicator',
        ),
        'li' => array(
        ),
        'a' => array(
        ),
        'i' => array(
            'class' => '',
        ),
    );
  }

  /**
   * Rejestruje link do wstawiania do menu zarzadzania
   */
  protected function registerLink($link) {
    $this->links[] = $link;
  }

  protected function prepareLink($menuItem) {
    $data = array(
        'li' => $this->getSettings('li', $menuItem),
        'a' => $this->getSettings('a', $menuItem),
        'i' => $this->getSettings('i', $menuItem),
    );

    $data['a']['href'] = $this->getSettings('empty_url', $menuItem);

    return $data;
  }

  protected function getMenuItemAdd($menuItem) {
    if ($menuItem['submenu_id']) {
      $data = $this->prepareLink($menuItem);
      if ($this->hasCategory($menuItem)) {
        $data['a']['onclick'] = "ManageForm.menuItemAdd(this, '" . $menuItem['submenu_id'] . "', '" . $menuItem['Translation'][$this->culture]['url'] . "');";
      } else {
        $data['a']['onclick'] = "ManageForm.menuItemAdd(this, '" . $menuItem['submenu_id'] . "');";
      }
      $data['i']['class'] .= " megaicon1-plus-1";
      $data['text'] = __('Dodaj pozycję');
      return $data;
    } else {
      return false;
    }
  }

  protected function getMenuItemEdit($menuItem) {
    $data = $this->prepareLink($menuItem);
    $data['a']['onclick'] = "ManageForm.menuItemEdit(this, '" . $menuItem['id'] . "');";
    $data['i']['class'] .= " megaicon1-pencil";
    $data['text'] = __('Edytuj pozycję');

    return $data;
  }

  protected function getMenuItemDelete($menuItem) {
    $data = $this->prepareLink($menuItem);
    $data['a']['onclick'] = "ManageForm.menuItemDelete(this, '" . $menuItem['id'] . "');";
    $data['i']['class'] .= " megaicon1-trash-1";
    $data['text'] = __('Usuń pozycję');

    return $data;
  }

  protected function getMenuItemListOrder($menuItem) {
    if ($menuItem['submenu_id']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuItemListOrder(this, '" . $menuItem['submenu_id'] . "');";
      $data['i']['class'] .= " megaicon1-ionicons-86";
      $data['text'] = __('Ustaw kolejność');

      return $data;
    } else {
      return false;
    }
  }

  protected function getMenuGroupAdd($menuItem) {
    if (!$menuItem['submenu_id']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuGroupAdd(this, '" . $menuItem['id'] . "');";
      $data['i']['class'] .= " megaicon1-plus";
      $data['text'] = __('Dodaj submenu');
      return $data;
    } else {
      return false;
    }
  }

  protected function getMenuGroupEdit($menuItem) {
    if ($menuItem['submenu_id']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuGroupEdit(this, '" . $menuItem['submenu_id'] . "');";
      $data['i']['class'] .= " megaicon1-pencil";
      $data['text'] = __('Edytuj submenu');
      return $data;
    } else {
      return false;
    }
  }

  protected function getMenuGroupDelete($menuItem) {
    if ($menuItem['submenu_id']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuGroupDelete(this, '" . $menuItem['submenu_id'] . "');";
      $data['i']['class'] .= " megaicon1-trash";
      $data['text'] = __('Usuń submenu');
      return $data;
    } else {
      return false;
    }
  }

  protected function getOrderUp($menuItem) {
    if (!$menuItem['last']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuItemOrderUp(this, '" . $menuItem['id'] . "');";
      $data['i']['class'] .= " megaicon1-arrow-right-1";
      $data['text'] = __('Przesuń w prawo');
      return $data;
    } else {
      return false;
    }
  }

  protected function getOrderDown($menuItem) {
    if (!$menuItem['first']) {
      $data = $this->prepareLink($menuItem);
      $data['a']['onclick'] = "ManageForm.menuItemOrderDown(this, '" . $menuItem['id'] . "');";
      $data['i']['class'] .= " megaicon1-arrow-left-1";
      $data['text'] = __('Przesuń w lewo');
      return $data;
    } else {
      return false;
    }
  }

  /**
   * Funkcja zwraca HTML i kod PHP na podstawie podanego
   * klucza linku do wygenerowania $link i przekazego elementu
   * 
   * @param string $link
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  protected function generateLink($link, $linkData, $menuItem) {
    // Sprawdzenie czy user moze widziec dany link
    $content = '<?php if ($sf_user->hasFmLink(\'' . $link . '\')) : ?>';

    $content .= tag('li', $linkData['li']);
    $content .= tag('a', $linkData['a']);

    $content .= content_tag('i', '', $linkData['i']).' '.$linkData['text'];

    $content .= '</a></li>';

    $content .= '<?php endif; ?>';
    return $content;
  }

  protected function checkLinks($menuItem) {
    foreach ($this->links as $link) {
      if ($this->checkLink($link, $menuItem)) {
        return true;
      }
    }

    return false;
  }

  protected function checkLink($link, $menuItem) {
    $method = 'get' . sfInflector::camelize($link);
    if (method_exists($this, $method)) {
      $generate = $this->$method($menuItem);

      if ($generate === false) {
        return false;
      } else {
        return true;
      }
    } else {
      throw new sfException('Brak metody "' . $method . '" do generowania linka.');
    }
  }

  protected function generateLinks($menuItem) {
    $content = '';
    foreach ($this->links as $link) {
      $generate = true;
      if ($this->hasSettings($link, $menuItem)) {
        // Mozemy wylaczyc generowanie konkretnego linku w JSON ustawiajac tam FALSE
        $generate = $this->getSettings($link, $menuItem);
      }

      if ($generate && $this->checkLink($link, $menuItem)) {
        $method = 'get' . sfInflector::camelize($link);
        $linkData = $this->$method($menuItem);
        $content .= $this->generateLink($link, $linkData, $menuItem);
      }
    }

    return $content;
  }

  /**
   * Funkcja odpowiedzialna za caly proces tworzenia ikonki i menu paska akcji.
   * Jesli 'enabled' = false lub 'app_fm_enabled' = false - nic nam nie zwroci. 
   * W przeciwnym wypadku zwraca kod HTML i PHP zawierajacy ikonke i menu paska
   * akcji. 
   * 
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  public function generateActionBar($menuItem) {
    if (!$this->settings['enabled'] || !sfConfig::get('app_fm_enabled', false) || !$this->checkLinks($menuItem)) {
      // Jesli jest nie aktywny lub wylaczony w app.yml to nic nie zwracaj
      return;
    } else {
      // Sprawdzamy czy warto wyswietlac menu (czy bedzie jakikolwiek link)
      $content = '<?php if ($sf_user->hasAnyFmLink(' . $this->generateLinksArrayString() . ')) : ?>';

      $content .= tag('div', $this->getSettings('div', $menuItem));
      $content .= tag('i', $this->getSettings('options_icon', $menuItem));

      $content .= content_tag('ul', $this->generateLinks($menuItem), $this->getSettings('ul', $menuItem));

      $content .= '</i>';
      $content .= '</div>';
      $content .= '<?php endif; ?>';

      return $content;
    }
  }

  /**
   * Funkcja tworzy string (reprezentujacy array) na podstawie kluczy 
   * z funkcji getDefaultLinksSettings().
   * np. array('article_add','article_edit',...,'reference_delete')
   * 
   * @return string 
   */
  protected function generateLinksArrayString() {
    $string = 'array(';

    $links = array_keys($this->links);
    foreach ($links as $link) {
      $string .= '\'' . $link . '\',';
    }
    $string = substr($string, 0, -1);
    $string .= ')';
    return $string;
  }

}
