<?php
/**
 * Klasa odpowiada za pobranie danych do menu dla danej kategorii (reprezentowanej
 * przez unique_name) i jezyka.
 *

 */
abstract class BaseGroupMenuDataCollector extends GroupMenuFunctions {
  
  protected $uniqueName = null;
  protected $culture = null;
  protected $settings = array();
  
  protected $menuGroups = array();
  protected $categories = array();
  protected $articles = array();
  
  protected $categoryIds = array();
  protected $articleIds = array();
  
  /**
   * Konstruktor przyjmuje kategorie w formie unique_name, tablice ustawien i jezyk
   * 
   * @param string $category
   * @param array $settings
   * @param string $culture 
   */
  public function __construct($uniqueName, $settings, $culture) {
    $this->uniqueName = $uniqueName;
    $this->culture = $culture;
    $this->initSettings($settings);
    $this->prepareData();
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * ustawien.
   * 
   * @param array $settings
   * @return array 
   */
  public function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien
   * 
   * @return array
   */
  public function getDefaultSettings()
  {
    return array(
        'depth' => 2,
      );
  }
  
  /**
   * Funkcja przygotowuje dane na podstawie kategorii, jezyka i ustawien. Wywoluje
   * zestaw funkcji odpowiedzialny za przygotowanie danych
   * 
   * @throws sfException 
   */
  protected function prepareData() {
    // Pobranie danych na 1 poziomie
    $menuGroups = Doctrine::getTable('MenuGroup')->getByUniqueName( $this->uniqueName );
    // Zabezpieczenie przed nie istniejaca lub pusta kategoria
    if (empty($menuGroups)) {
      throw new sfException('Podana MenuGroup o unique_name "'.$this->uniqueName.'" nie istnieje lub jest pusta.');
    }
    
    // Przetwarzanie rekordow danych, zbieranie ID artykulow i kategorii, tworzenie hierarchii
    $this->parseMenuGroups($menuGroups, 1);
    // JSON zostaje ustawiony w postaci tablicy
    $this->initJson();
    
    // Usuniecie duplikatow ID w kategoriach i artykulach
    $this->categoryIds = $this->removeDuplicates($this->categoryIds);
    $this->articleIds = $this->removeDuplicates($this->articleIds);
    
    // Pobranie i zapisanie danych o artykulach i kategoriach
    $this->initData();
  }
  
  /**
   * Dodaje podana MenuGroup do tablicy $this->menuGroups, jesli nie ma w niej jeszcze
   * MenuGroup o takim ID. Zwraca true jesli udalo sie dodac lub false wpp.
   * 
   * @param array $menuGroup Tablica z danymi MenuGroup
   * @return boolean Czy pomyslnie dodano MenuGroup
   * @throws sfException
   */
  protected function addMenuGroup($menuGroup) {
    if (isset($menuGroup['id'])) {
      if (!$this->existMenuGroup($menuGroup['id'])) {
        $this->menuGroups[] = $menuGroup;
        return true;
      } else {
        return false;
      }
    } else {
      throw new sfException('Podana tablica MenuGroup nie zawiera ID.');
    }
  }
  
  /**
   * Sprawdza czy w tablicy $this->menuGroups istnieje MenuGroup o podanym ID
   * 
   * @param integer $id ID MenuGroup
   * @return boolean
   */
  protected function existMenuGroup($id) {
    foreach ($this->menuGroups as $menuGroup) {
      if ($menuGroup['id'] == $id) {
        return true;
      }
    }
    
    return false;
  }
  
  /**
   * Zwraca MenuGroup o podanym ID
   * 
   * @param integer $id ID MenuGroup
   * @return aray
   */
  public function getMenuGroup($id) {
    foreach ($this->menuGroups as $menuGroup) {
      if ($menuGroup['id'] == $id) {
        return $menuGroup;
      }
    }
    
    return false;
  }
    
  /**
   * Funkcja rekurencyjna. Tworzy hierarchie na podstawie przekazanych rekordow.
   * Rozszerza tablice ID kategorii i artykulow o nowo przekazane ID elementow.
   * Posiada ograniczenie glebokosci parsowania do $this->settings['depth']
   * 
   * @param array $records
   * @param array $categories
   * @param array $articles
   * @param array $hierarchy
   * @param int $depth
   */
  protected function parseMenuGroups($menuGroups, $depth) {
    if ($depth > $this->settings['depth'] ) {
      // Jesli przekroczylismy glebokosc to konczymy
      return;
    }
    
    $newMenuGroups = array();
    foreach ($menuGroups as $menuGroup) {
      $added = $this->addMenuGroup($menuGroup);
      if ($added) {
        // Pomyslnie dodana grupa = jeszcze nie przerabialismy
        foreach ($menuGroup['MenuItem'] as $menuItem) {
          if ($menuItem['url_type'] == self::URL_ARTICLE
                  || $menuItem['url_type'] == self::URL_ARTICLE_TITLE) {
            // MenuItem zwiazany z artykulem, dodaj artykul do sciagniecia
            $this->articleIds[] = $menuItem['Translation'][$this->culture]['url'];
          }
          if ($menuItem['url_type'] == self::URL_CATEGORY
                  || $menuItem['url_type'] == self::URL_CATEGORY_TITLE) {
            // MenuItem zwiazany z artykulem, dodaj artykul do sciagniecia
            $this->categoryIds[] = $menuItem['Translation'][$this->culture]['url'];
          }
          if (!$this->existMenuGroup($menuItem['submenu_id'])) {
            // MenuItem ma MenuGroup jako submenu ktorej jeszcze nie pobralismy
            $newMenuGroups[] = $menuItem['submenu_id'];
          }
        }   
      }
    }

    if (count($newMenuGroups) > 0) {
      $menuGroupsToParse = Doctrine::getTable('MenuGroup')->getByIds( $newMenuGroups );
      $this->parseMenuGroups($menuGroupsToParse, $depth + 1);
    }
  }
  
  /**
   * Inicjuje dane dotyczace elementow. Rozdziela tablice $this->data na dwie tablice
   * zlozone z elementow danego typu.
   */
  protected function initData() {
    $categories = Doctrine::getTable('Category')->getCategoriesByIds($this->categoryIds, $this->culture);
    foreach ($categories as $category) {
      $this->categories[$category['id']] = $category;
    }
    
    $articles = Doctrine::getTable('Article')->getArticlesByIds($this->articleIds, $this->culture);
    foreach ($articles as $article) {
      $this->articles[$article['id']] = $article;
    }
  }
  
  /**
   * Ustawia dane JSON w postaci tablicy PHP wewnatrz kazdego z MenuItem
   */
  protected function initJson() {
    foreach ($this->menuGroups as &$menuGroup) {
      foreach ($menuGroup['MenuItem'] as &$menuItem) {
        if (isset($menuItem['json'])) {
          $menuItem['json'] = json_decode($menuItem['json'], true);
        }
      }
    }
  }
  
  /**
   * Zwraca glowna MenuGroup. Zakladamy ze pierwsza dodana MenuGroup jest glowna,
   * czyli ta z ktorej tworzymy cale menu
   * @return array
   * @throws sfException
   */
  public function getMainMenuGroup() {
    if (!isset($this->menuGroups[0])) {
      throw new sfException('Brak pierwszej grupy zawierajacej elementu menu.');
    }
    
    return $this->menuGroups[0];
  }
  
  /**
   * Funkcja usuwa z tablicy powtarzajace sie wartosci.
   * 
   * @param array $array
   * @return array 
   */
  public function removeDuplicates($array) {
    return array_unique($array);
  }
  
  /**
   * Zwraca artykul o podanym ID. Jesli artykul o takim ID nie zostal pobrany
   * wyrzucany jest wyjatek
   * @param integer $id ID Artykulu
   * @return array
   * @throws sfException
   */
  public function getArticle($id) {
    if (!isset($this->articles[$id])) {
      throw new sfException('Brak pobranego artykulu od ID "'.$id.'" dla języka "'.$this->culture.'".');
    } else {
      return $this->articles[$id];
    }
  }
  
  public function getArticles(){
    return $this->articles; 
  }
  
  /**
   * Zwraca kategorie o podanym ID. Jesli kategoria o takim ID nie zostala pobrana
   * wyrzucany jest wyjatek
   * @param integer $id ID Kategorii
   * @return array
   * @throws sfException
   */
  public function getCategory($id) {
    if (!isset($this->categories[$id])) {
      throw new sfException('Brak pobranej kategorii od ID "'.$id.'"');
    } else {
      return $this->categories[$id];
    }
  }
  
  public function getCategories(){
    return $this->categories;
  }
}

