<?php

/**
 * Klasa odpowiada za wygenerowanie kodu HTML i PHP zawierajacego pelne menu.
 *

 */
abstract class BaseGroupMenuGenerator extends GroupMenuFunctions {

  protected $settings = array();
  protected $dataCollector = null;
  protected $dataCollectorClass = null;
  protected $actionBar = null;
  protected $culture = null;

  /**
   * Konstruktor przyjmuje tablice ustawien oraz zainicjowane klasy kolektora
   * i actionBar'a
   * 
   * @param array $settings
   * @param object $dataCollector
   * @param object $actionBar 
   */
  public function __construct($settings, $dataCollector, $actionBar, $culture) {
    $this->initSettings($settings);
    $this->dataCollector = $dataCollector;
    $this->dataCollectorClass = get_class($dataCollector);

    $this->actionBar = $actionBar;

    $this->culture = $culture;
  }

  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow $settings
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace($this->getDefaultSettings(), $settings);
  }

  /**
   * Funkcja zwraca wartosci default dla ustawien
   * 
   * @return array
   */
  protected function getDefaultSettings() {
    return array(
      'active_class' => 'active',
      'first_class' => 'first',
      'last_class' => 'last',
      'li' => array(
        'class' => 'menu-li pr',
      ),
      'ul' => array(
        'class' => 'menu-ul',
      ),
      'div' => array(
        'class' => 'menu pr',
      ),
      'div_submenu' => array(
        'class' => 'submenu',
      ),
      'span' => array(
        'class' => 'menu-span',
      ),
      'a' => array(
        'class' => 'menu-a',
      ),
      'article_url' => 'article_show',
      'category_url' => 'article_list',
      'empty_url' => 'javascript:void(0);',
      'separator' => true,
    );
  }

  /**
   * Funkcja generuje string odpowiedzialny za sprawdzenie czy Item jest aktywny
   * 
   * @param array $menuItem
   * @return string 
   */
  protected function getIsActiveString($menuItem) {
    return '<?php echo (' . get_class($this) . '::isActive(' . $this->generateStringArray($menuItem) .
        ',\'' . $this->getItemUrl($menuItem) . '\') ? \' ' . $this->settings['active_class'] . '\' : \'\'); ?>';
  }

  /**
   * Funkcja generuje string jako tablice w kodzie PHP z podanej tablicy.
   * np. array(
   * 'key1' => 'val1',
   * 'key2' => 'val2',
   * )
   * 
   * @param type $array
   * @return string 
   */
  protected function generateStringArray($array) {
    $string = 'array(';
    foreach ($array as $key => $value) {
      if (!is_array($value) && !is_object($value)) {
        $string .= '\'' . $key . '\' => \'' . addslashes($value) . '\',';
      } else {
        $string .= '\'' . $key . '\' => ' . $this->generateStringArray($value) . ',';
      }
    }
    $string .= ')';

    return $string;
  }

  /**
   * Funkcja sprawdza czy dany element jest aktualnie aktywny. Jesli tak zostaje
   * dodana klasa $this->settings['is_active']
   * 
   * @param array $element
   * @param array $elementData
   * @return boolean 
   */
  public static function isActive($menuItem, $itemUrl) {
    if (isset($menuItem['json']['active_equal']) && $menuItem['json']['active_equal']) {
      // Ucinamy wszystko po ? jesli istnieje
      $pos = strpos($_SERVER["REQUEST_URI"], "?"); 
      if ($pos === false) {
        $uri = $_SERVER["REQUEST_URI"];
      } else {
        $uri = substr($_SERVER["REQUEST_URI"], 0, $pos);
      }
      if ($uri == $itemUrl) {
        return true;
      } else {
        return false;
      }
    }

    if ($itemUrl && !(strpos($_SERVER["REQUEST_URI"], $itemUrl) === false)) {
      return true;
    }

    return false;
  }

  protected function getItemUrl($menuItem) {
    $url = $menuItem['Translation'][$this->culture]['url'];
    $urlType = $menuItem['url_type'];

    if ($urlType == self::URL_ARTICLE || $urlType == self::URL_ARTICLE_TITLE) {
      // URL tworzony do artykulu
      $article = $this->dataCollector->getArticle($url);
      return $article->getUrl();
    }

    if ($urlType == self::URL_CATEGORY || $urlType == self::URL_CATEGORY_TITLE) {
      // URL tworzony do kategorii
      $category = $this->dataCollector->getCategory($url);
      return $category->getUrl();
    }

    if ($urlType == self::URL_ROUTING) {
      return url_for('@' . $url);
    }

    if ($urlType == self::URL_EMPTY) {
      return $this->settings['empty_url'];
    }

    if ($urlType == self::URL_RAW) {
      return $url;
    }

    throw new sfException('Podany url_type "' . $urlType . '" nie jest obslugiwany.');
  }

  protected function getItemTitle($menuItem) {
    $url = $menuItem['Translation'][$this->culture]['url'];
    $urlType = $menuItem['url_type'];

    if ($urlType == self::URL_ARTICLE_TITLE) {
      // Title z artykulu
      $article = $this->dataCollector->getArticle($url);
      return $article['Translation'][$this->culture]['title'];
    }

    if ($urlType == self::URL_CATEGORY_TITLE) {
      // Name z kategorii
      $category = $this->dataCollector->getCategory($url);
      return $category['Translation'][$this->culture]['name'];
    }

    return $menuItem['Translation'][$this->culture]['title'];
  }

  /**
   * Funkcja zwraca kod HTML i PHP dla calego menu
   * 
   * @return string
   */
  public function generateMenu() {
    $menuContent = '';
    $menuGroup = $this->dataCollector->getMainMenuGroup();

    $menuItemsCount = count($menuGroup['MenuItem']);

    if ($menuItemsCount > 0) {
      $menuContent = $this->generateTopMenu($menuGroup);
      
      $managingItem = $this->actionBar->generateActionBar($this->getManagingItem($menuGroup));
      if ($managingItem) {
        $menuContent .= '<li class="main-options-container pr">'.$managingItem.'</li>';
      }

      foreach ($menuGroup['MenuItem'] as $menuItem) {
        $menuItem['first'] = ($menuItem['order'] == 1);
        $menuItem['last'] = ($menuItem['order'] == $menuItemsCount);
        $menuContent .= $this->generateItem($menuItem, 1);
      }
      
      $menuContent .= $this->generateBottomMenu();
    }

    return $menuContent;
  }

  public function generateSubmenu($menuGroup, $depth) {
    $menuContent = '';
    $menuItemsCount = count($menuGroup['MenuItem']);

    if ($menuItemsCount > 0) {
      $menuContent = $this->generateTopSubmenu();

      foreach ($menuGroup['MenuItem'] as $menuItem) {
        $menuItem['first'] = ($menuItem['order'] == 1);
        $menuItem['last'] = ($menuItem['order'] == $menuItemsCount);
        $menuContent .= $this->generateItem($menuItem, $depth);
      }

      $menuContent .= $this->generateBottomSubmenu();
    }

    return $menuContent;
  }

  /**
   * Zwraca HTML zawierajacy separator jesli pozwalaja na to ustawienia
   * 
   * @return string 
   */
  protected function generateSeparator($menuItem) {
    if (!($this->getSettings('separator', $menuItem) === false)) {
      // Generujemy separator
      if ($this->getSettings('separator', $menuItem) === true) {
        // Bez separatora
        return '<span class="separator fl"></span>';
      } else {
        // Z znakiem separatora
        return '<span class="separator fl">' . $this->getSettings('separator', $menuItem) . '</span>';
      }
    } else {
      return '';
    }
  }

  protected function generateItem($menuItem, $depth) {
    // Zapytac czy nie lepiej tutaj dac tez dane z JSON'a (dopisywac lub nadpisywac dane do taga)
    $content = '';

    $liOptions = $this->getSettings('li', $menuItem);

    if ($menuItem['first']) {
      // Pierwszy
      $liOptions['class'] .= ' ' . $this->getSettings('first_class', $menuItem);
    } else {
      // Generowanie separatora
      $content .= $this->generateSeparator($menuItem);
    }

    if ($menuItem['last']) {
      // Ostatni
      $liOptions['class'] .= ' ' . $this->getSettings('last_class', $menuItem);
    }

    // Musimy zastosowac tutaj taki trick, bo tag() nam escapuje znaki <?php
    $liOptions['class'] .= '%isActiveString%';
    $content .= str_replace('%isActiveString%', $this->getIsActiveString($menuItem), tag('li', $liOptions, true));

    $content .= tag('span', $this->getSettings('span', $menuItem), true);

    $aOptions = $this->getSettings('a', $menuItem);
    $aOptions['href'] = $this->getItemUrl($menuItem);
    if ($menuItem['url_type'] == self::URL_RAW) {
      $aOptions['target'] = '_blank';
    }
    $content .= content_tag('a', $this->getItemTitle($menuItem), $aOptions);

    $content .= '</span>';

    if ($menuItem['submenu_id']) {
      $content .= $this->generateSubmenu($this->dataCollector->getMenuGroup($menuItem['submenu_id']), ++$depth);
    }

    $content .= $this->actionBar->generateActionBar($menuItem);
    $content .= '</li>';

    return $content;
  }

  /**
   * Tworzy gore menu pierwszego poziomu
   * 
   * @return string 
   */
  protected function generateTopMenu($menuGroup) {
    $content = tag('div', $this->settings['div'], true);
    $content .= $this->actionBar->generateActionBar($this->getManagingItem($menuGroup));
    $content .= tag('ul', $this->settings['ul'], true);

    return $content;
  }

  protected function getManagingItem($menuGroup) {
    $menuItem = array(
      'submenu_id' => $menuGroup['id'],
      'url_type' => self::URL_EMPTY,
      'json' => array(
        'menu_item_edit' => false,
        'menu_item_delete' => false,
        'menu_group_add' => false,
        'article_edit' => false,
        'category_edit' => false,
        'order_up' => false,
        'order_down' => false,
        'menu_group_delete' => false,
        'options_icon' => array(
          'class' => 'cp megaicon1-widget main',
          'onclick' => 'toggleActionBar(this);',
        ),
      )
    );

    return $menuItem;
  }

  /**
   * Tworzy dol menu pierwszego poziomu
   * 
   * @return string 
   */
  protected function generateBottomMenu() {
    return '</ul></div>';
  }

  /**
   * Tworzy gore dla menu drugiego i nizszych poziomow
   * 
   * @return string 
   */
  protected function generateTopSubmenu() {
    $content = tag('div', $this->settings['div_submenu'], true);
    $content .= tag('ul', $this->settings['ul'], true);
    return $content;
  }

  /**
   * Tworzy dol dla submenu
   * @return string 
   */
  protected function generateBottomSubmenu() {
    return '</ul></div>';
  }

}
