<?php
/**
 * Klasa odpowiada za wygenerowanie kodu HTML (oraz PHP) zawierajacego
 * ikonke rozwijania menu akcji i menu akcji z linkami. Klasa zwiazana z 
 * plikiem actionBar.js
 *

 */
abstract class BaseActionBar {
  
  protected $settings = array();
  protected $dataCollectorClass = null;
  
  /**
   * Konstruktor przyjmuje ustawienia oraz nazwe klasy collectora.
   * 
   * @param array $settings
   * @param string $dataCollectorClass 
   */
  public function __construct($settings, $dataCollectorClass) {
    $this->initSettings($settings);
    $this->dataCollectorClass = $dataCollectorClass;
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow.
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien. Wykorzystuje funkcje 
   * getDefaultLinksSettings() do pobrania ustawien linkow
   * 
   * @return array
   */
  protected function getDefaultSettings()
  {
    return array_merge(
       array(
        'enabled' => true,
        
        'empty_url' => 'javascript:void(0);',
        
        'div_class' => 'options-container',
        'options_class' => 'fr icon-cog pointer options-icon',
        'ul_class' => 'action-bar hide dropdown-menu indicator',
        'li_class' => 'w100 fl',
        'span_class' => 'mab-link w100 fl',
        'a_class' => 'w100 p0 fl',
        'a_text_class' => 'link-text fl',
      ),
      $this->getDefaultLinksSettings()
    );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien linkow. Jest odlaczona od
   * getDefaultSettings(), aby mozna bylo pobrac jakie linki mamy generowac.
   * Na podstawie kluczy w zwracanej tablicy sa generowane linki przez
   * funkcje generateOptionsMenu()
   * 
   * @return array
   */
  protected function getDefaultLinksSettings() {
    return array(
        'article_add' => true,
        'article_edit' => true,
        'article_meta_edit' => true,
        'article_gallery' => true,
        'article_delete' => true,
        
        'reference_add' => true,
        'reference_edit' => true,
        'reference_delete' => true,
        
        'category_add' => true,
        'category_edit' => true,
        'category_delete' => true,
        
        'order_up' => true,
        'order_down' => true,
    );
  }
  
  /**
   * Funkcja tworzy dane o linku na podstawie przekazanych elementow. Dane sa
   * wykorzystywane przez funkcje generateLink()
   * 
   * @param string $link
   * @param array $element
   * @param array $elementData
   * @return array 
   */
  protected function getLinkData($link, $element, $elementData) {
    $data = array();
    // Wszystkie URL akurat sa na onclick'u
    $data['url'] = false;
    // Dodajemy standardowe klasy zeby ladnie link wygladal
    $data['class'] = 'fl mt2 mr5 ml5 ';
    
    switch ($link) {
      case 'article_add':
        $data['onclick'] = "fmArticleAdd($(this), '".$element['id']."');";
        $data['class'] .= "icon-plus";
        $data['text'] = __('Dodaj artykuł');
        break;
      case 'article_edit':
        $data['onclick'] = "fmArticleEdit($(this), '".$element['id']."');";
        $data['class'] .= ' icon-pencil';
        $data['text'] = __('Edytuj artykuł');
        break;
      case 'article_meta_edit':
        $data['onclick'] = "fmArticleMetaEdit($(this), '".$element['id']."');";
        $data['class'] .= 'icon-wrench';
        $data['text'] = __('Edytuj metadane');
        break;
      case 'article_delete':
        $data['onclick'] = "fmArticleDelete($(this), '".$element['id']."');";
        $data['class'] .= 'icon-remove';
        $data['text'] = __('Usuń artykuł');
        break;
      case 'article_gallery':
        $data['onclick'] = "fmGalleryEdit($(this) ,'".$element['id']."');";
        $data['class'] .= ' icon-picture';
        $data['text'] = __('Edytuj galerię');
        break;
      case 'reference_add':
        $data['onclick'] = "fmReferenceAdd($(this), '".$element['id']."');";
        $data['class'] .= "icon-plus";
        $data['text'] = __('Dodaj referencję');
        break;
      case 'reference_edit':
        $data['onclick'] = "fmReferenceEdit($(this), '".$element['id']."');";
        $data['class'] .= " icon-pencil";
        $data['text'] = __('Edytuj referencję');
        break;
      case 'reference_delete':
        $data['onclick'] = "fmReferenceDelete($(this), '".$element['id']."');";
        $data['class'] .= "icon-remove";
        $data['text'] = __('Usuń referencję');
        break;
      case 'category_add':
        $data['onclick'] = "fmCategoryAdd($(this), '".$element['id']."');";
        $data['class'] .= 'icon-plus';
        $data['text'] = __('Dodaj kategorię');
        break;
      case 'category_edit':
        $data['onclick'] = "fmCategoryEdit($(this), '".$element['id']."');";
        $data['class'] .= ' icon-pencil';
        $data['text'] = __('Edytuj kategorię');
        break;
      case 'category_delete':
        $data['onclick'] = "fmCategoryDelete($(this), '".$element['id']."');";
        $data['class'] .= 'icon-remove';
        $data['text'] = __('Usuń kategorię');
        break;
      case 'order_up':
        $data['onclick'] = "fmOrderUp($(this), '".$element['article_category_id']."');";
        $data['class'] .= 'icon-arrow-right';
        $data['text'] = __('Przesuń w prawo');
        break;
      case 'order_down':
        $data['onclick'] = "fmOrderDown($(this), '".$element['article_category_id']."');";
        $data['class'] .= 'icon-arrow-left';
        $data['text'] = __('Przesuń w lewo');
        break;
    }
    
    return $data;
  }
  
  /**
   * Funkcja zwraca HTML i kod PHP na podstawie podanego
   * klucza linku do wygenerowania $link i przekazego elementu
   * 
   * @param string $link
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  protected function generateLink($link, $element, $elementData) {
    $data = $this->getLinkData($link, $element, $elementData);
    // Sprawdzenie czy user moze widziec dany link
    $content = '<?php if ($sf_user->hasFmLink(\''.$link.'\')) : ?>';

    $content .= '<li class="'.$element['li_class'].'">';
    $content .= '<span class="'.$element['span_class'].'">';
    
    if (isset($data['url']) && $data['url']) {
      $content .= '<a href="'.url_for($data['url']).'"';  
    } else {
      $content .= '<a href="'.$element['empty_url'].'"';
    }
    
    if (isset($data['onclick']) && $data['onclick']) {
      $content .= ' onclick="'.$data['onclick'].'"';
    }
    
    $content .= ' class="'.$element['a_class'].'">';
    $content .= '<i class="'.$data['class'].'"></i>';
    $content .= '<span class="'.$element['a_text_class'].'">'.$data['text'].'</span>';
    $content .= '</a></span></li>';
    $content .= '<?php endif; ?>';
    return $content;
  }
  
  /**
   * Funkcja tworzy HTML dla linkow pojawiajacych sie w rozwijanym menu.
   * 
   * @param array $element
   * @param array $elementData
   * @return string
   */
  protected function generateOptionsMenu($element, $elementData) {
    $content = '<ul class="'.$element['ul_class'].'">';
    
    $links = array_keys($this->getDefaultLinksSettings());
    foreach ($links as $link) {
      if ($element[$link]) {
        $content .= $this->generateLink($link, $element, $elementData);
      }
    }

    $content .= '</ul>';
    
    return $content;
  }
  
  /**
   * Funkcja tworzy HTML otwarcia dla ikonki rozwijajacej menu z linkami.
   * 
   * @param array $element
   * @param array $elementData
   * @return string
   */
  protected function generateOpenOptionsContainer($element, $elementData) {
    return '<div class="'.$element['div_class'].'">';
  }
  
  protected function generateOptionsIcon($element, $elementData) {
    return '<div class="'.$element['options_class'].'" onclick="toggleActionBar(this);"></div>';
  }
  
  /**
   * Funkcja tworzy HTML zamkniecia dla ikonki rozwijajacej menu z linkami.
   * 
   * @param array $element
   * @param array $elementData
   * @return string
   */
  protected function generateCloseOptionsContainer($element, $elementData) {
    return '</div>';
  }
  
  /**
   * Funkcja inicjuje zmienne elementu odpowiedzialne za generowanie linkow w 
   * rozwijanym menu. Merguje tablice $this->settings do elementu, aby mozna bylo
   * nadpisac ustawienia dla pojedynczego elementu w razie potrzeby.
   * 
   * @param array $element
   * @param array $elementData
   * @return array 
   */
  protected function initElement($element, $elementData) {
    $element = array_merge($this->settings, $element);
    $skipLinks = array();
    
    if ($this->isArticle($element)) {
      // Jesli jest artykulem nie wyswietlamy linkow dla kategorii oraz dodawania artykulu i referencji
      $skipLinks = array_merge($skipLinks, array('article_add', 'reference_add', 'category_add', 'category_edit', 'category_delete'));
    }
    
    if ($this->isReference($element)) {
      // Jesli jest referencja to usuwamy linki artykulow i dodawania kategorii
      $skipLinks = array_merge($skipLinks, array('article_add', 'article_edit', 'article_meta_edit', 'article_delete', 'article_gallery', 'category_add'));
    } else {
      // Jesli nie jest referencja to usuwamy linki referencji
      $skipLinks = array_merge($skipLinks, array('reference_add', 'reference_edit', 'reference_delete'));
    }
    
    if ($this->isCategory($element)) {
      // Jesli jest kategoria nie wyswietlamy linkow dla artykulu (poza dodawaniem)
      $skipLinks = array_merge($skipLinks, array('article_edit', 'article_meta_edit', 'article_delete', 'article_gallery', 'reference_edit', 'reference_delete'));
    }
    
    if ($element['first']) {
      $skipLinks = array_merge($skipLinks, array('order_down'));
    }
    
    if ($element['last']) {
      $skipLinks = array_merge($skipLinks, array('order_up'));
    }
    
    // Usuniecie niepotrzebnych linkow
    foreach ($skipLinks as $key) {
      $element[$key] = false;
    }
    
    return $element;
  }

  /**
   * Funkcja odpowiedzialna za caly proces tworzenia ikonki i menu paska akcji.
   * Jesli 'enabled' = false lub 'app_fm_enabled' = false - nic nam nie zwroci. 
   * W przeciwnym wypadku zwraca kod HTML i PHP zawierajacy ikonke i menu paska
   * akcji. Do generowania wykorzystuje tablice elementu wygenerowana przez
   * klase $this->dataCollectorClass
   * 
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  public function generateActionBar($element, $elementData) {
    if ( !$this->settings['enabled'] || !sfConfig::get('app_fm_enabled', false) ) {
      // Jesli jest nie aktywny lub wylaczony w app.yml to nic nie zwracaj
      return;
    } else {
      // Jesli jest aktywny, wykonaj logike, zwroc HTML
      $element = $this->initElement($element, $elementData);
      
      // Sprawdzamy czy warto wyswietlac menu (czy bedzie jakikolwiek link)
      $content = '<?php if ($sf_user->hasAnyFmLink('.$this->generateLinksArrayString().')) : ?>';
      
      $content .= $this->generateOpenOptionsContainer($element, $elementData);
      
      $content .= $this->generateOptionsIcon($element, $elementData);
      
      $content .= $this->generateOptionsMenu($element, $elementData);
      
      $content .= $this->generateCloseOptionsContainer($element, $elementData);
      
      $content .= '<?php endif; ?>';
      
      return $content;
    }
  }
  
  /**
   * Funkcja tworzy string (reprezentujacy array) na podstawie kluczy 
   * z funkcji getDefaultLinksSettings().
   * np. array('article_add','article_edit',...,'reference_delete')
   * 
   * @return string 
   */
  protected function generateLinksArrayString() {
    $string = 'array(';
    
    $links = array_keys($this->getDefaultLinksSettings());
    foreach ($links as $link) {
      $string .= '\''.$link.'\',';
    }
    $string = substr($string, 0, -1);
    $string .= ')';
    return $string;
  }
  
  /**
   * Sprawdza czy podany element jest artykulem.
   * (zrobione dla wygody)
   * @param array $element
   * @return boolean 
   */
  protected function isArticle($element) {
    $dataCollectorClass = $this->dataCollectorClass;
    return $dataCollectorClass::isArticle($element);
  }
  
  /**
   * Sprawdza czy podany element jest kategoria
   * (zrobione dla wygody)
   * @param array $element
   * @return boolean 
   */
  protected function isCategory($element) {
    $dataCollectorClass = $this->dataCollectorClass;
    return $dataCollectorClass::isCategory($element);
  }
  
  /**
   * Sprawdza czy podany element nalezy do referencji
   * (zrobione dla wygody)
   * @param array $element
   * @return boolean 
   */
  protected function isReference($element) {
    $dataCollectorClass = $this->dataCollectorClass;
    return $dataCollectorClass::isReference($element);
  }
}

?>
