<?php
/**
 * Description of BaseMenu
 *

 */
abstract class BaseMenu {
  protected $fileName = null;
  protected $filePath = null;
  protected $category = null;
  protected $culture = null;
  
  public function __construct($fileName) {
    $this->fileName = $fileName;
    $this->initFilePath();
    
    if (!is_readable( $this->filePath )) {
      throw new sfException('Podany plik "'.$this->filePath.'" nie istnieje lub brakuje praw do odczytu.');
    }
    
    $this->culture = sfContext::getInstance()->getUser()->getCulture();
  }
  
  /**
   * Funkcja ustawia sciezke do pliku z danymi do generowania menu
   */
  protected function initFilePath() {
    $this->filePath = sfConfig::get('sf_app_lib_dir') .'/'. $this->fileName;
  }
  
  /**
   * Funkcja usuwa klucze z wartosciami NULL
   * 
   * @param array $array 
   * @return array
   */
  public function removeNull($array) {
    $cleanArray = array();
    foreach ($array as $key => $value) {
      if (!is_null($value)) {
        if (is_array($value)) {
          $cleanArray[$key] = $this->removeNull($value);
        } else {
          $cleanArray[$key] = $value;
        }
      }
    }
    
    return $cleanArray;
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow przekazana z pliku.
   * 
   * @param array $settings
   * @return array 
   */
  public function initSettings($settings) {
    // Usuwamy klucze z wartosciami NULL, bo to znaczy ze chcemy tam miec
    // defualtowe wartosci
    $settings = $this->removeNull($settings);
    return artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla settings
   * @return array
   */
  public function getDefaultSettings()
  {
    return array(
        'options' => array(
            'category' => false,
            'data_collector' => 'MenuDataCollector',
            'action_bar' => 'ActionBar',
            'generator' => 'MenuGenerator',
            'cache' => 'MenuCache',
        ),
        'data_collector' => array(),
        'generator' => array(),
        'action_bar' => array(),
        'cache' => array(),
      );
  }
  
  /**
   * Funkcja odpowiada za zwrocenie HTML menu. Korzysta z klas ustawionych w 
   * settings.
   * 
   * @return string
   */
  public function generateMenu() {
    $settings = sfYaml::load( $this->filePath );
    $settings = $this->initSettings($settings);
    
    $dataCollectorClass = $settings['options']['data_collector'];
    $actionBarClass = $settings['options']['action_bar'];
    $generatorClass = $settings['options']['generator'];
    $cacheClass = $settings['options']['cache'];
    
    $cache = new $cacheClass($settings['cache'], $this->fileName, $this->culture);
    if (!$cache->isContentCached()) {
      $dataCollector = new $dataCollectorClass($settings['options']['category'], $settings['data_collector'], $this->culture);
      $actionBar = new $actionBarClass($settings['action_bar'], $dataCollectorClass);
      $generator = new $generatorClass($settings['generator'], $dataCollector, $actionBar);
      
      $content = $generator->generateMenu();
      $cache->setContent($content);
    }
    
    $content = $cache->getContent();
    return $content;
  }
  
  /**
   * Zwraca zainicjowana klase dataCollectora
   * Uzywane np. w artSitemapTask
   * @return \dataCollector 
   */
  public function getDataCollector() {
    $settings = sfYaml::load( $this->filePath );
    $settings = $this->initSettings($settings);
    
    $dataCollectorClass = $settings['options']['data_collector'];
    
    return new $dataCollectorClass($settings['options']['category'], $settings['data_collector'], $this->culture);
  }
  
  /**
   * Zwraca zainicjowana klase generatora
   * Uzywane np. w artSitemapTask
   * @return \generator 
   */
  public function getGenerator($dataCollector) {
    $settings = sfYaml::load( $this->filePath );
    $settings = $this->initSettings($settings);
    
    $generatorClass = $settings['options']['generator'];
    $actionBarClass = $settings['options']['action_bar'];
    $actionBar = new $actionBarClass($settings['action_bar'], $dataCollectorClass);
    
    return new $generatorClass($settings['generator'], $dataCollector, $actionBar);
  }
  
  public static function getMenu($fileName) {
    $className = get_called_class();
    
    $menuGenerator = new $className($fileName);
    return $menuGenerator->generateMenu();
  }
  
  /**
   * Funkcja odpowiada za zwrocenie URL dla elementu na danej pozycji.
   * 
   * @return string
   */
  public function generateHomepageUrl($position) {
    $settings = sfYaml::load( $this->filePath );
    $settings = $this->initSettings($settings);
    
    $dataCollectorClass = $settings['options']['data_collector'];
    $generatorClass = $settings['options']['generator'];
    // Wiecej niz 1 glebokosci nam nie potrzeba
    $settings['data_collector']['depth'] = 1;

    $dataCollector = new $dataCollectorClass($settings['options']['category'], $settings['data_collector'], $this->culture);
    $generator = new $generatorClass($settings['generator'], $dataCollector, null);

    return $generator->getHomepageUrl($position);
  }
  
  public static function getHomepageUrl($fileName, $position) {
    $className = get_called_class();
    
    $menuGenerator = new $className($fileName);
    return $menuGenerator->generateHomepageUrl($position);
  }
  
}

?>
