<?php
/**
 * Klasa obslugujaca cachowanie menu.
 *

 */
abstract class BaseMenuCache {
  
  protected $filePath = null;
  protected $culture = null;
  protected $settings = array();
  
  /**
   * Konstruktor przyjmuje tablice ustawien, nazwe pliku i jezyk
   * 
   * @param array $settings
   * @param string $fileName
   * @param string $culture 
   */
  public function __construct($settings, $fileName, $culture) {
    $this->initSettings($settings);
    $this->fileName = $fileName;
    $this->culture = $culture;
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow.
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien
   * 
   * @return array
   */
  protected function getDefaultSettings()
  {
    return array(
        'enabled' => true
      );
  }
  
  /**
   * Funkcja zwraca CacheKey wykorzystywany przy pobieraniu partiala z cache.
   * 
   * @return string
   */
  protected function getCacheKey() {
    return  $this->culture . rand(0, 10000000);
  }
  
  /**
   * Funkcja zwraca nazwe partiala, generuje go z nazwy pliku.
   * 
   * @return string
   */
  protected function getPartialName() {
    return substr($this->fileName, 0, strpos($this->fileName, '.')).'_'.$this->culture;
  }

  /**
   * Funkcja zwraca zawartosc partiala jesli istnieje w cache lub FALSE w przeciwnym wypadku.
   * 
   * @return mixed 
   */
  public function getContent() {
    try {
      $content = get_partial('menu/' . $this->getPartialName(), array(), array('sf_cache_key' => $this->getCacheKey() ) );
    } catch (Exception $e) {
      $content = false;
    }
    
    return $content;
  }
  
  /**
   * Funkcja sprawdza czy istnieje dany partial w cache. Jesli ustawilismy
   * 'enabled' => false, zawsze zwroci false.
   * 
   * @return boolean 
   */
  public function isContentCached() {
    if ($this->settings['enabled']) {
      $filename = $this->getPartialCacheDir() . '/_' . $this->getPartialName() . '.php';
      if (is_readable($filename)) {
        return true;
      }
    }
    
    return false;
  }
  
  /**
   * Funkcja zapisuje $content do partiala
   * 
   * @param string $content
   * @throws sfException
   */
  public function setContent($content) {
    $filename = $this->getPartialCacheDir() . '/_' . $this->getPartialName() . '.php';
    $file = fopen($filename, 'w+');
    fwrite($file, $content);
    fclose($file);
    chmod($filename, 0777);
    
    if (!is_readable($filename)) {
      throw new sfException('Plik "'.$filename.'" nie został stworzony lub brakuje praw do odczytu');
    }
  }
  
  /**
   * Funkcja tworzy i zwraca katalog przetrzymywania cache dla menu.
   * 
   * @return string
   * @throws sfException
   */
  protected function getPartialCacheDir()
  {
    $partialCacheDir = sfConfig::get('sf_module_cache_dir').'/autoMenu/templates';
    
    if (!is_dir($partialCacheDir)) {
      mkdir($partialCacheDir, 0777, true);
    }
    
    if (!is_dir($partialCacheDir)) {
      throw new sfException('Katalog "'.$partialCacheDir.'" nie został poprawnie stworzony.');
    }

    return $partialCacheDir;
  }
}

?>
