<?php
/**
 * Klasa odpowiada za pobranie danych do menu dla danej kategorii (reprezentowanej
 * przez unique_name) i jezyka. Zwracane sa tablice zamiast obiektow, aby zajmowaly
 * mniej pamieci. Dane w klasie sa dwoch "typow": artykul (TYPE_ARTICLE) i kategoria
 * (TYPE_CATEGORY). Poczatkowo tworzona jest tablica hierarchii, a nastepnie na jej
 * podstawie tworzone sa kolejno elementy dostepne przez funkcje getNext()
 *

 */
abstract class BaseMenuDataCollector {
  
  protected $category = null;
  protected $culture = null;
  protected $settings = array();
  protected $list = array();
  
  const TYPE_ARTICLE = 0;
  const TYPE_CATEGORY = 1;
  
  /**
   * Konstruktor przyjmuje kategorie w formie unique_name, tablice ustawien i jezyk
   * 
   * @param string $category
   * @param array $settings
   * @param string $culture 
   */
  public function __construct($category, $settings, $culture) {
    $this->category = $category;
    $this->culture = $culture;
    $this->initSettings($settings);
    $this->prepareData();
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * ustawien.
   * 
   * @param array $settings
   * @return array 
   */
  public function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien
   * 
   * @return array
   */
  public function getDefaultSettings()
  {
    return array(
        'with_articles' => true,
        'with_categories' => true,
        'depth' => 4,
        'categories' => array(),
        'articles' => array()
      );
  }
  
  /**
   * Funkcja przygotowuje dane na podstawie kategorii, jezyka i ustawien. Wywoluje
   * zestaw funkcji odpowiedzialny za przygotowanie danych
   * 
   * @throws sfException 
   */
  protected function prepareData() {
    // Pobranie danych na 1 poziomie
    $records = Doctrine::getTable('ArticleCategory')->getByUniqueName( $this->category, $this->settings['with_articles'], $this->settings['with_categories'] );
    // Zabezpieczenie przed nie istniejaca lub pusta kategoria
    if (empty($records)) {
      throw new sfException('Podana kategoria "'.$this->category.'" nie istnieje lub jest pusta.');
    }
    // Inicjalizacja tablic artykulow i kategorii zawierajacych ID, tablicy hierarchii i listy
    $articles = $categories = $hierarchy = $list = array();
    // Przetwarzanie rekordow danych, zbieranie ID artykulow i kategorii, tworzenie hierarchii
    $this->parseRecords($records, $categories, $articles, $hierarchy, 1);
    // Pobranie i zapisanie danych o artykulach i kategoriach
    $this->initData($categories, $articles);
    // Zamiana hierarchi na liste z dodaniem wlasnych wartosci
    $this->parseHierarchy($hierarchy, $list, $records[0]['category_id'], 1);
    // Inicjalizacja listy wyjsciowych elementow
    $this->initList($list);
  }
  
  /**
   * Inicjuje dane dotyczace elementow. Rozdziela tablice $this->data na dwie tablice
   * zlozone z elementow danego typu. Przyjmuje dwie tablice (kategorii i artykulow)
   * zawierajace ID danych do pobrania
   * 
   * @param array $categories
   * @param array $articles 
   */
  protected function initData($categories, $articles) {
    // Pobranie danych
    $categoriesData = $this->getCategoriesData($categories);
    // Przygotowanie tablicy
    $categoriesData = $this->prepareDataArray($categoriesData);
    
    // Pobranie danych
    $articlesData = $this->getArticlesData($articles);
    // Przygotowanie tablicy
    $articlesData = $this->prepareDataArray($articlesData);

    $this->data[self::TYPE_CATEGORY] = $categoriesData;
    $this->data[self::TYPE_ARTICLE] = $articlesData;
  }
  
  /**
   * Przygotowuje tablice danych. Przygotowana tablica zawiera klucze reprezentujace
   * ID obiektu. Usuwa hierarchie jezykow (bo mamy tylko jeden)
   * 
   * @param array $array
   * @return array
   */
  protected function prepareDataArray($array) {
    $preparedArray = array();
    foreach ($array as $value) {
      $preparedArray[$value['id']] = $value;
      // Zeby wygodnie bylo korzystac z tablicy, pozbywamy sie hierarchi jezykow
      unset($preparedArray[$value['id']]['Translation']);
      $preparedArray[$value['id']]['Translation'] = $value['Translation'][$this->culture];
    }
    
    return $preparedArray;
  }
  
  /**
   * Funkcja zwraca dodatkowe dane z tablicy ustawien dla podanego unique_name i typu.
   * Jesli dane nie istnieja, zwraca pusta tablice
   * 
   * @param string $uniqueName
   * @param int $type 
   * @return array
   */
  protected function getCustomData($uniqueName, $type) {
    if ( $type == self::TYPE_CATEGORY && isset($this->settings['categories'][$uniqueName]) ) {
      return $this->settings['categories'][$uniqueName];
    } elseif ( $type == self::TYPE_ARTICLE && isset($this->settings['articles'][$uniqueName]) ) {
      return $this->settings['articles'][$uniqueName];
    } else {
      return array();
    }
  }
  
  /**
   * Funkcja wstawia dodatkowe dane do elementu. Wykorzystuje funkcje getCustomData()
   * do pobrania dodatkowych danych
   * 
   * @param int $id
   * @param int $type
   * @param array $element
   */
  protected function addCustomData($id, $type, &$element) {
    $uniqueName = $this->data[$type][$id]['unique_name'];
    $element = array_merge($element, $this->getCustomData($uniqueName, $type) );
  }
  
  /**
   * Inicjuje liste elementow. Resetuje wskaznik listy
   * 
   * @param array $list 
   */
  protected function initList($list) {
    $this->list = $list;
    reset($this->list);
  }
  
  /**
   * Funkcja rekurencyjna. Na podstawie hierarchi uzupelniana jest lista elementow.
   * Tworzony element uzupelniany jest o potrzebne informacje:
   * $element = array(
   * 'id' => ID elementu,
   * 'type' => typ elementu,
   * 'category_id' => ID kategorii do ktorej nalezy,
   * 'position' => pozycja w menu (kazdy nastepny poziom menu restartuje numer pozycji),
   * 'depth' => glebokosc elementu,
   * 'first' => czy element jest pierwszy w menu,
   * 'last' => czy element jest pierwszy w menu,
   * 'is_reference' => czy element jest referencja,
   * 'submenu' => czy generowac submenu,
   * 'url_raw' => cy link ma byc generowany przez routing symfonii
   * )
   * 
   * @param array $hierarchy
   * @param array $list
   * @param int $categoryId
   * @param int $depth
   * @param boolean $isReference 
   */
  protected function parseHierarchy($hierarchy, &$list, $categoryId, $depth, $isReference = false) { 
    $isReferenceLocal = false;
    $position = 1;
    $first = count($list);
    foreach ($hierarchy as $key => $value) {
      $keyArr = explode('_', $key);
      $element = array(
          'id' => $keyArr[0],
          'type' => $keyArr[1],
          'article_category_id' => $keyArr[2],
          'category_id' => $categoryId,
          'position' => $position,
          'depth' => $depth,
          'first' => false,
          'last' => false,
          'is_reference' => $isReference,
          'submenu' => true,
          'active_equal' => false,
          'url_raw' => false,
      );
      // Dodanie dodatkowych danych do elementu
      $this->addCustomData($keyArr[0], $keyArr[1], $element);
      // Obsluga referencji. Jesli element ma zdefiniowane is_reference = true,
      // kazdy element pod nim bedzie mial ustawione is_reference = true
      if ($element['is_reference']) {
        $isReferenceLocal = true;
      }
      
      $list[] = $element;
      if (is_array($value) && !empty($value) && $element['submenu']) {
        // Jesli wartosc jest tablica i nie jest pusta to zejdz nizej
        $this->parseHierarchy($value, $list, $keyArr[0], $depth + 1, $isReferenceLocal);
      }
      
      $isReferenceLocal = false;
      $position++;
    }
    
    // Przypisanie pierwszy / ostatni
    $list[$first]['first'] = true;
    $list[count($list) - 1]['last'] = true;
    
  }
  
  /**
   * Funkcja rekurencyjna. Tworzy hierarchie na podstawie przekazanych rekordow.
   * Rozszerza tablice ID kategorii i artykulow o nowo przekazane ID elementow.
   * Posiada ograniczenie glebokosci parsowania do $this->settings['depth']
   * 
   * @param array $records
   * @param array $categories
   * @param array $articles
   * @param array $hierarchy
   * @param int $depth
   */
  protected function parseRecords($records, &$categories, &$articles, &$hierarchy, $depth) {
    if ($depth > $this->settings['depth'] ) {
      // Jesli przekroczylismy glebokosc to konczymy
      return;
    }
    $count = count($categories);
    
    foreach ($records as $record) {
      if (isset($record['article_id'])) {
        $articles[] = $record['article_id'];
        $toKey = $record['category_id'].'_'.self::TYPE_CATEGORY;
        $addKey = $record['article_id'].'_'.self::TYPE_ARTICLE.'_'.$record['id'];
        $this->addToHierarchy($hierarchy, $toKey , $addKey, NULL, $depth);
      } elseif (isset($record['subcategory_id'])) {
        $categories[] = $record['subcategory_id'];
        $toKey = $record['category_id'].'_'.self::TYPE_CATEGORY;
        $addKey = $record['subcategory_id'].'_'.self::TYPE_CATEGORY.'_'.$record['id'];
        $this->addToHierarchy($hierarchy, $toKey , $addKey, array(), $depth);
      }
    }   
    // Usuniecie powtarzajacych sie wartosci
    $articles = $this->removeDuplicates($articles);
    $categories = $this->removeDuplicates($categories);
    
    if (count($categories) > $count) {
      // Mamy wiecej kategorii niz bylo, pobieramy nowe kategorie i parsujemy rekordy
      $categoriesAfterParse = array_slice($categories, $count);
      $data = Doctrine::getTable('ArticleCategory')->getByCategoryIds( $categoriesAfterParse, $this->settings['with_articles'], $this->settings['with_categories'] );
      $this->parseRecords($data, $categories, $articles, $hierarchy, $depth + 1);
    }
  }
  
  /**
   * Funkcja usuwa z tablicy powtarzajace sie wartosci.
   * 
   * @param array $array
   * @return array 
   */
  protected function removeDuplicates($array) {
    return array_unique($array);
  }
  
  /**
   * Dodaje wpis do hierarchii. Wpis zostanie dodany do klucza $toKey, bedzie
   * mial klucz $addKey i wartosc $addValue. Funkcja bedzie szukala maksymalnie do
   * glebokosci $maxDepth
   * 
   * @param array $hierarchy
   * @param string $toKey
   * @param string $addKey
   * @param string $addValue
   * @param int $maxDepth
   * @param int $depth
   */
  protected function addToHierarchy(&$hierarchy, $toKey, $addKey, $addValue, $maxDepth, $depth = 1) {
    if ($maxDepth == 1) {
      // Pierwszy poziom obslugujemy prosto
      $hierarchy[$addKey] = $addValue;
      return true;
    } else {
      // Kolejne musimy juz wyszukiwac klucze
      foreach ($hierarchy as $key => $value) {
        // Nie znamy dokladnego klucza poniewaz nie mamy AAC ID
        $keyToCompare = substr($key, 0, strrpos($key, "_"));
        if ($toKey == $keyToCompare) {
          $hierarchy[$key][$addKey] = $addValue;
          return;
        }
        // Sprawdzamy czy wogule mozemy pojsc nizej i czy wartosc jest nie pusta tablica
        if ($maxDepth >= $depth && is_array($value) && !empty($value))  {
          $this->addToHierarchy($hierarchy[$key], $toKey, $addKey, $addValue, $maxDepth, $depth + 1);
        }
      }
    }
  }

  /**
   * Zwraca dane nalezace do przekazanego elementu
   * 
   * @param array $element
   * @return array
   * @throws sfException 
   */
  public function getData( $element ) {
    if (!isset($element['id']) || !isset($element['type'])) {
      // Pobierasz cos co nie ma zdefiniowanego typu i id
      throw new sfException('Dane które probujesz pobrać nie mają zdefiniowanego typu lub ID.');
    }
    $type = $element['type'];
    $id = $element['id'];
    if (!isset($this->data[$type]) || !isset($this->data[$type][$id])) {
      // Starasz sie pobrac cos co nie istnieje
      throw new sfException('Dane które próbujesz pobrać, nie istnieją. Próbojesz pobrac dane o typie "'.$type.'" i ID "'.$id.'"');
    } else {
      return $this->data[$type][$id]; 
    }
  }
  
  /**
   * Zwraca dane dotyczace kategorii na podstawie przekazanej tablicy ID kategorii 
   * i $this->culture
   * 
   * @param array $categories
   * @return array 
   */
  protected function getCategoriesData( $categories ) {
    return Doctrine::getTable('Category')->getCategoriesByIds($categories, $this->culture);
  }
  
  /**
   * Zwraca dane dotyczace artykulow na podstawie przekazanej tablicy ID artykulow 
   * i $this->culture
   * 
   * @param array $categories
   * @return array 
   */
  protected function getArticlesData( $articles ) {
    return Doctrine::getTable('Article')->getArticlesByIds($articles, $this->culture);
  }
  
  /**
   * Zwraca kolejny element z listy
   * 
   * @return array 
   */
  public function getNext() {
    list ($key, $value) = each ($this->list);
    return $value;
  }
  
  /**
   * Zwraca cala liste elementow
   * @return array
   */
  public function getAll() {
    return $this->list;
  }
  
  /**
   * Zwraca unique_name nadrzednej kategorii menu
   * @return string
   */
  public function getMainCategory() {
    return $this->category;
  }
  
  /**
   * Funkcja sprawdza czy przekazany element jest kategoria
   * @param array $element
   * @return boolean
   */
  public static function isCategory($element) {
    return ($element['type'] == self::TYPE_CATEGORY);
  }
  
  /**
   * Funkcja sprawdza czy przekazany element jest artykulem
   * @param array $element
   * @return boolean
   */
  public static function isArticle($element) {
    return ($element['type'] == self::TYPE_ARTICLE);
  }
  
  /**
   * Funkcja sprawdza czy przekazany element nalezy do referencji
   * @param array $element
   * @return boolean
   */
  public static function isReference($element) {
    return ($element['is_reference']);
  }
}

?>
