<?php
/**
 * Klasa odpowiada za wygenerowanie kodu HTML i PHP zawierajacego pelne menu.
 *

 */
abstract class BaseMenuGenerator {
  
  protected $settings = array();
  protected $dataCollector = null;
  protected $dataCollectorClass = null;
  protected $actionBar = null;
  
  /**
   * Konstruktor przyjmuje tablice ustawien oraz zainicjowane klasy kolektora
   * i actionBar'a
   * 
   * @param array $settings
   * @param object $dataCollector
   * @param object $actionBar 
   */
  public function __construct($settings, $dataCollector, $actionBar) {
    $this->initSettings($settings);
    $this->dataCollector = $dataCollector;
    $this->dataCollectorClass = get_class($dataCollector);
    
    $this->actionBar = $actionBar;
  }
  
  /**
   * Funkcja inicjuje ustawienia. Wykorzystuje getDefaultSettings() i tablice
   * parametrow $settings
   * 
   * @param array $settings
   */
  protected function initSettings($settings) {
    $this->settings = artArray::array_merge_recursive_replace( $this->getDefaultSettings() , $settings );
  }
  
  /**
   * Funkcja zwraca wartosci default dla ustawien
   * 
   * @return array
   */
  protected function getDefaultSettings()
  {
    return array(
        'menu_class' => 'menu',
        'element_class' => 'element',
        'active_class' => 'active',
        'link_class' => 'link',
        'first_class' => 'first',
        'last_class' => 'last',
        'submenu_class' => 'submenu hide',
        'ul_class' => 'menu-ul',
        'li_class' => 'menu-li',
//        'article_url' => 'article_show',
//        'category_url' => 'article_list',
        'empty_url' => 'javascript:void(0);',
        'onclick' => false,
        'separator' => false,
        'separator_char' => false,
        'artbox' => 'false',
        'artbox_class' => 'artbox',
        // Do MAB glownej kategorii
        'main_category_options_class' => 'fr icon-cog-large pointer',
      );
  }
  
  /**
   * Funkcja inicjuje element. Jesli element ma juz ustawione dane, nie zostana nadpisane.
   * 
   * @param array $element
   * @return array 
   */
  protected function initElement($element, $elementData) {
    $this->initElementUrl($element, $elementData);
    $this->initElementText($element, $elementData);
    $this->initElementClass($element, $elementData);
    
    return $element;
  }
  
  /**
   * Funkcja generuje element na na podstawie przekazanego elementu. Zwraca 
   * kod HTML
   * 
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  protected function generateElement($element, $elementData) {
    $elementContent = '';
    if ($this->settings['artbox']) {
      $elementContent .= '<span class="'.$this->settings['artbox_class'].'-left"></span><span class="'.$this->settings['artbox_class'].'-center">';
    }
    
    $elementContent .= '<span class="'.$this->settings['element_class'].'">';
    
    $elementContent .= '<a href="'.$element['url'].'" class="'.$element['class'].'"';
    
    if (isset($element['onclick']) && $element['onclick']) {
      $elementContent .= ' onclick="'.$element['onclick'].'"';
    }
    
    $elementContent .= '>'.$element['text'].'</a>';
    
    $elementContent .= $this->actionBar->generateActionBar($element, $elementData);
    
    $elementContent .= '</span>';
    
    if ($this->settings['artbox']) {
      $elementContent .= '</span><span class="'.$this->settings['artbox_class'].'-right"></span>';
    }
    
    return $elementContent;
  }
  
  /**
   * Ustawia url dla podanego elementu
   * 
   * @param array $element
   * @param array $elementData
   * @param boolean $applyUrlFor Czy na url ma zostac uzyta funkcja url_for()
   */
  public function initElementUrl(&$element, $elementData, $applyUrlFor = true) {
    if (isset($element['url']) && $element['url'] === false) {
      $element['url'] = $this->settings['empty_url'];
    } else {
      
      if (!isset($element['url'])) {
        // Jesli nie jest juz ustawiony
        $element['url'] = $this->getElementUrl($element, $elementData);
      }
      
      if ($element['url_raw'] === false) {
        $element['url'] = '@'.$element['url'];
        if ($applyUrlFor) {
          $element['url'] = url_for($element['url']);
        }
      }
    }
  }
  
  /**
   * Zwraca URL dla elementu w zaleznosci od jego typu
   * 
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  public function getElementUrl($element, $elementData) {
//    if ( $this->isArticle($element) ) {
      return $elementData->getUrl();
//    } else {
//      return $this->settings['category_url'].'?category='.$elementData['unique_name'];
//    } 
  }
  
  /**
   * Funkcja inicjuje tekst dla podanego elementu. W przypadku ustawionego wczesniej
   * tekstu stosuje na nim funkcje do tlumaczenia __(tekst)
   * 
   * @param array $element
   * @param array $elementData
   */
  protected function initElementText(&$element, $elementData) {
    if (!isset($element['text'])) {
      // Jesli nie jest juz ustawiony
      if ( $this->isArticle($element) ) {
        $element['text'] = $elementData['Translation']['title'];
      } else {
        $element['text'] = $elementData['Translation']['name'];
      } 
    } else {
      // Dodajemy translacje jesli wczesniej byl ustawiony
      $element['text'] = __($element['text']);
    }
  }
  
  /**
   * Funkcja inicjuje klasy na podstawie podanego elementu. Ustawia rowniez kod
   * PHP odpowiedzialny za sprawdzenie czy element jest aktywny
   * 
   * @param array $element
   * @param array $elementData
   */
  protected function initElementClass(&$element, $elementData) {
    $class = array($this->settings['link_class']);
    if ($element['first']) {
      $class[] = $this->settings['first_class'];
    }
    
    if ($element['last']) {
      $class[] = $this->settings['last_class'];
    }
    
    // Dodanie funkcji ustawiajacej klase is_active
    $class[] = 
    
    $element['class'] = implode(' ', $class);
  }
  
  /**
   * Funkcja generuje string odpowiedzialny za sprawdzenie czy element jest aktywny
   * 
   * @param array $element
   * @param array $elementData
   * @return string 
   */
  protected function getIsActiveString($element, $elementData) {
    return '<?php echo ('.get_class($this).'::isActive('.$this->generateStringArray($element).','.$this->generateStringArray($elementData).') ? \' '.$this->settings['active_class'].'\' : \'\'); ?>';
  }
  
  /**
   * Funkcja generuje string jako tablice w kodzie PHP z podanej tablicy.
   * np. array(
   * 'key1' => 'val1',
   * 'key2' => 'val2',
   * )
   * 
   * @param type $array
   * @return string 
   */
  protected function generateStringArray($array) {
    $string = 'array(';
    foreach ($array as $key => $value) {
      if (!is_array($value)) {
        $string .= '\''.$key.'\' => \''.addslashes($value).'\',';
      }
    }
    $string .= ')';
    
    return $string;
  }

  /**
   * Funkcja sprawdza czy dany element jest aktualnie aktywny. Jesli tak zostaje
   * dodana klasa $this->settings['is_active']
   * 
   * @param array $element
   * @param array $elementData
   * @return boolean 
   */
  public static function isActive($element, $elementData) {
    if ($element['active_equal']) {
      if ($_SERVER["REQUEST_URI"] == $element['url']) {
        return true;
      } else {
        return false;
      }
    }
    
    if (!(strpos($_SERVER["REQUEST_URI"], $element['url']) === false)) {
      return true;
    }
    
    return false;
  }
  
  /**
   * Funkcja zwraca kod HTML i PHP dla calego menu
   * 
   * @return string
   */
  public function generateMenu() {
    $firstElement = $this->dataCollector->getNext();
    
    $menuContent = $this->generateTopMenu($firstElement);
    // Ustawianie konfiguracji strony glownej
    $lastDepth = 1;

    for ($element = $firstElement ; $element != false; $element = $this->dataCollector->getNext() ) {
      if ($this->depthUp($element, $lastDepth)) {
        $menuContent .= $this->generateBottomSubmenu($lastDepth - $element['depth']);
      }
      
      if ($this->depthDown($element, $lastDepth)) {
        $menuContent .= $this->generateTopSubmenu();
      }

      // Generowanie separatora
      if (!$element['first']) {
        if ($this->settings['separator']) {
          $menuContent .= $this->generateSeparator();
        }
      }

      if (!$element['first']) {
        $menuContent .= '</li>';
      }
      
      $elementData = $this->dataCollector->getData($element);
      $element = $this->initElement($element, $elementData);

      $menuContent .= '<li class="'.$this->settings['li_class'].$this->getIsActiveString($element, $elementData).'">';
      
      $menuContent .= $this->generateElement(
              $element,
              $elementData
              );
      
      $lastDepth = $element['depth'];
    }
    
    $menuContent .= $this->generateBottomSubmenu($lastDepth - 1);
    $menuContent .= $this->generateBottomMenu();
    
    return $menuContent;
  }
  
  /**
   * Funkcja zwraca URL elementu na danej pozycji
   * 
   * @param int $position Pozycja elementu (od 1)
   * @return string
   */
  public function getHomepageUrl($position) {
    $i = 1;
    for ($element = $this->dataCollector->getNext() ; $element != false; $element = $this->dataCollector->getNext() ) {
      if ($position == $i) {
        $elementData = $this->dataCollector->getData($element);
        return $this->getElementUrl($element, $elementData);
      }
      
      $i++;
    }
    
    throw new sfException('Wybrana pozycja '.$position.' nie istnieje w menu.');
  }
  
  /**
   * Zwraca HTML zawierajacy separator, jesli ustawienia klasy na to pozwalaja
   * 
   * @return string 
   */
  protected function generateSeparator() {
    if ($this->settings['separator_char']) {
      return '<span class="separator fl">'.$this->settings['separator_char'].'</span>';
    } else {
      return '<span class="separator fl"></span>';
    }
  }
  
  /**
   * Tworzy gore menu pierwszego poziomu
   * 
   * @return string 
   */
  protected function generateTopMenu($element) {
    $content = '<div class="'.$this->settings['menu_class'].'">';
    // Przygotowanie elementu
    $element['options_class'] = $this->settings['main_category_options_class'];
    $element['id'] = $element['category_id'];
    $dataCollectorClass = $this->dataCollectorClass;
    $element['type'] = $dataCollectorClass::TYPE_CATEGORY;
    $element['category_add'] = true;
    $element['article_add'] = true;
    $element['category_edit'] = false;
    $element['category_delete'] = false;
    $content .= ActionBarHelper::getActionBar($element, $this->isArticle($element), false, true);
    $content .= '<ul class="'.$this->settings['ul_class'].'">';
    return $content;
  }
  
  /**
   * Tworzy dol menu pierwszego poziomu
   * 
   * @return string 
   */
  protected function generateBottomMenu() {
    return '</li></ul></div>';
  }
  
  /**
   * Tworzy gore dla menu drugiego i nizszych poziomow
   * 
   * @return string 
   */
  protected function generateTopSubmenu() {
    return '<div class="'.$this->settings['submenu_class'].'"><ul class="'.$this->settings['ul_class'].'">';
  }
  
  /**
   * Tworzy dol dla menu drugiego i nizszych poziomow. Parametr $depthDiff mowi
   * o roznicy glebokosci i wplywa na ilosc znacznikow zamykajacych
   * jakie beda wygenerowane
   * 
   * @param int $depthDiff
   * @return string 
   */
  protected function generateBottomSubmenu($depthDiff) {
    $content = '';
    for ($i = 0; $i < $depthDiff; $i++) {
      $content = '</li></ul></div>'.$content;
    }
    return $content;
  }
  
  /**
   * Sprawdza czy weszlismy wyzej z poziomem
   * 
   * @param array $element
   * @param int $lastDepth
   * @return boolean 
   */
  protected function depthUp($element, $lastDepth) {
    return ($element['depth'] < $lastDepth);
  }
  
  /**
   * Zwraca ustawienia dla Generatora
   * @return array
   */
  public function getSettings() {
    return $this->settings;
  }
  
  /**
   * Sprawdza czy zeszlismy nizej z poziomem
   * 
   * @param array $element
   * @param int $lastDepth
   * @return boolean 
   */
  protected function depthDown($element, $lastDepth) {
    return ($element['depth'] > $lastDepth);
  }
  
  /**
   * Sprawdza czy podany element jest artykulem.
   * (zrobione dla wygody)
   * @param array $element
   * @return boolean 
   */
  protected function isArticle($element) {
    $dataCollectorClass = $this->dataCollectorClass;
    return $dataCollectorClass::isArticle($element);
  }
  
  /**
   * Sprawdza czy podany element jest kategoria
   * (zrobione dla wygody)
   * @param array $element
   * @return boolean 
   */
  protected function isCategory($element) {
    $dataCollectorClass = $this->dataCollectorClass;
    return $dataCollectorClass::isCategory($element);
  }
}

?>
