<?php

/**
 * Frontend Management actions
 *
 * @subpackage frontend_management

 */
class PluginFrontendManagementActions extends artCmsPluginActions {

  /**
   * Przeprowadza walidacje parametru task.
   * 
   * @param sfWebRequest $request
   * @throws sfException
   */
  protected function validateManageFormRequest(sfWebRequest $request) {
    
    $allowedTasks = array('add', 'edit', 'delete', 'order', 'list_order', 'list', 'table_method');
    
    if (!in_array($request->getParameter('task'), $allowedTasks)) {
      throw new sfException('Podany task "' . $request->getParameter('task') . '" nie jest obsługiwany.');
    }

    $form = $request->getParameter('form');
    
    if ($form == 'sfGuardChangeSigninPasswordForm' || $form == sfConfig::get('app_sf_guard_plugin_signin_form', 'sfGuardFormSignin')) {
      // Musimy ponownie sprawdzic ID bo ktos mogl sobie w HTML zmienic
      if ($request->getParameter('id') != $this->getUser()->getId()) {
        throw new sfException('Niepoprawny ID obiektu.');
      }
    }
  }

  /**
   * Funkcja sprawdza czy user ma prawa podane w tablicy $parmissions
   * @param array $permissions
   * @return boolean 
   */
  protected function checkPermissions($permissions) {
    return $this->getUser()->hasPermission($permissions, false);
  }

  /**
   * Przeprowadza walidacje $request i przygotowuje parametry potrzebne
   * do obsluzenia formularzy
   * 
   * @param sfWebRequest $request
   */
  protected function preManageForm(sfWebRequest $request) {
    $this->validateManageFormRequest($request);
    $this->task = $request->getParameter('task');
    $this->formClass = $request->getParameter('form');
    $this->requestParameters = $this->getRequestParameters($request);
    $this->formUrl = $this->getFormUrl($this->requestParameters);
    $this->getContext()->getConfiguration()->loadHelpers('I18N');
  }

  /**
   * Zwraca parametry z GET oraz routingu w postaci tablicy
   * @param sfWebRequest $request
   * @return array
   */
  protected function getRequestParameters(sfWebRequest $request) {
    $requestParameters = $request->getGetParameters();
    $requestParameters['form'] = $request->getParameter('form');
    $requestParameters['task'] = $request->getParameter('task');

    return $requestParameters;
  }

  /**
   * Przygotowuje regule routingu z parametrami na podstawie tablicy parametrow
   * @param array $params
   * @return string
   */
  protected function getFormUrl(array $params) {
    $parameters = array();

    foreach ($params as $key => $value) {
      if ($value != '') {
        $parameters[] = $key . '=' . $value;
      }
    }

    $url = '@manage_form_' . $params['task'] . (!empty($parameters) ? '?' . implode('&', $parameters) : '');
    return $url;
  }

  public function executeManageFormAdd(sfWebRequest $request) {
    
    $this->preManageForm($request);
    $this->form = new $this->formClass(null, $this->requestParameters);

    if ($this->form->bindFromRequestAndSave($request)) {
      
      $this->getUser()->setNotice($this->getNotice('success', $this->form, $this->task));
      $this->fmClearCache();

      if (method_exists($this->form, 'getObject') && method_exists($this->form->getObject(), 'getUrl')) {

        if ($this->form->getObject()->getUrl()) {
          $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($this->form->getObject())));
          return $this->renderText($this->form->getObject()->getUrl());
        }
      }

      return $this->renderText('Success');
    }

    $this->postManageForm();
  }

  public function executeManageFormEdit(sfWebRequest $request) {
    $this->preManageForm($request);

    // Tworzymy tymczasowy form aby pobrac ModelName
    $form = new $this->formClass(null, $this->requestParameters);
    $object = Doctrine::getTable($form->getModelName())->findOneById($request->getParameter('id'));
    if (!$object) {
            $this->getUser()->setFlash('error', _('Obiekt który chcesz zmienić nie istnieje.'));
//      $this->redirect('@homepage');

      $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($this->form->getObject())));
      return $this->renderText($this->getContext()->getRouting()->generate('homepage'));
    }

    $this->form = new $this->formClass($object, $this->requestParameters);

    if ($this->form->bindFromRequestAndSave($request)) {
      $this->getUser()->setNotice($this->getNotice('success', $this->form, $this->task));
      $this->fmClearCache();

      if (method_exists($this->form, 'getObject') && method_exists($this->form->getObject(), 'getUrl')) {
        if ($this->form->getObject()->getUrl()) {
          $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($this->form->getObject())));
          return $this->renderText($this->form->getObject()->getUrl());
        }
      }
      
      return $this->renderText('Success');
    }

    $this->postManageForm();
  }

  public function executeManageFormDelete(sfWebRequest $request) {
    $this->preManageForm($request);

    // Tworzymy tymczasowy form aby pobrac ModelName
    $form = new $this->formClass(null, $this->requestParameters);
    $object = Doctrine::getTable($form->getModelName())->findOneById($request->getParameter('id'));
    if (!$object) {
            $this->getUser()->setFlash('error', _('Obiekt który chcesz zmienić nie istnieje.'));
      $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($this->form->getObject())));
      return $this->renderText($this->getContext()->getRouting()->generate('homepage'));
    }

    $this->form = new $this->formClass($object, $this->requestParameters);
    // Wylaczamy pola (poza ID) bo wyswietlamy w widoku tylko prosbe o potwierdzenie usuniecia
    $this->form->useFields(array('id'));

    if ($this->form->bindFromRequestAndSave($request)) {
      $this->form->getObject()->delete();
      $this->getUser()->setNotice($this->getNotice('success', $this->form, $this->task));
      $this->fmClearCache();
      return $this->renderText('Success');
    }

    $this->postManageForm();
  }

  public function executeManageFormOrder(sfWebRequest $request) {
    $this->preManageForm($request);

    // Tworzymy tymczasowy form aby pobrac ModelName
    $form = new $this->formClass(null, $this->requestParameters);
    $object = Doctrine::getTable($form->getModelName())->findOneById($request->getParameter('id'));
    if (!$object) {
            $this->getUser()->setFlash('error', _('Obiekt który chcesz zmienić nie istnieje.'));
      $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($this->form->getObject())));
      return $this->renderText($this->getContext()->getRouting()->generate('homepage'));
    }

    $this->form = new $this->formClass($object, $this->requestParameters);
    // Wylaczamy pola (poza ID) bo wyswietlamy w widoku tylko prosbe o potwierdzenie zmiany kolejnosci
    $this->form->useFields(array('id'));

    if ($this->form->bindFromRequestAndSave($request)) {
      $this->getUser()->setNotice($this->getNotice('success', $this->form, $this->task));
      $this->fmClearCache();
      return $this->renderText('Success');
    }

    $this->postManageForm();
  }

  public function executeManageFormListOrder(sfWebRequest $request) {
    // Pamietac ze tu nie ma formularza, a partial pobierany jest z parametru
    $this->preManageForm($request);

    $tableMethod = $this->requestParameters['table_method'];
    $this->model = $this->requestParameters['model'];
    $this->list = Doctrine::getTable($this->model)->$tableMethod($this->requestParameters);

    $orderedList = $request->getParameter('order', array());
    if (count($orderedList) > 0) {
      // Mamy przekazana liste posortowana wiec powinismy ja zapisac
      $order = 1;
      foreach ($orderedList as $oldOrder => $objectId) {
        $object = Doctrine::getTable($this->model)->findOneById($objectId);
        $object->setOrder($order++);
        $object->save();
      }

      $this->getUser()->setNotice($this->getNotice('success', $this->model, $this->task));
      $this->fmClearCache();
      return $this->renderText('Success');
    }

    $this->postManageForm();
    $this->icon='megaicon1-ionicons-86';
  }

  public function executeManageFormList(sfWebRequest $request) {
    // Pamietac ze tu nie ma formularza, a partial pobierany jest z parametru
    $this->preManageForm($request);

    $tableMethod = $this->requestParameters['table_method'];
    $this->model = $this->requestParameters['model'];

    $this->pager = new sfDoctrinePager(
        $this->model, $this->requestParameters['max_items']
    );
    $this->pager->setQuery(Doctrine::getTable($this->model)->$tableMethod($this->requestParameters));
    $this->pager->setPage($this->requestParameters['page']);
    $this->pager->init();

    $this->postManageForm();
    $this->icon='megaicon1-settings';
  }

  public function executeManageFormTableMethod(sfWebRequest $request) {
    // Pamietac ze tu nie ma formularza, a partial pobierany jest z parametru
    $this->preManageForm($request);

    $tableMethod = $this->requestParameters['table_method'];
    $model = $this->requestParameters['model'];
    // Formularz tylko po to aby sprawdzony zostal _csrf
    $this->form = new PluginFmValidationForm();

    if ($this->form->bindFromRequestAndSave($request)) {
      Doctrine::getTable($model)->$tableMethod($this->requestParameters);
      $this->getUser()->setNotice($this->getNotice('success', $model, $this->task));
      $this->fmClearCache();
      return $this->renderText('Success');
    }

    $this->postManageForm();
  }

  /**
   * Zwraca przetlumaczony ciag znakow jako notice na podstawie parametrow
   * @param string $suffix
   * @param FormObject or string $prefix
   * @param string $task
   * @return string
   */
  protected function getNotice($suffix, $prefix, $task) {
    if (is_object($prefix)) {
      return __t(get_class($prefix) . '_' . $task . '_popup_' . $suffix, array(), 'fm');
    } else {
      return __t($prefix . '_' . $task . '_popup_' . $suffix, array(), 'fm');
    }
  }
  
  /**
   * Ustawia nastepujace zmienne zwiazane z ManageForm
   * $this->partial - Partial do contentu
   * $this->isNarrow - Czy ma byc wyswietlone jako waski formularz
   * $this->icon - Ikonka wykorzystywana w widokach
   */
  public function postManageForm() {
    if (method_exists($this->form, 'getPartial')) {
      $this->partial = $this->form->getPartial();
    } else {
      $this->partial = $this->getRequest()->getParameter('partial', false);
    }
    
    if (method_exists($this->form, 'getIsNarrow')) {
      $this->isNarrow = $this->form->getIsNarrow();
    } else {
      $this->isNarrow = false;
    }
    
    if (method_exists($this->form, 'getIcon')) {
      $this->icon = $this->form->getIcon();
    } else {
      $this->icon = 'megaicon1-questionmark';
    }
  }

  /**
   * Zwraca nazwe partiala lub false pobranego z danego formularza
   * @param FormObject $form
   * @return string
   */
  protected function getPartialForForm($form) {
    
    if (method_exists($form, 'getPartial')) {
      return $form->getPartial();
    } else {
      return false;
    }
  }

  public function setHttpHeaders() {
    $this->getResponse()->setHttpHeader('Pragma', 'no-cache');
    $this->getResponse()->setHttpHeader('Expires', 'Mon, 26 Jul 1997 05:00:00 GMT');
    $this->getResponse()->setHttpHeader('Last-Modified', gmdate("D, d M Y H:i:s") . " GMT");
    $this->getResponse()->setHttpHeader('Cache-Control', 'no-store, no-cache, must-revalidate');
    $this->getResponse()->setHttpHeader('Cache-Control', 'post-check=0, pre-check=0');
  }

  public function executeUploadFile(sfWebRequest $request) {

    $fileSystem = new sfFileSystem();
    $this->setHttpHeaders();

    $id = $request->getParameter('id', false);
    $targetDir = sfConfig::get('sf_upload_dir') . DIRECTORY_SEPARATOR . 'articles' . DIRECTORY_SEPARATOR . $id . DIRECTORY_SEPARATOR . 'original';

    $fileSystem->mkdirs($targetDir, 0755);

    $cleanupTargetDir = false;
    $maxFileAge = 5 * 3600;
    @set_time_limit(5 * 60);

    $chunk = $request->getParameter('chunk');
    $chunks = $request->getParameter('chunks');
    $name = $request->getParameter('name');

    $chunk = isset($chunk) ? intval($chunk) : 0;
    $chunks = isset($chunks) ? intval($chunks) : 0;
    $fileName = isset($name) ? $name : '';

    $fileName = preg_replace('/[^\w\._]+/', '_', $fileName);

    if ($chunks < 2 && file_exists($targetDir . DIRECTORY_SEPARATOR . $fileName)) {
      $ext = strrpos($fileName, '.');
      $fileName_a = substr($fileName, 0, $ext);
      $fileName_b = substr($fileName, $ext);
      $count = 1;

      while (file_exists($targetDir . DIRECTORY_SEPARATOR . $fileName_a . '_' . $count . $fileName_b))
        $count++;

      $fileName = $fileName_a . '_' . $count . $fileName_b;
    }

    $filePath = $targetDir . DIRECTORY_SEPARATOR . $fileName;

    if (!file_exists($targetDir))
      $fileSystem->mkdirs($targetDir);

    if ($cleanupTargetDir) {
      if (is_dir($targetDir) && ($dir = opendir($targetDir))) {
        while (($file = readdir($dir)) !== false) {
          $tmpfilePath = $targetDir . DIRECTORY_SEPARATOR . $file;

          if (preg_match('/\.part$/', $file) && (filemtime($tmpfilePath) < time() - $maxFileAge) && ($tmpfilePath != "{$filePath}.part")) {
            @unlink($tmpfilePath);
          }
        }
        closedir($dir);
      } else {
        die('{"jsonrpc" : "2.0", "error" : {"code": 100, "message": "Failed to open temp directory."}, "id" : "id"}');
      }
    }

    if ($request->getHttpHeader('Content-Type', '')) {
      $contentType = $request->getHttpHeader('Content-Type', '');
    }
    
    if ($request->getHttpHeader('HTTP_CONTENT_TYPE')) {
      $contentType = $request->getHttpHeader('HTTP_CONTENT_TYPE');
    }
    
    if ($request->getHttpHeader('CONTENT_TYPE')) {
      $contentType = $request->getHttpHeader('CONTENT_TYPE');
    }

    $files = $request->getFiles();

    if (strpos($contentType, "multipart") !== false) {
      if (isset($files['file']['tmp_name']) && is_uploaded_file($files['file']['tmp_name'])) {

        $out = @fopen("{$filePath}.part", $chunk == 0 ? "wb" : "ab");

        if ($out) {
          $in = @fopen($files['file']['tmp_name'], "rb");

          if ($in) {
            while ($buff = fread($in, 4096))
              fwrite($out, $buff);
          } else
            die('{"jsonrpc" : "2.0", "error" : {"code": 101, "message": "Failed to open input stream."}, "id" : "id"}');

          @fclose($in);
          @fclose($out);
          @unlink($files['file']['tmp_name']);
        } else
          die('{"jsonrpc" : "2.0", "error" : {"code": 102, "message": "Failed to open output stream."}, "id" : "id"}');
      } else
        die('{"jsonrpc" : "2.0", "error" : {"code": 103, "message": "Failed to move uploaded file."}, "id" : "id"}');
    } else {

      $out = @fopen("{$filePath}.part", $chunk == 0 ? "wb" : "ab");

      if ($out) {
        $in = @fopen("php://input", "rb");

        if ($in) {
          while ($buff = fread($in, 4096))
            fwrite($out, $buff);
        } else
          die('{"jsonrpc" : "2.0", "error" : {"code": 101, "message": "Failed to open input stream."}, "id" : "id"}');

        @fclose($in);
        @fclose($out);
      } else
        die('{"jsonrpc" : "2.0", "error" : {"code": 102, "message": "Failed to open output stream."}, "id" : "id"}');
    }

    if (!$chunks || $chunk == $chunks - 1) {
      $fileSystem->rename("{$filePath}.part", $filePath);
    }

    $this->convertPhotoes($filePath, $id, $fileName);

    die('{"jsonrpc" : "2.0", "result" : null, "id" : "id"}');
  }

  public function convertPhotoes($filePath, $id, $fileName) {
    $article = Doctrine::getTable('Article')->findOneById($id);

    if (file_exists($filePath)) {
      $articlePhoto = new ArticlePhoto();
      $articlePhoto->setArticleId($id);
      
      foreach (GalleryManagement::getPhotoSizes($article) as $size => $options) {
        if (isset($options['extension'])) {
          $extension = $options['extension'];
        } else {
          $extension = '.png';
        }

        $fileName = artPhotoConverter::generateFileName('article_photo', $extension);
        $targetDir = $article->getGalleryPath($size, true) . $fileName;

        artPhotoConverter::resizePhoto(
            $options, $filePath, $targetDir
        );
        
        $articlePhoto->set($size.'_image', $fileName);
      }
      
      $articlePhoto->save();
      
      Doctrine::getTable('ArticlePhoto')->repairOrders($articlePhoto->getArticleId());
    }
  }

  /**
   * Akcja AJAX edycji artykulu inline.
   * routing.yml: fm_article_edit_inline
   * Potrzebne prawa: [admin, article_edit]
   * (Poza security.yml poniewaz chcemy inaczej obsluzyc brak praw)
   * 
   * @param sfWebRequest $request
   */
  public function executeInlineSave(sfWebRequest $request) {
    if (!$this->checkPermissions(array('admin', 'article_edit'))) {
      return $this->renderText(Redirector::get('AjaxNoPermission', array('user' => $this->getUser())));
    }

    $id = $request->getParameter('id');
    $model = $request->getParameter('model');
    $formClass = $request->getParameter('formClass');

    $object = Doctrine::getTable($model)->findOneById($id);

    if (!$object) {
      return $this->renderText(Redirector::get('FmArticleNotExist', array('user' => $this->getUser())));
    }

    $form = new $formClass($object);

    if ($form->bindFromRequestAndSave($request)) {
      $this->fmClearCache();

      if (method_exists($form->getObject(), 'getUrl')) {

        if ($form->getObject()->getUrl()) {
          $this->getResponse()->setStatusCode(301, sprintf('Przekierowanie, %s', get_class($form->getObject())));
          return $this->renderText($form->getObject()->getUrl());
        }
      }
      
      $this->getUser()->setNotice("Zmiany zostały zapisane!"); 
    }

    return $this->renderText('Success');
  }

  /**
   * Funkcja przeznaczona do zmiany, jesli chcemy ograniczyc zakres czyszczenia
   * cache po zmianie artykulow. Wywoluje sie po dodaniu, edycji, usunieciu
   * artykulow, referencji, kategorii.
   */
  public function fmClearCache() {
    $cacheDir = sfConfig::get('sf_cache_dir') . '/';
    $cache = new sfFileCache(array('cache_dir' => $cacheDir));
    $cache->clean();
  }

}
