<?php
/**
 * Task wyszukuje elementy i18n w plikach PHP. Jak parametr przyjmuje nazwę aplikacji.
 *
 * @package    artI18nPlugin
 * @subpackage task
 */

class artI18nExtractTask extends sfBaseTask
{
    /**
     * @see sfTask
     */
    protected function configure()
    {
        $this->addArguments(array(
            new sfCommandArgument('application', sfCommandArgument::REQUIRED, 'Nazwa aplikacji'),
            new sfCommandArgument('culture', sfCommandArgument::REQUIRED, 'Język'),
        ));

        $this->addOptions(array(
            new sfCommandOption('catalogue', null, sfCommandOption::PARAMETER_REQUIRED, 'Katalog', 'messages'),
            new sfCommandOption('display-new', null, sfCommandOption::PARAMETER_NONE, 'Wyświetla wszystkie znalezione łańcuchy znakowe.'),
            new sfCommandOption('display-old', null, sfCommandOption::PARAMETER_NONE, 'Wyświetla stare wyniki.'),
            new sfCommandOption('auto-save', null, sfCommandOption::PARAMETER_NONE, 'Zapis do bazy danych.'),
            new sfCommandOption('auto-delete', null, sfCommandOption::PARAMETER_NONE, 'Usuwanie starych wyników.'),
        ));

        $this->namespace = 'artI18n';
        $this->name = 'extract';
        $this->aliases = array("e");
        $this->briefDescription = 'Wyciąga łańcuchy znakowe do tłumaczenia z aplikacji.';

        $this->detailedDescription = <<<EOF
    [artI18n:extract|INFO]. Task wyszukuje elementy i18n w plikach PHP. Jak parametr przyjmuje nazwę aplikacji.

      [./symfony artI18n:extract pl|INFO]

    Domyślnie task wyświetli tylko numery nowych i starych elementów znalezionych w plikach projektu.

    Jeżeli chcesz mienić domyślny katalog, użyj opcji [--catalogue=messages]:

      [./symfony artI18n:extract --display-new frontend fr|INFO]

    Jeżeli chcesz wyświetlić nowe znalezione elementy użyj opcji [--display-new|COMMENT]:

      [./symfony artI18n:extract --display-new frontend fr|INFO]

    Aby zapisać wyniki do bazy danych użyj opcji [--auto-save|COMMENT]:

      [./symfony artI18n:extract --auto-save frontend fr|INFO]

    Jeżeli chcesz wyświetlić elementy znajdujące się w aplikacji a nie znajdujące się w aplikacji.
    [--display-old|COMMENT]:

      [./symfony artI18n:extract --display-old frontend fr|INFO]

    Opcja usuwa stare tłumaczenia z bazy danych [--auto-delete|COMMENT].

      [./symfony artI18n:extract --auto-delete frontend fr|INFO]
EOF;
    }

    /**
     * @see sfTask
     */
    public function execute($arguments = array(), $options = array())
    {
        $this->logSection('i18n', sprintf('Szukanie tekstów i18n z aplikacji "%s"', $arguments['application']));

        // get i18n configuration from factories.yml
        $config = sfFactoryConfigHandler::getConfiguration($this->configuration->getConfigPaths('config/factories.yml'));

        $class = $config['i18n']['class'];
        $params = $config['i18n']['param'];
        unset($params['cache']);

        $class = new $class($this->configuration, new sfNoCache(), $params);
        $class->getMessageSource()->setCatalogueName($options['catalogue']);

        $extract = new sfI18nApplicationExtract($class, $arguments['culture']);

        $extract->extract();

        $this->logSection('i18n', sprintf('Znaleziono "%d" nowych elementów i18n', count($extract->getNewMessages())));
        $this->logSection('i18n', sprintf('Znaleziono "%d" starych elementów i18n', count($extract->getOldMessages())));

        if ($options['display-new']) {
            $this->logSection('i18n', sprintf('Wyświetl nowe elementy i18n', count($extract->getOldMessages())));
            foreach ($extract->getNewMessages() as $message) {
                $this->log('               ' . $message . "\n");
            }
        }

        if ($options['auto-save']) {
            $this->logSection('i18n', 'Zapis nowych elementów i18n do bazy danych');

            $extract->saveNewMessages();
            $this->logSection('i18n', sprintf('Zapisano "%d" elementów', $class->getMessageSource()->getNumSaved()));
        }

        if ($options['display-old']) {
            $this->logSection('i18n', sprintf('Wyświetlenie starych elementów i18n', count($extract->getOldMessages())));
            foreach ($extract->getOldMessages() as $message) {
                $this->log('               ' . $message . "\n");
            }
        }

        if ($options['auto-delete']) {
            $this->logSection('i18n', 'Usuwanie starych elementów i18n');

            $extract->deleteOldMessages();
        }
    }
}
