<?php
/**
 * Task wyszukuje elementy i18n w plikach PHP. Jak parametr przyjmuje nazwę pluginu.
 *
 * @package    artI18nPlugin
 * @subpackage task
 */

class artI18nExtractPluginTask extends sfBaseTask
{
  /**
   * @see sfTask
   */
  protected function configure()
  {
    $this->addArguments(array(
      new sfCommandArgument('plugin', sfCommandArgument::REQUIRED, 'Nazwa pluginu'),
      new sfCommandArgument('culture', sfCommandArgument::REQUIRED, 'W skrócie język (pl, en itp.)'),
      new sfCommandArgument('application', sfCommandArgument::OPTIONAL, 'Fake app', 'frontend'),
    ));

    $this->addOptions(array(
      new sfCommandOption('catalogue', null, sfCommandOption::PARAMETER_REQUIRED, 'Katalog', 'messages'),
      new sfCommandOption('display-new', null, sfCommandOption::PARAMETER_NONE, 'Wyświetla wszystkie znalezione łańcuchy znakowe.'),
      new sfCommandOption('display-old', null, sfCommandOption::PARAMETER_NONE, 'Wyświetla stare wyniki.'),
      new sfCommandOption('auto-save', null, sfCommandOption::PARAMETER_NONE, 'Zapis do bazy danych.'),
      new sfCommandOption('auto-delete', null, sfCommandOption::PARAMETER_NONE, 'Usuwanie starych wyników.'),
    ));

    $this->namespace = 'artI18n';
    $this->name = 'extract-plugins';
    $this->aliases = array("ep");
    $this->briefDescription = 'Wyciąga łańcuchy znakowe do tłumaczenia z plików PHP pluginu';

    $this->detailedDescription = <<<EOF
    [artI18n:extract|INFO]. Task wyszukuje elementy i18n w plikach PHP. Jak parametr przyjmuje nazwę pluginu.

      [./symfony artI18n:extract-plugins pl|INFO]

    Domyślnie task wyświetli tylko numery nowych i starych elementów znalezionych w plikach projektu.

    Jeżeli chcesz mienić domyślny katalog, użyj opcji [--catalogue=messages]:

      [./symfony artI18n:extract-plugins --display-new frontend fr|INFO]

    Jeżeli chcesz wyświetlić nowe znalezione elementy użyj opcji [--display-new|COMMENT]:

      [./symfony artI18n:extract-plugins --display-new frontend fr|INFO]

    Aby zapisać wyniki do bazy danych użyj opcji [--auto-save|COMMENT]:

      [./symfony artI18n:extract-plugins --auto-save frontend fr|INFO]

    Jeżeli chcesz wyświetlić elementy znajdujące się w aplikacji a nie znajdujące się w aplikacji.
    [--display-old|COMMENT]:

      [./symfony artI18n:extract-plugins --display-old frontend fr|INFO]

    Opcja usuwa stare tłumaczenia z bazy danych [--auto-delete|COMMENT].

      [./symfony artI18n:extract-plugins --auto-delete frontend fr|INFO]
EOF;
  }

  /**
   * @see sfTask
   */
  public function execute($arguments = array(), $options = array())
  {
    $arguments['application'] = 'frontend';

    $this->logSection('i18n', sprintf('Wyciąganie łańcychów znakowych i18n z pluginu "%s" plugin', $arguments['plugin']));

    $config = sfFactoryConfigHandler::getConfiguration($this->configuration->getConfigPaths('config/factories.yml'));

    $class = $config['i18n']['class'];
    $params = $config['i18n']['param'];
    unset($params['cache']);

    $class = new $class($this->configuration, new sfNoCache(), $params);
    $class->getMessageSource()->setCatalogueName($options['catalogue']);


    $extract = new artI18nExtract($class, $arguments['culture']);

    if($extract->extractPlugin($arguments['plugin'])){

        $this->logSection('i18n', sprintf('Znaleziono "%d" nowych wyników', count($extract->getNewMessages())));
        $this->logSection('i18n', sprintf('Znaleziono "%d" starych wyników', count($extract->getOldMessages())));

        if ($options['display-new'])
        {
            $this->logSection('i18n', sprintf('Lista znalezionych nowych wyników.', count($extract->getOldMessages())));
            foreach ($extract->getNewMessages() as $message)
            {
                $this->log('               '.$message."\n");
            }
        }

        if ($options['auto-save'])
        {
            $this->logSection('i18n', 'Zapis elementów.');

            $extract->saveNewMessages();

            $this->logSection('i18n', sprintf('Zapisano "%d" wiadomośći',  $class->getMessageSource()->getNumSaved()));
        }

        if ($options['display-old'])
        {
            $this->logSection('i18n', sprintf('Wyświetlenie starych elemntów.', count($extract->getOldMessages())));
            foreach ($extract->getOldMessages() as $message)
            {
                $this->log('               '.$message."\n");
            }
        }

        if ($options['auto-delete'])
        {
            $this->logSection('i18n', 'Usuwanie starych wyników.');

            $extract->deleteOldMessages();
        }

    } else {
        throw new Exception(sprintf('Plugin o nazwie "%s" nie istnieje',  $arguments['plugin']));
    };

  }
}
