<?php

/**
 * artLESS to klasa umożliwiająca kompilację kodu LESS w projektach symfony.
 *
 * @package    artLESSPlugin
 * @subpackage less

 * @version    1.0.0
 */
class artLESS {

  /**
   * Tablica ze stylami LESS
   *
   * @var array
   */
  protected static $results = array();

  /**
   * Błędy kompilatora
   *
   * @var array
   */
  protected static $errors = array();

  /**
   * Aktualnie kompilowany plik. Wyswietalny podczas błędu kompilacji.
   *
   * @var string
   */
  protected $currentFile;

  /**
   * Klasa konfiguracyjna.
   *
   * @var artLESSConfig
   */
  protected static $config;

  /**
   * Konstructor
   *
   * @param
   */
  public function __construct(artYamlLESSConfig $config = null) {
    if ($config) {
      self::$config = $config;
    }
  }

  /**
   * Zwaraca klase konfiguracujną.
   *
   * @return  artLESSConfig instance
   */
  public static function getConfig() {
    self::$config = self::$config ? self::$config : new artYamlLESSConfig();
    return self::$config;
  }

  /**
   * Zwraca tablicę z kompilowanymi stylami.
   *
   * @return  array
   */
  public static function getCompileResults() {
    return self::$results;
  }

  /**
   * Zwraca tablicę z błędami.
   *
   * @return  array
   */
  public static function getCompileErrors() {
    return self::$errors;
  }

  /**
   * Zwraca wszystkie pliki w folderzez z CSS'ami.
   *
   * @return  array
   */
  public function findCssFiles() {
    return sfFinder::type('file')
                    ->exec(array('artLESSUtils', 'isCssLessCompiled'))
                    ->name('*.css')
                    ->in(self::getConfig()->getCssPaths());
  }

  /**
   * Zwraca wszystkie pliki w folderzez z LESS'ami.
   *
   * @return  array
   */
  public function findLessFiles() {
    return sfFinder::type('file')
                    ->name('*.less')
                    ->discard('_*')
                    ->follow_link()
                    ->in(self::getConfig()->getLessPaths());
  }

  /**
   * Zwraca zwraca ściezkę CSS, alternatywną do pliku LESS.
   *
   * @param   string  $lessFile LESS file path
   *
   * @return  string            CSS file path
   */
  public function getCssPathOfLess($lessFile) {
    $file = preg_replace('/\.less$/', '.css', $lessFile);
    $file = preg_replace(sprintf('/^%s/', preg_quote(self::getConfig()->getLessPaths(), '/')), self::getConfig()->getCssPaths(), $file);
    return $file;
  }

  /**
   * Uruchamia task, łączący obrazki w spraity. 
   */
  public function runSpriteJoiner() {
    chdir(sfConfig::get('sf_root_dir'));
    $task = new sfCssSpritesTask(sfContext::getInstance()->getEventDispatcher(), new sfFormatter());
    $task->run(array(), array());
  }

  /**
   * Kompilacja pliku.
   *
   * @param   string  $lessFile , plik LESS
   *
   * @return  boolean
   */
  public function compile($lessFile) {
    // Tworzy timer
    $timer = new sfTimer;

    // Pobiera sciezkę do plików
    $cssFile = $this->getCssPathOfLess($lessFile);
    artLESSUtils::createFolderIfNeeded($cssFile);

    // Czy plik skompilowano
    $isCompiled = false;

    // Jeśli sprawdzamy daty, rekompilacja tylko starych plików.
    if (self::getConfig()->isCheckDates()) {
      try {
        $d = new artLESSDependency(sfConfig::get('sf_web_dir'), sfConfig::get('app_less_plugin_check_dependencies', false));
        $shouldCompile = !is_file($cssFile) || $d->getMtime($lessFile) > filemtime($cssFile);
      } catch (Exception $e) {
        $shouldCompile = false;
      }
    } else {
      $shouldCompile = true;
    }

    if ($shouldCompile) {
      $buffer = $this->callLesscCompiler($lessFile, $cssFile);
      if ($buffer !== false) {
        $isCompiled = $this->writeCssFile($cssFile, $buffer) !== false;

        // Sprite joiner
        if (self::getConfig()->isSpriteJoinerEnabled()) {
          $this->runSpriteJoiner();
        }
      }
    }


    // Dodanie informacjo debugera, do tablicy debugera.
    self::$results[] = array(
        'lessFile' => $lessFile,
        'cssFile' => $cssFile,
        'compTime' => $timer->getElapsedTime(),
        'isCompiled' => $isCompiled
    );

    return $isCompiled;
  }

  /**
   * Zapisuje treść CSS do pliku.
   *
   * @param   string  $cssFile  , plik
   * @param   string  $buffer  , zawartość
   *
   * @return  boolean
   */
  public function writeCssFile($cssFile, $buffer) {
    // Napraw zduplikowane linie kodu.
    if (self::getConfig()->getFixDuplicate()) {
      $buffer = artLESSUtils::fixDuplicateLines($buffer);
    }

    // Komprensuj pliki jesli właczona taka opcja.
    if (self::getConfig()->isUseCompression()) {
      $buffer = artLESSUtils::getCompressedCss($buffer);
    }

    // dodaj nagłówek o "nie edytowalności pliku", i zapisz plik.
    $status = file_put_contents($cssFile, artLESSUtils::getCssHeader() . "\n\n" . $buffer);

    if ('\\' != DIRECTORY_SEPARATOR && $status !== false && fileowner($cssFile) == posix_geteuid()) {
      // Zmiana praw
      chmod($cssFile, 0666);
    }

    return $status;
  }

  /**
   * Wywołanie lessc kompilerera dla pliku LESS
   *
   * @param   string  $lessFile
   * @param   string  $cssFile
   *
   * @return  string
   */
  public function callLesscCompiler($lessFile, $cssFile) {
    // Ustawianie aktualnego pliku, zostanie wuświalony jesli wystapią błędy.
    $this->currentFile = $lessFile;

    if (self::getConfig()->isUseLessphp()) {
      require_once(sfConfig::get('sf_plugins_dir') . '/artLessPlugin/lib/lessphp/LessCompiler.class.php');
      $less = new LessCompiler($lessFile);
      file_put_contents($cssFile, $less->parse());
    } else {
      // Komplilacja poprzez lessc, powinien być w katalogu lib/compilators.
      $fs = new sfFilesystem;
      $command = sprintf('lessc "%s" "%s"', $lessFile, $cssFile);

      if ('1.3.0' <= SYMFONY_VERSION) {
        try {
          $fs->execute($command, null, array($this, 'throwCompilerError'));
        } catch (RuntimeException $e) {
          return false;
        }
      } else {
        $fs->sh($command);
      }
    }

    // Ustaw aktualny plik na nulla.
    $this->currentFile = null;

    return file_get_contents($cssFile);
  }

  /**
   * Zwraca true, jeżli kompilator może wyrzucić wyjątek.
   *
   * @return boolean
   */
  public function canThrowExceptions() {
    return (('prod' !== sfConfig::get('sf_environment') || !sfConfig::get('sf_app')) &&
            !(sfConfig::get('sf_web_debug') && sfConfig::get('app_less_plugin_toolbar', true))
            );
  }

  /**
   * Zwraca błąd o zmienionym formacie.
   *
   * @param   string  $line error
   *
   * @return  boolean
   */
  public function throwCompilerError($line) {
    // Gnerowanie opisu błędu.
    $errorDescription = sprintf("Błąd kompilacji LESS w \"%s\":\n\n%s", $this->currentFile, $line);

    // Dodanie opisu błedu do listy błędów.
    self::$errors[$this->currentFile] = $errorDescription;

    // Wyrzuc wyjatek.
    if ($this->canThrowExceptions()) {
      throw new sfException($errorDescription);
    } else {
      sfContext::getInstance()->getLogger()->err($errorDescription);
    }
    return false;
  }
}