<?php
/**
 *
 */
class BaseTabFormConfiguration
{
    /**
     * zmienna decydująca o tym, czy w tablicy konfiguracji wymagana jest opcja "order". Służy ona do ustawiania
     * kolejności zakładek poprzez opcję w tablicy, a nie poprzez kolejność zakładek w tablicy
     * @var boolean
     */
    protected $withOrder = false;
    /**
     *
     * @var array
     */
    protected $configuration = array();
    /**
     *
     * @var sfContext
     */
    protected $context;
    /**
     *
     * @var sfWebRequest
     */
    protected $request;
    /**
     * nazwa zakładki w której jesteśmy
     * @var string
     */
    protected $tabName;
    /**
     * nazwa modułu, w którym jestesmy
     * @var string
     */
    protected $module;
    protected $foreignKeys = array();
    protected $tabsInOrder = array();
    protected $hasMultipartForm = false;

    public function __construct(artTabActions $actionsObject)
    {
        $this->validateConfiguration($actionsObject);
        $this->context = sfContext::getInstance();
        $this->request = $this->context->getRequest();
        $this->tabName = $this->request->getParameter('tab_name', '');
        $this->module = $this->request->getParameter('module', '');
        $this->foreignKeys = $this->getForeignKeysFromRequest();
        $this->setTabsInOrder();
    }

    public function getConfiguration($tabName = null)
    {
        if (!is_null($tabName) && isset($this->configuration[$tabName]))
            return $this->configuration[$tabName];
        else
            return $this->configuration;
    }

    public function getForeignKeys()
    {
        return $this->foreignKeys;
    }

    protected function setTabsInOrder()
    {
        $this->tabsInOrder = array();

        if (!$this->withOrder)
            foreach ($this->configuration as $tabName => $tabOptions)
                $this->tabsInOrder[] = $tabName;
        else {
            foreach ($this->configuration as $tabName => $tabOptions) {
                if (isset($tabOptions['order']))
                    $order = $tabOptions['order'];
                else
                    throw new sfException("Dla zakładki $tabName nie zdefiniowano 'order', pomimo, że dla danego modułu jest wymuszone podawanie kolejności zakładek.");

                if (isset($this->tabsInOrder[$order]))
                    throw new sfException('Dwie zakładki mają taki sam order!');

                $this->tabsInOrder[$order] = $tabName;
            }

            ksort($this->tabsInOrder);
            $this->tabsInOrder = array_values($this->tabsInOrder);
        }
    }

    /**
     * pobierz wartości kluczy obcych z requesta
     * @return array
     */
    public function getForeignKeysFromRequest()
    {
        $foreignKeys = array();

        foreach ($this->configuration as $tabOptions) {
            $keys = artArray::getFromArray($tabOptions, 'keys_from_request', array());
            $keys[] = artArray::getFromArray($tabOptions, 'foreign_key_name');

            foreach ($keys as $key)
                if ($key != '')
                    $foreignKeys[$key] = $this->request->getParameter($key, null);
        }

        return $foreignKeys;
    }

    /**
     * pobierz wartość klucza obcego dla danej zakładki
     * @return type
     */
    public function getForeignKeyForTab()
    {
        $foreignKeyName = $this->configuration[$this->tabName]['foreign_key_name'];

        return $this->request->getParameter($foreignKeyName, null);
    }

    /**
     * czy istnieje w requescie klucz obcy dla danej zakladki
     * @return boolean
     */
    public function issetForeignKeyForTabInRequest()
    {
        return !is_null($this->getForeignKeyForTab());
    }

    /**
     * funkcja pomocnicza, pobierająca wartość klucza $value z tablicy zakładki w której jesteśmy. Wstawia $default jeżeli nie istnieje
     * @param type $value
     * @param type $default
     * @return type
     */
    protected function getFromTabArray($value, $default = '')
    {
        return artArray::getFromArray($this->configuration[$this->tabName], $value, $default);
    }

    public function getFormClass()
    {
        $formClass = $this->getFromTabArray('form_class', null);

        if (class_exists($formClass))
            return $formClass;
        else
            return 'Plugin' . $formClass;
    }

    public function isMultipartForm()
    {
        return $this->getFromTabArray('multipart_form', false);
    }

    public function getForeignKeyName()
    {
        return $this->getFromTabArray('foreign_key_name', null);
    }

    /**
     * zwraca "podstawowy" url dla danej zakladki w module
     * @return string
     */
    public function getUrl()
    {
        $this->context->getConfiguration()->loadHelpers(array('Url'));

        return url_for('@get_tab_form?module=' . $this->module . '&tab_name=' . $this->tabName);
    }

    /**
     * zwraca urla razem z kluczami obcymi, ktore sa ustawione w requescie
     * @return string url z kluczami obcymi
     */
    public function getUrlWithKeys()
    {
        $url = $this->getUrl() . '?';

        foreach ($this->getForeignKeys() as $key => $value)
            if (!is_null($value))
                $url .= $key . '=' . $value . '&';

        return substr($url, 0, -1);
    }

    public function getDoctrineObjectClass()
    {
        return $this->getFromTabArray('doctrine_object_class', '');
    }

    public function getModule()
    {
        return $this->module;
    }

    /**
     * funkcja zwraca tablicę javascriptową na zasadzie $foreignKey => $foreignKeyValue
     * @return string tablica javascriptowa
     */
    public function getJavascriptForeignKeysArray()
    {
        $jsArray = array();

        foreach ($this->getForeignKeys() as $key => $value) {
            $value = is_null($value) ? 'null' : '"' . $value . '"';
            $jsArray[] = '"' . $key . '" : ' . $value;
        }

        return '{ ' . implode(', ', $jsArray) . ' }';
    }

    /**
     * funkcja zwraca tablicę javascriptową na zasadzie $tabName => $foreignKeyName
     * @return string tablica javascriptowa
     */
    public function getJavascriptReturnedFromTabArray()
    {
        $jsArray = array();

        foreach ($this->getConfiguration() as $tabName => $tabOptions) {
            $value = artArray::getFromArray($tabOptions, 'foreign_key_name', null);
            $value = is_null($value) ? 'null' : '"' . $value . '"';
            $jsArray[] = '"' . $tabName . '" : ' . $value;
        }

        return '{ ' . implode(', ', $jsArray) . ' }';
    }

    /**
     * funkcja zwraca tablicę javascriptową na zasadzie $tabName => true/false
     * true jeżeli jest formularz na danej zakładce, false jeżeli nie ma.
     * @return string tablica javascriptowa
     */
    public function getJavascriptTabsHasFormArray()
    {
        $jsArray = array();

        foreach ($this->getConfiguration() as $tabName => $tabOptions) {
            $value = artArray::getFromArray($tabOptions, 'form_class', false);
            $value = $value === false ? 'false' : 'true';
            $jsArray[] = '"' . $tabName . '" : ' . $value;
        }

        return '{ ' . implode(', ', $jsArray) . ' }';
    }

    /**
     * @return string tablica javascriptowa
     */
    public function getJavascriptSubmitFormOnChangeTabArray()
    {
        $jsArray = array();

        foreach ($this->getConfiguration() as $tabName => $tabOptions) {
            $value = artArray::getFromArray($tabOptions, 'submit_form_on_change_tab', true);
            $value = $value === false ? 'false' : 'true';
            $jsArray[] = '"' . $tabName . '" : ' . $value;
        }

        return '{ ' . implode(', ', $jsArray) . ' }';
    }

    /**
     * funkcja zwraca dwustopniową tablice javascriptową na zasadzie:
     * array($tabName = $tabName['required_foreign_keys'])
     * @return string tablica javascriptowa
     */
    public function getJavascriptRequiredForeignKeysArray()
    {
        $jsArray = array();

        foreach ($this->getConfiguration() as $tabName => $tabOptions) {
            $requiredsArray = array();

            foreach (artArray::getFromArray($tabOptions, 'required_foreign_keys', array()) as $requiredForeignKey)
                $requiredsArray[] = '"' . $requiredForeignKey . '"';

            $jsArray[] = '"' . $tabName . '" : [' . implode(', ', $requiredsArray) . ']';
        }

        return '{ ' . implode(', ', $jsArray) . ' }';
    }

    public function getTabName()
    {
        return $this->tabName;
    }

    public function getTabsInOrder()
    {
        return $this->tabsInOrder;
    }

    public function setForeignKeyForCurrentTab($foreignKeyValue)
    {
        $this->foreignKeys[$this->getForeignKeyName()] = $foreignKeyValue;
    }

    public function setForeignKeys($foreignKeysValuesArray)
    {
        $this->foreignKeys = array_merge($this->foreignKeys, $foreignKeysValuesArray);

        foreach ($this->foreignKeys as $key => $value)
            $this->request->setParameter($key, $value);
    }

    public function getFirstTab()
    {
        if ($this->request->hasParameter('open_tab'))
            return $this->request->getParameter('open_tab');

        $lastTab = end($this->tabsInOrder);
        reset($this->tabsInOrder);

        do {
            $tabName = current($this->tabsInOrder);
            next($this->tabsInOrder);
        } while (!$this->isEnabledTab($tabName) && $tabName != $lastTab);

        return $tabName;
    }

    /**
     * funkcja sprawdza, czy dana zakładka jest aktywna - jest to sprawdzane na podstawie wartości ustawionych w tablicy
     * $this->foreignKeys oraz wymaganych kluczy obcych do aktywacji danej zakładki
     * @param type $tabName
     * @return type
     */
    public function isEnabledTab($tabName)
    {
        foreach (artArray::getFromArray($this->getConfiguration($tabName), 'required_foreign_keys', array()) as $requiredKey)
            if (!isset($this->foreignKeys[$requiredKey]) || is_null($this->foreignKeys[$requiredKey]))
                return false;

        return true;
    }

    public function isDisabledTab($tabName)
    {
        return !$this->isEnabledTab($tabName);
    }

    /**
     * funkcja walidująca tablicę konfiguracji - tzn, jeżeli coś podaliśmy źle tablicy, dostaniemy exceptiona z informacją co dokładnie zepsuliśmy
     * @param artTabActions $actionsObject
     */
    protected function validateConfiguration(artTabActions $actionsObject)
    {
        $requiredsEmpty = false;

        foreach ($this->configuration as $tabName => $tabOptions) {
            if (artArray::emptyExceptKeys($tabOptions, array('required_foreign_keys', 'multipart_form', 'order')))
                if (!method_exists($actionsObject, 'getTabsForm' . $tabName))
                    throw new sfException('Dla zakładki ' . $tabName . ' nie podałeś tablicy opcji, więc w klasie ' . get_class($actionsObject) . ' powinieneś stworzyć metodę getTabsForm' . $tabName . '().');

            if (isset($tabOptions['form_class']))
                if (!isset($tabOptions['foreign_key_name']) && !isset($tabOptions['doctrine_object_class']))
                    throw new sfException("Dla zakładki $tabName nie zdefiniowałeś opcji 'foreign_key_name' ani 'doctrine_object_class'.");

            $requiredForeignKeysForTab = artArray::getFromArray($tabOptions, 'required_foreign_keys', array());

            if (empty($requiredForeignKeysForTab))
                $requiredsEmpty = true;

            if (artArray::getFromArray($tabOptions, 'multipart_form', false))
                $this->hasMultipartForm = true;
        }

        if (!$requiredsEmpty)
            throw new sfException('Nie zdefiniowano żadnej zakładki, w której tablica "required_foreign_keys" byłaby pusta, co oznacza, że żadna zakładka nie będzie aktywna przy wchodzeniu na nowy formularz!');
    }

    public function hasMultipartForm()
    {
        return $this->hasMultipartForm;
    }

    public function getTabI18nParams($tabName)
    {
        return artArray::getFromArray($this->getConfiguration($tabName), 'tab_i18n_params', array());
    }

    public function getTabI18nName($tabName)
    {
        return artArray::getFromArray($this->getConfiguration($tabName), 'tab_i18n_name', '');
    }

    public function getActualTabSubmitName()
    {
        return $this->getFromTabArray('tab_submit_name', 'Save');
    }
}