<?php

abstract class artBaseSitemapTask extends sfBaseTask {

  protected $controller = null;
  protected $configuration = null;
  protected $context = null;
  protected $urls = array();
  protected $cultures = array();
  protected $defaultCulture = null;

  public function configure() {
    $this->namespace = 'arteneo';
    $this->name = 'generate-sitemap';
    $this->briefDescription = 'Generuje mape strony w XML';
  }

  public function execute($arguments = array(), $options = array()) {
    $this->configuration = sfApplicationConfiguration::getApplicationConfiguration('frontend', 'prod', true);
    $this->context = sfContext::createInstance($this->configuration);
    $this->controller = $this->context->getController();
    $this->cultures = sfConfig::get('sf_cultures', array());
    $this->defaultCulture = sfConfig::get('sf_default_culture', 'pl');
    ArtConfiguration::loadConfiguration();

    $filename = ArtConfiguration::getSitemapFilename();
    $enabled = ArtConfiguration::getSitemapEnabled();
    $this->host = ArtConfiguration::getHost();

    if ($enabled) {
      $host = ArtConfiguration::getHost();

      if (!$this->isHostValid($host)) {
        $this->logSection('generate-sitemap', 'Niepoprawnie zdefiniowany host w factories.yml.', null, 'ERROR');
      } else {
        // Ustawiamy hosta z ArtConfiguration i prefix, aby routingi byly poprawne
        $this->context->getRouting()->setHost($host);
        $this->context->getRouting()->setPrefix('');

        $this->registerUrls();
        $document = $this->prepareXmlStructure();

        if ($document->save('web/' . $filename)) {
          $this->logSection('generate-sitemap', 'Pomyślnie wygenerowano plik ' . $filename);
        } else {
          $this->logSection('generate-sitemap', 'Wystąpił błąd przy tworzeniu pliku.', null, 'ERROR');
        };
      }
    } else {
      $this->logSection('generate-sitemap', 'Generowanie mapy strony zostalo wylaczone.');
    }
  }

  /**
   * Przygotowuje strukture XML z listy urli
   * @return \DOMDocument 
   */
  protected function prepareXmlStructure() {
    $domtree = new DOMDocument('1.0', 'UTF-8');
    $domtree->formatOutput = true;

    $xmlRoot = $domtree->createElement("urlset");
    $xmlRoot->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $xmlRoot->setAttribute('xmlns:xhtml', 'http://www.w3.org/1999/xhtml');
    $xmlRoot = $domtree->appendChild($xmlRoot);

    foreach ($this->urls as $singleEntry) {
      $urlNode = $domtree->createElement("url");
      $xmlRoot->appendChild($urlNode);
      foreach ($singleEntry as $key => $value) {
        if ($key == 'cultures') {
          foreach ($value as $culture => $loc) {
            $element = $domtree->createElement('xhtml:link');
            $element->setAttribute('rel', 'alternate');
            $element->setAttribute('hreflang', $culture);
            $element->setAttribute('href', $loc);
            $urlNode->appendChild($element);
          }
        } else {
          $urlNode->appendChild($domtree->createElement($key, $value));
        }
      }
    }

    return $domtree;
  }

  /**
   * Dodaje url o podanych parametrach do listy. Podany jezyk jest traktowany
   * jako podstawowy
   * @param string $loc url
   * @param float $priority Priorytet
   * @param string $changefreq Changefreq
   * @param string $culture Jezyk
   * @return string Indeks dodanego elementu
   */
  protected function addUrl($loc, $priority, $changefreq, $culture) {
    $url = array(
        'loc' => $this->host.$loc,
        'changefreq' => $changefreq,
        'priority' => $priority,
        'cultures' => array(
            $culture => $this->host.$loc
        )
    );

    $url['cultures'][$culture] = $this->host.$loc;

    $this->urls[$loc] = $url;

    return $loc;
  }

  /**
   * Usuwa url o podanym indeksie
   * @param type $index Indeks url do usuniecia
   * @return boolean Czy url zostal pomyslnie usuniety
   */
  protected function removeUrl($index) {
    if (isset($this->urls[$index])) {
      unset($this->urls[$index]);
      return true;
    }

    return false;
  }

  /**
   * Dodaje informacje o stronie w innym jezyku do podanego URL
   * @param string $index Indeks na liscie urli
   * @param string $url Url
   * @param string $culture Jezyk
   * @return string Indeks zmienionego url
   */
  protected function addUrlCulture($index, $url, $culture) {
    if (!isset($this->urls[$index])) {
      $this->logSection('generate-sitemap', 'Brak elementu "' . $index . '" na liscie urli.', null, 'ERROR');
    } else {
      $this->urls[$index]['cultures'][$culture] = $url;
    }

    return $index;
  }

  /**
   * Zwraca url dla artykulu
   * @param string $slug
   * @return string 
   */
  protected function getArticleUrl($slug) {
    return $this->genUrl('@article_show?slug=' . $slug, true);
  }

  /**
   * Zwraca url dla kategorii
   * @param string $uniqueName
   * @return string
   */
  protected function getCategoryUrl($uniqueName) {
    return $this->genUrl('@article_list?category=' . $uniqueName, true);
  }

  /**
   * Funkcja dodaje url dla wszystkich artykulow i dla kazdego jezyka
   */
  protected function registerArticles($priority = 0.5, $changefreq = 'weekly') {

    
    foreach ($this->cultures as $culture) {
      $articles = Doctrine::getTable('Article')->getArticles($culture);

      foreach ($articles as $article) {
        if ($article->getUrl() != '' && $article->getUrl() && $article->isDisplayedAndActive()) {
          $index = $this->addUrl($article->getUrl(), $priority, $changefreq, $culture);
        }
      }
    }
  }

  /**
   * Funkcja dodaje url dla wszystkich kategorii i dla kazdego jezyka
   */
  protected function registerCategories($priority = 0.4, $changefreq = 'weekly') {
    
    foreach ($this->cultures as $culture) {
      
      $categories = Doctrine::getTable('Category')->getCategories($culture);
      
      foreach ($categories as $category) {
        if ($category->getUrl() != '' && $category->getUrl()) {
          $index = $this->addUrl($category->getUrl(), $priority, $changefreq, $culture);
        }
      }
    }
  }

  /**
   * Prosta funkcja priorytyzujaca
   * @param float $oldPriority Istniejacy priorytet
   * @param array $element Element menu
   * @return float 
   */
  protected function priorityRaiseSimple($oldPriority, $element) {
    $newPriority = $oldPriority;

    if ($element['depth'] == 1) {
      $newPriority += 0.3;
    } elseif ($element['depth'] == 2) {
      $newPriority += 0.2;
    } elseif ($element['depth'] == 3) {
      $newPriority += 0.1;
    } else {
      $newPriority += 0.05;
    }

    return min(1, $newPriority);
  }

  /**
   * Funkcja sprawdza wszystkie urle z menu. Aplikuje funkcje priorytyzujaca dla
   * istniejacych juz url. Jesli url nie istnieje i flaga $addNonExisingUrls jest
   * ustawiona na true, url jest dodawany z wartosciami podanymi w parametrach.
   * Glowna kategoria menu jest usuwana z listy urli.
   * 
   * @param string $menuFile Nazwa pliku menu
   * @param string $menuClass Nazwa klasy menu
   * @param string $prioritizeFunction Nazwa funkcji priorytyzujacej
   * @param float $priority Priorytet dla nowych url
   * @param string $changefreq Changefreq dla nowych url
   * @param boolean $addNonExisingUrls  Czy dodawac nowe url z menu
   */
  protected function registerMenu($menuName, $menuClass, $prioritizeFunction, $priority = 0.6, $changefreq = 'weekly', $addNonExisingUrls = true) {

    $context = sfContext::getInstance();
    $this->articleIds = array(); 
    $this->categoryIds = array(); 
    
    foreach ($this->cultures as $culture) {
      
      $context->getUser()->setCulture($culture);

      $this->dataCollector = new $menuClass(
              $menuName, $menuClass, $culture);

      $menuGroup = $this->dataCollector->getMainMenuGroup();

      foreach ($menuGroup['MenuItem'] as $menuItem) {

        if ($menuItem['url_type'] == $menuClass::URL_ARTICLE || $menuItem['url_type'] == $menuClass::URL_ARTICLE_TITLE) {
          $this->articleIds[] = $menuItem['Translation'][$culture]['url'];
        }

        if ($menuItem['url_type'] == $menuClass::URL_CATEGORY || $menuItem['url_type'] == $menuClass::URL_CATEGORY_TITLE) {
          $this->categoryIds[] = $menuItem['Translation'][$culture]['url'];
        }
        
        if ($menuItem['url_type'] == $menuClass::URL_ROUTING){
          
          $url = $context->getRouting()->generate($menuItem['Translation'][$culture]['url'], $params);
          
          $index = $this->addUrl($url, $priority, $changefreq, $culture);
        }
        
        if ($menuItem['url_type'] == $menuClass::URL_RAW){
          // Pusty link
        }
      }

      $this->articleIds = $this->dataCollector->removeDuplicates($this->articleIds);
      $articles = Doctrine::getTable('Article')->getArticlesByIds($this->articleIds, $culture);
      
      foreach ($articles as $article) {
        $index = $this->addUrl($article->getUrl(), $priority, $changefreq, $culture);
      }
      
      $this->categoryIds = $this->dataCollector->removeDuplicates($this->categoryIds);
      $categories = Doctrine::getTable('Category')->getCategoriesByIds($this->categoryIds, $culture);
      
      foreach ($categories as $category) {
        $index = $this->addUrl($category->getUrl(), $priority, $changefreq, $culture);
      }
    }
    
    $context->getUser()->setCulture($this->defaultCulture);
  }

  /**
   * Funkcja rejestruje wszystkie url jakie sa potrzebne
   * do wygenerowania XML.
   */
  protected function registerUrls() {
    $this->registerArticles(0.3, 'weekly');
    $this->registerCategories(0.25, 'weekly');

    $this->registerMenu('topMenu.yml', 'Menu', 'priorityRaiseSimple', 0.6, 'weekly', true);
    $this->registerMenu('footerMenu.yml', 'Menu', 'priorityRaiseSimple', 0.5, 'weekly', true);
  }

  /**
   * Sprawdza czy host jest poprawnie zdefiniowany (czy nie jest defaultowy)
   * @param string $host
   * @return boolean 
   */
  protected function isHostValid($host) {
    if ($host == 'creator' || $host == 'symfony') {
      return false;
    }

    return true;
  }

  /**
   * Generuje URL korzystajac z kontekstu
   * @param string $url Regula z routingu
   * @return string Wygenerowany URL
   */
  protected function genUrl($url) {
    return $this->controller->genUrl($url, true);
  }

}
?>
