<?php

class arteneoGenerateLogDbTask extends arteneoGenerateDbTask
{

    private $logModelArray = array();
    private $foreignLogKeys = array();
    private $translationFields = array();
    private $coulmnsCount;

    protected function configure()
    {
        $this->addOptions(array(
            new sfCommandOption('without-dataload', null, sfCommandOption::PARAMETER_NONE,
                'Do not load fixtures'),
            new sfCommandOption('without-sql', null, sfCommandOption::PARAMETER_NONE,
                'Do not load sql-files')
        ));

        $this->namespace = 'arteneo';
        $this->name = 'generate-log-db';
        $this->briefDescription = 'Runs arteneo:build-all, data-load and exec-sql and generates log tables ';
        $this->detailedDescription = <<<EOF
The [arteneo:generate-db|INFO] task does things.
Call it with:

  [php symfony arteneo:generate-db|INFO]
EOF;
    }

    protected function execute($arguments = array(), $options = array())
    {
        echo "Executing LogDB Task";
        $logModelArray = SchemaConfig::getLogTables();
        $this->logModelArray = $logModelArray;

        if (!empty($logModelArray)) {
            $this->emptySqlLogDir();
            $this->connectToDatabase();
            $columnsArray = $this->getColumnNames($logModelArray);
            $this->generateDatabase($columnsArray);
            $this->runGenerateDbTask($options);


//      
            foreach ($this->logModelArray as $array) {
                if (!$this->checkModuleExists('backend', sfInflector::tableize('log_' . $array)))
                    $this->runTask('arteneo:generate-log', array('application' => 'backend', 'route_or_model' => 'Log' . ucfirst($array)));
            }

        } else
            exit("No tables to log");
    }

    public function checkModuleExists($app, $module)
    {
        return (is_dir(sfConfig::get('sf_apps_dir') . '/' . $app . '/modules/' . $module));
    }

    protected function connectToDatabase()
    {
        $databaseManager = new sfDatabaseManager($this->configuration);
    }

    protected function runGenerateDbTask($options)
    {
        echo 'Generating Database';
        $this->runTask('arteneo:build-all');
//    $this->runTask('doctrine:build-model');
        $this->createTriggers($this->prepareData());
        if (!$options['without-sql'])
            $this->runTask('arteneo:exec-sql');

        if (!$options['without-dataload'])
            $this->runTask('arteneo:data-load');
    }

    protected function emptySqlLogDir()
    {
        $dir = sfConfig::get('sf_data_dir') . '/sql-log/';

        if (!$handle = @opendir($dir)) {
            mkdir($dir, 0777);
            $handle = opendir($dir);
        }

        while ($file = readdir($handle))
            if (artString::endsWith('.sql', $dir . $file))
                unlink($dir . $file);
    }

    protected function getColumnNames($logModelArray)
    {
        $modelArray = array();

        foreach ($logModelArray as $logModel)
            $modelArray[$logModel] = Doctrine_Core::getTable($logModel)->getColumnNames();

        return $modelArray;
    }

    protected function generateDatabase($tablesArray)
    {
        $content = $oldContent = $newContent = '';

        foreach ($tablesArray as $tableName => $columns) {
            $oldContent = $newContent = $i18nContent = '';
            $table = Doctrine_Core::getTable($tableName);
            $content .= $this->printTableHeader($tableName, $table);
            $indexesWithModels = $primaryKeys = array();
            $this->columnsCount = count($columns);

            foreach ($columns as $column) {
                $columnDefinition = $table->getColumnDefinition($column);
                $primaryKeys = array();
                $oldContent .= $this->printTableColumn($table, $column, $columnDefinition, 'old_', $primaryKeys);
                $newContent .= $this->printTableColumn($table, $column, $columnDefinition, 'new_', $primaryKeys);
            }

            $i18nContent .= $this->printi18nColumn($table);
            $content .= $oldContent . $newContent . $i18nContent . $this->appendPredefinedColumns();
            $indexesWithModels[$tableName] = $primaryKeys;
            $content .= $this->printTableFooter($table, $tableName, $indexesWithModels);
//      $this->createTriggers($tableName, $columns, $this->translationFields, $table);
        }
        $file = sfConfig::get('sf_config_dir') . '/doctrine/schema_log.yml';
        $handle = fopen($file, 'w+');
        fwrite($handle, $content);
        fclose($handle);
    }

    protected function printi18nColumn($table)
    {
        $column = '';
        if ($table->hasTemplate('Doctrine_Template_I18n')) {
            $template = $table->getTemplate('Doctrine_Template_I18n');
            $fields = $template->getOption('fields');
            $fields[] = 'lang';
            $this->translationFields = array('translation_id');

            foreach ($fields as $field) {
                $this->translationFields[] = $field;
                $definition = $table->getRelation('Translation')->getTable()->getColumnDefinition($field);
                $column .= sprintf(<<<EOF
    old_%1\$s:
      type: %2\$s
      length: %3\$s
    new_%1\$s:
      type: %2\$s
      length: %3\$s\n
EOF
                    , $field, $definition['type'], $definition['length']);
            }

            $pk = $table->getIdentifier();
            $definition = $table->getColumnDefinition($pk);
            $column .= sprintf(<<<EOF
    translation_old_%3\$s:
      type: %1\$s
      length: %2\$s
    translation_new_%3\$s:
      type: %1\$s
      length: %2\$s\n
EOF
                , $definition['type'], $definition['length'], $pk);
        }

        return $column;
    }

    protected function printTableHeader($tableName, $table)
    {
        $content = 'Log' . ucfirst($tableName) . ":\n";

        if ($table->hasTemplate('Encrypted')) {
            $options = $table->getTemplate('Encrypted')->getOptions();

            $content .= sprintf(<<<EOF
  actAs:
    Encrypted:
      fields: [%s]\n
EOF
                , 'new_' . implode(', new_', $options['fields']) . ', old_' . implode(', old_', $options['fields']));
        }
        $content .= sprintf(<<<EOF
  columns:
    id:
      type: integer(8)
      primary: true
      notnull: true
      autoincrement: true\n
EOF
        );


        return $content;
    }

    protected function printTableColumn($table, $columnName, $columnDefinition, $colPrefix, &$primaryKeys)
    {
        $isLogged = false;

        $pk = $table->getIdentifier();
        if (!is_array($pk))
            $pk = array($pk);

        foreach ($pk as $p)
            $primaryKeys[] = $colPrefix . $p;

        $column = '';
        if (!in_array($columnName, $pk))
            $column = $this->printColumnName($table, $colPrefix, $columnName, $isLogged);
        else
            $column = "    " . $colPrefix . $columnName . ":\n";

        if ($isLogged)
            $column .= sprintf(<<<EOF
      type: %s\n
EOF
                , 'integer(8)');

        else {
            foreach ($columnDefinition as $propertyName => $propertyValue) {
                if ($this->validateColumn($propertyName)) {
                    $column .= sprintf(<<<EOF
      %s: %s\n
EOF
                        , $propertyName, $propertyValue);
                }
            }
        }

        return $column;
    }

    protected function validateColumn($propertyName)
    {
        return $propertyName == 'type' || $propertyName == 'length';
//    return $propertyName != 'default'
//            && $propertyName != 'notnull'
//            && $propertyName != 'autoincrement'
//            && $propertyName != 'primary'
//            && $propertyName != 'unique';
    }

    protected function printColumnName($table, $colPrefix, $columnName, &$isLogged)
    {
        $relations = $table->getRelations();
        $bool = true;
        foreach ($relations as $rel) {
//      if ($oldRelation)
            $bool = $columnName == $rel->getLocalFieldName();
            if (!$bool)
                continue;

            $class = sfInflector::classify($rel['table']->getTableName());

            if (in_array($class, $this->logModelArray)) // && ($rel->getType() == Doctrine_Relation::ONE) && $bool && $rel['owningSide'])
            {
                $isLogged = true;
                $colPrefix .= 'log_';
//          $this->foreignLogKeys[$class] = $columnName;
            }
        }

        return "    " . $colPrefix . $columnName . ":\n";
    }


    protected function appendPredefinedColumns()
    {
        return sprintf(<<<EOF
    user_id:
      type: integer(8)
    date:
      type: datetime
      notnull: true
    flag:
      type: enum
      notnull: true
      values: [0, 1, 2, 3]\n
EOF
        );
    }

    protected function printTableFooter($table, $tableName, $indexesWithModels)
    {
        $content = sprintf(<<<EOF
  relations:
    sfGuardUser:
      local: user_id
      foreign: id
      foreignType: many
      owningSide: true
      foreignAlias: log%s
      foreignKeyName: fk_log_%2\$s_sfGuard\n
EOF
            , ucfirst($tableName), sfInflector::tableize($tableName));
        $content .= $this->printTableRelations($table);
        $content .= $this->printTableIndexes($indexesWithModels, $tableName);

        return $content;
    }

    protected function printTableIndexes($indexesWithModels, $tableName)
    {
        $content = sprintf(<<<EOF
  indexes:
    fk_%s_date:
      fields: [date]\n
EOF
            , $tableName);
        foreach ($indexesWithModels as $modelName => $modelWithIndexes)
            foreach ($modelWithIndexes as $fieldName) {
                $content .= sprintf(<<<EOF
    fk_%s_%2\$s:
      fields: [%s]\n
EOF
                    , $modelName, $fieldName);
            }

        return $content;
    }

    protected function printTableRelations($table)
    {
        $relations = $table->getRelations();
        $tableName = $table->getTableName();
        $content = '';

        foreach ($relations as $name => $relation) {
            $local = $relation->getLocal();
            $foreign = $relation->getForeign();
            $class = $relation->getClass();
            $name = ucfirst($name);

            if (!$relation->getType() == Doctrine_Relation::ONE || !$relation['owningSide'])
                continue;

            $pk = $table->getIdentifier();
            if (!is_array($pk))
                $pk = array($pk);

            if ($this->checkTableLogStatus($relation) && !in_array($local, $pk)) {
                $name = 'Log' . $name;
                $local = 'log_' . $local;
                $class = 'Log' . $class;
                $foreign = 'id';
            }

            $content .= sprintf(<<<EOF
    Old%s:
      local: old_%s
      foreign: %s
      class: %s
      owningSide: true
      foreignKeyName: old_fk_%1\$s_%5\$s
    New%1\$s:
      local: new_%2\$s
      foreign: %3\$s
      class: %4\$s
      owningSide: true
      foreignKeyName: new_fk_%1\$s_%5\$s\n
EOF
                , $name, $local, $foreign, $class, $tableName);
            $this->dropConstraints($tableName, $name);
        }
        $this->dropSfGuardConstraints($tableName);

        return $content;

    }

    protected function checkTableLogStatus($rel)
    {
        $class = sfInflector::classify($rel['table']->getTableName());
        return in_array(ucfirst($class), $this->logModelArray) && ($rel->getType() == Doctrine_Relation::ONE);
    }

    private function getColumnsNames($tableName)
    {
        $columns = Doctrine_Core::getTable($tableName)->getColumns();
        $returnArray = array();
        foreach ($columns as $columnName => $column)
            $returnArray[] = $columnName;

        return $returnArray;
    }

    private function printValues()
    {
        return sprintf(<<<EOF
        `user_id`,
        `date`,
        `flag`
        )VALUES (\n
EOF
        );
    }


    protected function dropConstraints($tableName, $fkName)
    {
        $dropSql = sprintf(<<<EOF
ALTER TABLE log_%1\$s DROP FOREIGN KEY old_fk_%2\$s_%1\$s;         
ALTER TABLE log_%1\$s DROP FOREIGN KEY new_fk_%2\$s_%1\$s;\n        
EOF
            , $tableName, $fkName);
        $file = sfConfig::get('sf_data_dir') . '/sql-log/dropConstraints.sql';
        $handle = fopen($file, 'a+');
        fwrite($handle, $dropSql);
        fclose($handle);
    }

    protected function dropSfGuardConstraints($tableName)
    {
        $dropSql = sprintf(<<<EOF
ALTER TABLE log_%1\$s DROP FOREIGN KEY fk_log_%1\$s_sfGuard;\n
EOF
            , $tableName);
        $file = sfConfig::get('sf_data_dir') . '/sql-log/dropConstraints.sql';
        $handle = fopen($file, 'a+');
        fwrite($handle, $dropSql);
        fclose($handle);
    }

    private function printTriggerFooter($flag)
    {
        return sprintf(<<<EOF
 _id,
 NOW(),
 %s);
  END
//
DELIMITER ;\n
EOF
            , $flag);
    }

    private function prepareData()
    {
        $dataArray = array();
        $i = 0;

        foreach ($this->logModelArray as $table) {
            $baseTable = Doctrine_Core::getTable($table);
            $logTable = Doctrine_Core::getTable('Log' . $table);
            $dataArray[$i]['name'] = $table;
            $dataArray[$i]['log_id_type'] = 'integer(8)';

            foreach ($logTable->getColumns() as $columnName => $column) {
                if (artString::startsWith('old_', $columnName))
                    $dataArray[$i]['old_log_fields'][] = $columnName;

                if (artString::startsWith('new_', $columnName))
                    $dataArray[$i]['new_log_fields'][] = $columnName;

                if (!artString::startsWith('old_', $columnName) && !artString::startsWith('new_', $columnName) && $columnName != 'id') {
                    if (artString::startsWith('translation_', $columnName))
                        $dataArray[$i]['translation_pks'][] = $columnName;
                    else
                        $dataArray[$i]['special_log_fields'][] = $columnName;

                }
            }

            $dataArray[$i]['primary_keys'] = $this->getPrimaryKeys($baseTable);

            if ($baseTable->hasTemplate('Doctrine_Template_I18n')) {
                $dataArray[$i]['translation_fields'] = $this->geti18nColumns($baseTable);
                $dataArray[$i]['translation_fields'][] = 'lang';
                $dataArray[$i]['is_translation'] = true;
            }
            $this->getRelations($dataArray[$i]['Relations'], $baseTable, $logTable);
            $i++;
        }
        return $dataArray;
    }

    private function getRelations(&$dataArray, $baseTable, $logTable)
    {
        $baseRelations = $baseTable->getRelations();
        $baseTableName = $baseTable->getTableName();
        $logTableName = $logTable->getTableName();
        $logRelations = $logTable->getRelations();

        foreach ($logRelations as $relation) {
            $localKey = $relation->getLocal();

            if (($localKey != 'user_id') && $relation['owningSide']) {
                if ($logTableName == $relation['table']->getTableName())
                    $dataArray[$localKey]['type'] = 2;
                else if (artString::startsWith('log_', ($relation['table']->getTableName())))
                    $dataArray[$localKey]['type'] = 1;

                $dataArray[$localKey]['primary_keys'] = $this->getPrimaryKeys($relation['table']);
                $dataArray[$localKey]['relation_log_table'] = $relation['table']->getTableName();
            }
        }
    }

    private function geti18nColumns($table)
    {
        return $table->getTemplate('Doctrine_Template_I18n')->getOption('fields');
    }

    private function getPrimaryKeys($table)
    {
        $pk = $table->getIdentifier();

        if (is_array($pk))
            return $pk;

        return array($pk);
    }

    private function createTriggers($dataArray)
    {
        foreach ($dataArray as $table) {
            if (isset($table['is_translation'])) {
                $trigger =
                    $this->createTrigger($table, 'insert', 3) .
                        $this->createTrigger($table, 'update', 1) .
                        $this->createTrigger($table, 'delete', 2) .
                        $this->createTranslationTrigger($table, 'insert', 0) .
                        $this->createTranslationTrigger($table, 'update', 1) .
                        $this->createTranslationTrigger($table, 'delete', 2);
            } else {
                $trigger =
                    $this->createTrigger($table, 'insert', 0) .
                        $this->createTrigger($table, 'update', 1) .
                        $this->createTrigger($table, 'delete', 2);
            }
            $this->saveToFile($trigger, $table['name']);
        }
    }

    private function createTrigger($table, $flagName, $flag)
    {
        $trigger = $this->printTriggerHeader($flagName, $table);
        $values = null;

        $this->createPoolsAndValues($table, $flagName, $trigger, $values);

        foreach ($table['special_log_fields'] as $field)
            $trigger .= '`' . $field . "`,";
        $trigger = substr($trigger, 0, -1);
        $trigger .= ")\nVALUES (\n" . $values;


        return $trigger . $this->printTriggerFooter($flag);

    }

    private function createTranslationTrigger($table, $flagName, $flag)
    {
        $trigger = $this->printTriggerHeader($flagName, $table, true);
        $values = null;


        $this->createPoolsAndValuesForTranslation($table, $flagName, $trigger, $values);

        foreach ($table['special_log_fields'] as $field)
            $trigger .= '`' . $field . "`,";
        $trigger = substr($trigger, 0, -1);
        $trigger .= ")\nVALUES (\n" . $values;
        return $trigger . $this->printTriggerFooter($flag);
    }

    private function createPoolsAndValues($table, $flagName, &$trigger, &$values)
    {
        if ($flagName == 'update') {
            $this->createPoolsAndValues($table, 'delete', $trigger, $values);
            $this->createPoolsAndValues($table, 'insert', $trigger, $values);
        } else {
            if ($flagName == 'insert') {
                $prefix = 'new';
            }

            if ($flagName == 'delete')
                $prefix = 'old';

            $i = 0;
            foreach ($table[$prefix . '_log_fields'] as $column) {
                if (isset($table['translation_fields']) && in_array(((substr($column, 4))), $table['translation_fields']))
                    continue;

                $trigger .= '`' . $column . "`,\n";

                if (isset($table['Relations'][$column]['type'])) {
                    if ($table['Relations'][$column]['type'] == 1)
                        $values .= $this->getSql($table['Relations'][$column], $prefix, $column);

                    if ($table['Relations'][$column]['type'] == 2) {
                        $values .= "query_" . ++$i . $flagName . ",\n";
                        $sql = $this->getSqlWithDeclare($table['Relations'][$column], $prefix, $column, $i . $flagName);
                        $trigger = str_replace('#PLACE_FOR_VARIABLE', $sql, $trigger);
                    }

                } else
                    $values .= $this->getTriggerPoolName($prefix, $column) . "`,\n";
            }
        }

    }

    private function createPoolsAndValuesForTranslation($table, $flagName, &$trigger, &$values)
    {

        if ($flagName == 'update') {
            $this->createPoolsAndValuesForTranslation($table, 'delete', $trigger, $values);
            $this->createPoolsAndValuesForTranslation($table, 'insert', $trigger, $values);
        } else {

            if ($flagName == 'insert') {
                $prefix = 'new';
                $trigger .= "`" . $table['translation_pks'][1] . "`,\n";
                $values .= "NEW.`" . $table['primary_keys'][0] . "`,\n";
            }

            if ($flagName == 'delete') {
                $prefix = 'old';
                $trigger .= "`" . $table['translation_pks'][0] . "`,\n";
                $values .= "OLD.`" . $table['primary_keys'][0] . "`,\n";
            }

            $i = 0;
            foreach ($table[$prefix . '_log_fields'] as $column) {

                if (!in_array(((substr($column, 4))), $table['translation_fields'])) {
                    if (isset($table['Relations'][$column]['type'])) {
                        if ($table['Relations'][$column]['type'] == 1)
                            $values .= $this->getSql($table['Relations'][$column], $prefix, $column);

                        if ($table['Relations'][$column]['type'] == 2) {
                            $values .= "query_" . ++$i . $flagName . ",\n";
                            $sql = $this->getSqlDeclareForTranslation($table, $prefix, $column, $i . $flagName);
                            $trigger = str_replace('#PLACE_FOR_VARIABLE', $sql, $trigger);
                        }

                    } else {
                        $values .= "(SELECT `" . $this->preparePoolNameForSelect($prefix, $column) . "` FROM `" . sfInflector::tableize($table['name']) .
                            "` WHERE `" . $table['primary_keys'][0] . '` = ' . strtoupper($prefix) . '.`' .
                            $table['primary_keys'][0] . "` LIMIT 1),\n";
                    }
                } else {
                    $values .= $this->getTriggerPoolName($prefix, $column) . "`,\n";
                }
                $trigger .= '`' . $column . "`,\n";
            }
        }
    }

    private function getSqlWithDeclare($relations, $prefix, $column, $iter)
    {
        return "DECLARE query_" . $iter . " BIGINT(20);
            #PLACE_FOR_VARIABLE 
             SELECT `id` INTO query_" . $iter . " FROM `" . $relations['relation_log_table'] .
            "` WHERE " . $this->getWhereCondition($relations['primary_keys'], $prefix, $column) .
            " AND `flag` <> 3 ORDER BY `date` DESC  LIMIT 1;\n";
    }

    private function getSql($relations, $prefix, $column)
    {
        return '(SELECT id FROM `' . $relations['relation_log_table'] .
            '` WHERE ' . $this->getWhereCondition($relations['primary_keys'], $prefix, $column) .
            " AND `flag` <> 3 ORDER BY `date` DESC  LIMIT 1),\n";
    }


    private function preparePoolNameForSelect($prefix, $columnName)
    {
        if (artString::startsWith($prefix . '_log_', $columnName))
            return substr($columnName, 8);

        if (artString::startsWith($prefix, $columnName))
            return substr($columnName, 4);
    }

    private function preparePoolName($prefix, $columnName)
    {
        if (artString::startsWith($prefix . '_log_', $columnName)) {
            $columnName = substr($columnName, 8);
            $columnName = $prefix . '_' . $columnName;
        }
        return $columnName;
    }

    private function getTriggerPoolName($prefix, $columnName)
    {
        $columnName = $this->preparePoolName($prefix, $columnName);
        return str_replace($prefix . '_', strtoupper($prefix) . '.`', $columnName);
    }

    private function getWhereCondition($primaryKeys, $prefix, $column)
    {
        $where = null;
        $i = 0;
        foreach ($primaryKeys as $primary) {
            $where .= '`new_' . $primary . '` = ' . strtoupper($prefix) . '.`' . str_replace($prefix . '_log_', '', $column) . '`';
            if ($i > 0)
                $where .= ' AND ';
            $i++;
        }

        return $where;
    }

    private function getSqlDeclareForTranslation($table, $prefix, $column, $iter)
    {
        return "DECLARE query_" . $iter . " BIGINT(20);
            #PLACE_FOR_VARIABLE 
             SELECT `id` INTO query_" . $iter . " FROM `" . $table['Relations'][$column]['relation_log_table'] .
            "` WHERE `new_" . $table['Relations'][$column]['primary_keys'][0] . '` = (SELECT `'
            . str_replace($prefix . '_log_', '', $column) . '` FROM `' . sfInflector::tableize($table['name']) .
            "` WHERE `" . $table['Relations'][$column]['primary_keys'][0] . "` = " . strtoupper($prefix) . '.`' . $table['Relations'][$column]['primary_keys'][0] . '`)' .
            " ORDER BY `date` DESC  LIMIT 1;\n";
    }

    private function printTriggerHeader($flagName, $table, $trans = false)
    {
        $tableName = $table['name'];

        if ($trans)
            $tableName = $table['name'] . '_translation';

        if ($flagName == 'delete')
            $triggerString = 'BEFORE DELETE';

        else
            $triggerString = 'AFTER ' . strtoupper($flagName);

        return sprintf(<<<EOF
DROP TRIGGER IF EXISTS `%1\$s_%2\$s`;
DELIMITER //
CREATE TRIGGER `%1\$s_%2\$s` %4\$s ON `%s`
  FOR EACH ROW BEGIN
    DECLARE _id BIGINT(20);
    #PLACE_FOR_VARIABLE
    CREATE TEMPORARY TABLE IF NOT EXISTS logged_in (id bigint(20));
    SELECT id FROM logged_in INTO _id;
    IF _id IS NULL THEN
      SET _id = NULL;
    END IF;
    INSERT INTO `log_%3\$s` (\t
      
EOF
            , sfInflector::tableize($tableName), $flagName, sfInflector::tableize($table['name']), $triggerString);
    }

    private function saveToFile($content, $fName)
    {
        $dir = sfConfig::get('sf_data_dir') . '/sql-log/';
        $fileName = 'log' . $fName . '_triggers.sql';
        $handle = fopen($dir . $fileName, 'w+');
        fwrite($handle, $content);
        fclose($handle);
    }
}
