<?php
/**
 * sfJSDriverBase.
 *
 * @package     sfJSPlugin
 * @subpackage  driver
 */

abstract class sfJSDriverBase
{
    protected
        $options = array(),
        $optimizedContent = null,
        $optimizedSize = null,
        $originalSize = null,
        $processed = false;

    /**
     * Konstruktor.
     *
     * @param array $options
     */
    public function __construct(array $options = array())
    {
        $this->options = $options;
    }

    /**
     * @param $file
     * @param bool $replace
     * @return mixed
     */
    abstract public function doProcessFile($file, $replace = false);

    /**
     * Zwraca stopień konwersji.
     *
     * @return float|null
     */
    public function getOptimizationRatio()
    {
        return 0 !== $this->originalSize ? round($this->optimizedSize * 100 / $this->originalSize, 2) : null;
    }

    /**
     * Zwraca wynik optymalizacji.
     *
     * @return array
     * @throws LogicException
     */
    public function getResults()
    {
        if (!$this->processed) {
            throw new LogicException('Optimizacja nie została przeprowadzona');
        }

        return array(
            'optimizedContent' => $this->optimizedContent,
            'originalSize' => $this->originalSize,
            'optimizedSize' => $this->optimizedSize,
            'ratio' => $this->getOptimizationRatio(),
        );
    }

    /**
     * Tworzy nowy plik.
     *
     * @param $file
     * @param bool $replace
     * @return sfJSDriverBase
     */
    public function processFile($file, $replace = false)
    {
        $this->originalSize = filesize($file);

        $result = $this->doProcessFile($file, $replace);

        if ($replace) {
            clearstatcache();

            $this->optimizedSize = filesize($result);
        } else {
            $this->optimizedSize = strlen($result);

            $this->optimizedContent = $result;
        }

        $this->processed = true;

        return $this;
    }

    /**
     * Podmienia pliki.
     *
     * @param $file
     * @param $content
     * @return mixed
     * @throws RuntimeException
     */
    protected function replaceFile($file, $content)
    {
        if (!file_put_contents($file, $content)) {
            throw new RuntimeException(sprintf('Unable to replace file "%s" with optimized contents', $file));
        }

        return $file;
    }

    /**
     * Czyści instancje
     *
     * @return sfJSDriverBase
     */
    public function reset()
    {
        $this->optimizedContent = null;
        $this->optimizedSize = null;
        $this->originalSize = null;
        $this->processed = false;

        return $this;
    }
}
