<?php

/**
 * Klasa bazowa optymalizera.
 *
 * @package     sjJSPlugin.
 * @subpackage  optimizer
 */
abstract class sfJSOptimizerBase {

  protected
          $assetFiles = array(),
          $baseAssetsDir = null,
          $driver = null,
          $driverName = null,
          $dispatcher = null,
          $files = array(),
          $replaceFiles = false;

  /**
   * Konstruktor.
   *
   * @param  sfEventDispatcher
   * @param  array
   * @param  string|null
   */
  public function __construct(sfEventDispatcher $dispatcher, array $configuration, $baseAssetsDir) {
    $this->baseAssetsDir = $baseAssetsDir;

    $this->dispatcher = $dispatcher;

    $this->configure($configuration);

    $this->dispatcher->notify(new sfEvent($this, 'optimizer.post_configure', array(
        'configuration' => $configuration,
    )));
  }

  /**
   * 
   * @return type
   */
  public function getBaseAssetsDir() {
    return $this->baseAssetsDir;
  }

  /**
   * Zwraca ściezki do plików (np. "main", "main.js", "/css/main.css")
   *
   * @param  string  $file
   */
  abstract public function getAssetFilepath($file);

  /**
   * Zwraca listę plików do procesu optymalizacji.
   *
   * @return array
   */
  public function getAssetFiles() {
    return $this->assetFiles;
  }

  /**
   * Zwraca akbolutną sciezkę do assetów.
   *
   * @param  string  $asset
   * @param  string  $extension
   * @param  string  $webPath
   *
   * @return string|null
   */
  public function computeAssetFilepath($asset, $extension, $webPath = '/') {
    if (preg_match('/^http[s]?:/i', $asset)) {

      return null;
    }

    $webPath = !preg_match('#^/#', $asset) ? sprintf('%s/', $webPath) : '';

    $fileName = preg_match(sprintf('/\.%s$/i', $extension), $asset) ? $asset : sprintf('%s.%s', $asset, $extension);

    return sprintf('%s%s%s', $this->baseAssetsDir, $webPath, $fileName);
  }

  /**
   * Konfiguruje optymalizer.
   *
   * @param  array  $configuration
   */
  public function configure(array $configuration = array()) {
    if (!isset($configuration['driver'])) {
      throw new sfConfigurationException('Nie zdefiniowano nazwy drivera do optymalizacji');
    }

    $this->driverName = $configuration['driver'];

    $driverClass = sprintf('sfJSDriver%s', $this->driverName);

    if (!class_exists($driverClass, true) || !in_array('sfJSDriverBase', class_parents($driverClass))) {
      throw new sfConfigurationException(sprintf('Klasa "%s" nie istnieje lub nie dziedziczy po sfJSDriverBase', $driverClass));
    }

    $this->driver = new $driverClass(isset($configuration['driverOptions']) ? $configuration['driverOptions'] : array());
  }

  /**
   * Zwraca obj.
   *
   * @return obj.
   */
  public function getDriver() {
    return $this->driver;
  }

  /**
   * Zwraca aktualnie używany driver.
   *
   * @return string
   */
  public function getDriverName() {
    return $this->driverName;
  }

  /**
   * Zwraca listę plików.
   *
   * @return array
   */
  public function getFiles() {
    return $this->files;
  }

  /**
   * Optymalizuje wszystkie pliki.
   *
   * @return  array
   * @throws RuntimeException
   */
  public function optimize() {
    if (!count($this->files)) {
      throw new RuntimeException(sprintf('Brak plików do optymalizacji'));
    }

    $results = array();

    foreach ($this->files as $file) {
      $results[$file] = $this->optimizeFile($file);
    }

    return array('statistics' => $results);
  }

  /**
   * Optymalizuje pojedyńczy plik.
   *
   * @param  string   $file
   * @return array
   */
  public function optimizeFile($file) {
    return $this->getDriver($this->driver)
                    ->reset()
                    ->processFile($file, $this->replaceFiles)
                    ->getResults();
  }

  /**
   * Dodaje pliki do procesu optymaliza..
   *
   * @param  array  $files
   * @throws RuntimeException
   */
  public function setFiles(array $files = array()) {
    $this->assetFiles = $files;

    foreach ($files as $i => $file) {

      if (!file_exists($file) && !file_exists($files[$i] = $this->getAssetFilepath($file))) {
        throw new RuntimeException(sprintf('Plik "%s" nie istnieje, ("%s")', $file, $files[$i]));
      }
    }

    $this->files = $files;
  }

}
