<?php

/**
 * Optymalizacja wielu plików JS.
 *
 * @package     sfJSPlugin
 * @subpackage  optimizer
 */
abstract class sfJSOptimizerCombinableBase extends sfJSOptimizerBase {

  protected
          $destination = null,
          $timestamp = false;

  /**
   * @see sfJSOptimizerBase
   */
  public function configure(array $configuration = array()) {
    parent::configure($configuration);

    if (isset($configuration['files'])) {
      parent::setFiles($configuration['files']);
    }

    if (!isset($configuration['destination'])) {
      throw new sfConfigurationException('Musisz zdefiniować opcję "destination" w pliku app.yml');
    }

    $this->destination = $configuration['destination'];
    $this->destinationBase = $configuration['destination'];

    if (isset($configuration['timestamp'])) {
      $this->timestamp = $configuration['timestamp'];
    }

    $this->replaceFiles = false;
  }

  /**
   * Zwraca dostępne rozszeżenia assetów. 
   */
  public function availableFileExtensions() {

    return array('.js', '.css');
  }

  
  public function getDestinationBase(){
    
    return $this->destinationBase; 
  }
  
  
  /**
   * Zwraca ścieżkę absolutną do plików.
   *
   * @return string
   */
  public function getOptimizedFileSystemPath() {
    return realpath(sprintf('%s/%s', $this->baseAssetsDir, $this->destination));
  }

  /**
   * Zwraca ścieżkę (przez web) do plików.
   *
   * @return string
   */
  public function getOptimizedFileWebPath() {
    if (true === $this->timestamp) {
      return $this->generateTimestampedAssetName();
    } else {
      return $this->destination;
    }
  }

  /**
   * Zwraca ścieżkę do pliku (web), wraz ze znacznikiem timestample.
   *
   * @return string
   */
  public function generateTimestampedAssetName() {
    if (file_exists($filePath = $this->getOptimizedFileSystemPath())) {

      $date = date("mdyGis", filemtime($filePath));
      $exts = $this->availableFileExtensions();

      foreach ($exts as $ext) {
        $this->destination = str_replace($ext, $date . $ext, $this->destination);
      }

      return sprintf('%s', $this->destination);
    }

    return $this->destination;
  }

  /**
   * Łączy wyniki z wielu plików do jendego.
   *
   * @return  array
   * @throws RuntimeException
   */
  public function optimize() {
    $results = parent::optimize();

    $optimizedContents = '';

    foreach ($results['statistics'] as $file => $statistic) {
      $optimizedContents .= $statistic['optimizedContent'];
    }

    if (empty($optimizedContents)) {
      throw new RuntimeException('Nie zoptymalizowano żadnych plików!');
    }

    if (!file_put_contents($optimizedFile = sprintf('%s%s', $this->baseAssetsDir, $this->destination), $optimizedContents)) {
      throw new RuntimeException(sprintf('Nie można zapisać zoptymalizowanych plików "%s", (Sprawdź prawa ?).', $optimizedFile));
    }

    $date = date("mdyGis", filemtime($optimizedFile));
    $exts = $this->availableFileExtensions();
    $destinationFile = $optimizedFile; 

    foreach ($exts as $ext) {
      $destinationFile = str_replace($ext, $date . $ext, (string) $destinationFile);
    }

    copy($optimizedFile, $destinationFile);

    return array_merge($results, array('generatedFile' => $destinationFile));
  }

}
