<?php
/**
 * sfAssetsOptimizerService - optymalizacja assetów.
 *
 * @package     sfJSPLugin.
 * @subpackage  service
 */
class sfAssetsOptimizerService
{
    /**
     * Konfiguracja.
     *
     * @var mixed|null|string
     */
    protected
        $baseAssetsDir = null,
        $configuration = array(
        'png_image' => array(
            'enabled' => false,
            'class' => 'sfJSOptimizerPngImage',
            'params' => array(
                'driver' => 'Pngout',
            ),
        ),
        'javascript' => array(
            'enabled' => false,
            'class' => 'sfJSOptimizerJavascript',
            'params' => array(
                'driver' => 'JSMin',
                'files' => array(),
                'destination' => '/js/optimized/optimized.js',
            ),
        ),
        'stylesheet' => array(
            'enabled' => false,
            'class' => 'sfJSOptimizerStylesheet',
            'params' => array(
                'driver' => 'Cssmin',
                'files' => array(),
                'destination' => '/css/optimized/optimized.css',
            ),
        ),
    ),
        $dispatcher = null;

    /**
     * Konstruktor.
     *
     * @param  sfEventDispatcher  $dispatcher
     * @param  array              $configuration
     * @param  string|null        $baseAssetsDir
     */
    public function __construct(sfEventDispatcher $dispatcher, array $configuration = array(), $baseAssetsDir = null)
    {
        $this->dispatcher = $dispatcher;

        $this->configuration = sfToolkit::arrayDeepMerge($this->configuration, $configuration);

        $this->baseAssetsDir = null !== $baseAssetsDir ? $baseAssetsDir : sfConfig::get('sf_web_dir');
    }

    /**
     * 
     * @return type
     */
    public function getEnabledOptimizers(){
      
      $optimizers = array(); 
      
      foreach ($this->configuration as $type => $data) {
        
        if($data['enabled']){
          $optimizers[] = $type; 
        }
      }
      
      return $optimizers;
    }
    
    /**
     * Optymalizacja javascriptów.
     *
     * @return array
     */
    public function optimizeJavascripts()
    {
        if (true !== $this->configuration['javascript']['enabled']) {
            return array();
        }

        return $this->getOptimizer('javascript')->optimize();
    }

    /**
     * Optymalizacja plików PNG.
     *
     * @return array
     */
    public function optimizePngImages()
    {
        if (true !== $this->configuration['png_image']['enabled']) {
            return array();
        }

        return $this->getOptimizer('png_image')->optimize();
    }

    /**
     * Optymalizacja CSS
     *
     * @return array
     */
    public function optimizeStylesheets()
    {
        if (true !== $this->configuration['stylesheet']['enabled']) {
            return array();
        }

        return $this->getOptimizer('stylesheet')->optimize();
    }

    /**
     * Zamienia oryginale pliki JS na zoptymalizowaną wersję.
     *
     * @param  sfWebresponse  $response
     */
    public function replaceJavascripts(sfWebResponse $response)
    {
        if (is_null($javascriptOptimizer = $this->getOptimizer('javascript'))) {
            return;
        }

        foreach ($javascriptOptimizer->getAssetFiles() as $file) {
            $response->removeJavascript($file);
        }

        $response->addJavascript($javascriptOptimizer->getOptimizedFileWebPath()
                , sfConfig::get('app_sf_assets_plugin_configuration_javascript_params_position', ''));
    }

    /**
     * Zamienia oryginale pliki CSS na zoptymalizowaną wersję.
     *
     * @param  sfWebresponse  $response
     */
    public function replaceStylesheets(sfWebResponse $response)
    {
        if (is_null($stylesheetOptimizer = $this->getOptimizer('stylesheet'))) {
            return;
        }

        foreach ($stylesheetOptimizer->getAssetFiles() as $file) {
            $response->removeStylesheet($file);
        }

        $response->addStylesheet($stylesheetOptimizer->getOptimizedFileWebPath()
                , sfConfig::get('app_sf_assets_plugin_configuration_stylesheet_params_position', ''));
    }

    /**
     * Tworzy instancję obiektu.
     *
     * @param  string  $type  The optimizer type
     * @return object.
     * @throws sfConfigurationException
     */
    public function getOptimizer($type)
    {
        if (!in_array($type, $supported = array_keys($this->configuration))) {
            throw new sfConfigurationException(sprintf('Typ "%s" nie jest wspierany. Dostępne typy: %s', $type, implode(', ', $supported)));
        }

        if (false === $this->configuration[$type]['enabled']) {
            return null;
        }

        $className = $this->configuration[$type]['class'];

        if (!class_exists($className, true) || !in_array('sfJSOptimizerBase', class_parents($className))) {
            throw new sfConfigurationException(sprintf('Klasa "%s" nie istnieje lub nie dziedziczy po klasie bazowej sfJSOptimizerBase.', $className));
        }

        return new $className($this->dispatcher, $this->configuration[$type]['params'], $this->baseAssetsDir);
    }
}