<?php

/**
 * Task przeprowadzający optymalizację Assetów.
 *
 * @package     sfJSPLugin.
 * @subpackage  task
 */
class sfJSOptimizeAssetsTask extends sfBaseTask {

    static protected $types = array('all', 'javascript', 'png_image', 'stylesheet');
    protected $optimizer = null;

    /**
     * @see sfTask
     */
    public function configure() {

        $this->addArguments(array(
            new sfCommandArgument('application', sfCommandArgument::REQUIRED, 'The application name'),
        ));

        $this->addOptions(array(
            new sfCommandOption('env', null, sfCommandOption::PARAMETER_REQUIRED, 'Nazwa środowiska domyślnie (dev)', 'dev'),
            new sfCommandOption('type', null, sfCommandOption::PARAMETER_REQUIRED, sprintf('Nazwa assetów do optymalizacji, domyślnie (all)', implode(', ', self::$types)), $default = 'all'),
        ));

        $this->namespace = 'optimize';
        $this->name = 'assets';
        $this->aliases = array("oa");
        $this->briefDescription = 'Optymalizuje pliki JS, CSS oraz PNG.';
    }

    /**
     * @see sfTask
     */
    public function execute($arguments = array(), $options = array()) {

        if ($this->configuration instanceof sfApplicationConfiguration) {
            sfContext::createInstance($this->configuration);
        }
        
        $serviceConfiguration = sfConfig::get('app_sf_assets_plugin_configuration', array());

        if (sfContext::hasInstance()) {
            $viewArray = sfViewConfigHandler::getConfiguration(array(sfConfig::get('sf_app_dir') . '/config/view.yml'));

            $serviceConfiguration['javascript']['params']['files'] = $viewArray['all']['optimizer_javascript_files'];
            $serviceConfiguration['stylesheet']['params']['files'] = $viewArray['all']['optimizer_stylesheet_files'];
        }

        $this->optimizerService = new sfAssetsOptimizerService($this->dispatcher, $serviceConfiguration);

        $this->logSection('optimizing', $options['type']);

        if (!in_array($options['type'], self::$types)) {
            throw new Exception(sprintf('Nie wspierany typ optymalizacji "%s". Dostępne opcje: %s', $options['type'], implode(', ', self::$types)));
        }

        if ('all' === $options['type']) {
            foreach (self::$types as $type) {
                if ('all' !== $type) {
                    $this->optimize($type, $arguments, $options);
                }
            }
        } else {
            $this->optimize($options['type'], $arguments, $options);
        }
    }

    /**
     * @param $type
     */
    public function optimize($type, $arguments, $options) {

        if ($type == 'stylesheet' || $type == 'all') {

            if ($this->configuration instanceof sfApplicationConfiguration) {
                sfContext::createInstance($this->configuration);
            }

            $serviceConfiguration = sfConfig::get('app_sf_assets_plugin_configuration', array());

            if (sfContext::hasInstance()) {
//                $this->logSection('less', sprintf('Kompilowanie plików LESS (Proszę czekać...)'));
//                $this->runTask('less:compile', array('application' => $arguments['application']));

                $viewArray = sfViewConfigHandler::getConfiguration(array(sfConfig::get('sf_app_dir') . '/config/view.yml'));

                $serviceConfiguration['javascript']['params']['files'] = $viewArray['all']['optimizer_javascript_files'];
                $serviceConfiguration['stylesheet']['params']['files'] = $viewArray['all']['optimizer_stylesheet_files'];
            }

            if (isset($serviceConfiguration['stylesheet']['sprites']) && $serviceConfiguration['stylesheet']['sprites']) {
                $this->logSection('sprite', sprintf('Generowanie spraitów (Proszę czekać...)'));
                $this->runTask('optimize:generate-sprites', array(), array());
            }
        }

        if (is_null($optimizer = $this->optimizerService->getOptimizer($type))) {
            $this->logSection('pominięto', sprintf('Optymalizacja %s została wyłączona dla środowiska: "%s".', ucfirst($type), $this->configuration->getEnvironment()), null, 'COMMENT');

            return;
        }

        $this->deleteOldFiles($type);
        $this->logSection($type, sprintf('Optymalizacja %s ["%s"] (Proszę czekać...)', $type, $optimizer->getDriverName()));
        $this->logResults($type, $optimizer->optimize());
    }

    public function deleteOldFiles($type) {
      
        $service = $this->optimizerService->getOptimizer($type); 
        $files = array();
        $baseFile = $service->getBaseAssetsDir() . $service->getDestinationBase(); 
        
        if(is_file($baseFile))
        {
          $files[] = $baseFile; 
        } 
        
        $optymizedFile = $service->getBaseAssetsDir() . $service->getOptimizedFileWebPath(); 
        
        if(is_file($optymizedFile))
        {
          $files[] = $optymizedFile;
        } 
        
        if (count($files) > 0){
          $this->logSection('filesystem', 'Trwa usuwanie starych plików '.$type.' (Proszę czekać...)', null, 'INFO');
          $finder = sfFinder::type('file');
          $this->getFilesystem()->remove($files);          
        }
    }

    /**
     * @param $section
     * @param array $results
     * @throws RuntimeException
     */
    protected function logResults($section, array $results = array()) {
        if (!isset($results['statistics'])) {
            throw new RuntimeException('Nie wygenerowano statystyk');
        }

        foreach ($results['statistics'] as $file => $statistic) {
            if (100 === (int) $statistic['ratio']) {
                $this->logSection($section, sprintf('Zoptymalizowano: %s', $this->formatTaskFilePath($file)));
            } else {
                $this->logSection($section, sprintf('Zmniejszono rozmiar pliku %s: %s', $statistic['ratio'] . '%', $this->formatTaskFilePath($file)));
            }
        }

        if (isset($results['generatedFile'])) {
            $generated = $this->formatTaskFilePath($results['generatedFile']);

            $this->logSection($section, sprintf('Utworzono plik %s (%db.)', $generated, filesize($results['generatedFile'])));
        } else {
            $this->logSection($section, 'Brak opcji optymalizacji');
        }
    }

    /**
     * @param $assetPath
     * @return mixed
     */
    protected function formatTaskFilePath($assetPath) {
        return str_replace(sfConfig::get('sf_web_dir'), 'web', $assetPath);
    }

}
