<?php

/**
 * Cupcake - Categories Controller
 *
 * @author 		Miles Johnson - www.milesj.me
 * @copyright	Copyright 2006-2009, Miles Johnson, Inc.
 * @license 	http://www.opensource.org/licenses/mit-license.php - Licensed under The MIT License
 * @link		www.milesj.me/resources/script/forum-plugin
 */
class CategoriesController extends ForumAppController {

	/**
	 * Models.
	 *
	 * @access public
	 * @var array
	 */
	public $uses = array('Forum.ForumCategory', 'Forum.CategoryObserve');
	/**
	 * Pagination.
	 *
	 * @access public
	 * @var array
	 */
	public $paginate = array(
		'Topic' => array(
			'order' => 'LastPost.created DESC',
			'conditions' => array('Topic.type' => 0),
			'contain' => array('User.id', 'User.username', 'LastPost.created', 'LastUser.username', 'Poll.id')
		)
	);

	/**
	 * Redirect.
	 *
	 * @access public
	 */
	public function index() {
		$this->ForumToolbar->goToPage();
	}

	function observe($category_id=null, $create=true) {
		if (!$category_id)
			$this->redirect($this->referer());

		$category = $this->ForumCategory->getCategoryForViewing($category_id, $this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'));

		// Access
		$this->ForumToolbar->verifyAccess(array(
			'exists' => $category,
			'permission' => $category['ForumCategory']['accessRead']
		));

		// @TODO: #translate
		if ($create) {
			if ($this->CategoryObserve->add($category_id)) {
				$this->message('Kategoria została dodana do obserwowanych');
			} else
				$this->alert('Kategoria nie została dodana do obserwowanych');
		} else {
			if ($this->CategoryObserve->remove($category_id)) {
				$this->message('Kategoria została usunięta z obserwowanych');
			} else
				$this->alert('Kategoria nie została usunięta z obserwowanych');
		}

		$this->redirect($this->referer());
	}

	/**
	 * Read a category.
	 *
	 * @access public
	 * @param string $slug
	 */
	public function view() {

		$id = !empty($this->params['id']) ? $this->params['id'] : false;
		if (!$id)
			$this->redirect($this->referer());

		$category = $this->ForumCategory->getCategoryForViewing($id, $this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'));

		// Access
		$this->ForumToolbar->verifyAccess(array(
			'exists' => $category,
			'permission' => $category['ForumCategory']['accessRead']
		));

		// Paginate
		$this->paginate['Topic']['limit'] = $this->ForumToolbar->settings['topics_per_page'];
		$this->paginate['Topic']['conditions']['Topic.forum_category_id'] = $category['ForumCategory']['id'];

		$this->ForumToolbar->pageTitle($category['ForumCategory']['title']);
		$this->set('category', $category);
		$this->set('topics', $this->paginate('Topic'));
		$this->set('stickies', $this->ForumCategory->Topic->getStickiesInForum($category['ForumCategory']['id']));
		$this->set('feedId', $id);

		$this->set('observed', $this->CategoryObserve->check($category['ForumCategory']['id']));
	}

	/**
	 * Moderate a category.
	 *
	 * @access public
	 * @param string $slug
	 */
	public function moderate($id) {

		$category = $this->ForumCategory->get($id, null, array('Parent', 'Forum.title', 'Forum.slug'));

		// Access
		$this->ForumToolbar->verifyAccess(array(
			'exists' => $category,
			'permission' => $category['ForumCategory']['accessRead'],
			'moderate' => $category['ForumCategory']['id']
		));

		// Processing
		if ($this->RequestHandler->isPost()) {
			if (!empty($this->data['Topic']['items'])) {
				$items = $this->data['Topic']['items'];
				$action = $this->data['Topic']['action'];

				foreach ($items as $topic_id) {
					if (is_numeric($topic_id)) {
						$this->ForumCategory->Topic->id = $topic_id;

						if ($action == 'delete') {
							$this->ForumCategory->Topic->destroy($topic_id);
							$this->message(sprintf(__d('forum', 'A total of %d topic(s) have been permanently deleted', true), count($items)));
						} else if ($action == 'close') {
							$this->ForumCategory->Topic->saveField('status', 1);
							$this->message(sprintf(__d('forum', 'A total of %d topic(s) have been locked to the public', true), count($items)));
						} else if ($action == 'open') {
							$this->ForumCategory->Topic->saveField('status', 0);
							$this->message(sprintf(__d('forum', 'A total of %d topic(s) have been re-opened', true), count($items)));
						} else if ($action == 'move') {
							$this->ForumCategory->Topic->saveField('forum_category_id', $this->data['Topic']['move_id']);
							$this->message(sprintf(__d('forum', 'A total of %d topic(s) have been moved to another forum category', true), count($items)));
						}
					}
				}
			}
		}

		// Paginate
		$this->paginate['Topic']['limit'] = $this->ForumToolbar->settings['topics_per_page'];
		$this->paginate['Topic']['conditions'] = array('Topic.forum_category_id' => $category['ForumCategory']['id']);

		$this->ForumToolbar->pageTitle(__d('forum', 'Moderate', true), $category['ForumCategory']['title']);
		$this->set('category', $category);
		$this->set('topics', $this->paginate('Topic'));
		$this->set('forums', $this->ForumCategory->getHierarchy($this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'), 'read'));
		$this->set('feedId', $id);
	}

	/**
	 * RSS Feed.
	 *
	 * @access public
	 * @param string $slug
	 */
	public function feed() {
		$id = !empty($this->params['id']) ? $this->params['id'] : false;
		if (!$id)
			$this->redirect($this->referer());

		if ($this->RequestHandler->isRss()) {
			$category = $this->ForumCategory->get($id);
			$this->ForumToolbar->verifyAccess(array('exists' => $category));

			$this->paginate['Topic']['limit'] = $this->ForumToolbar->settings['topics_per_page'];
			$this->paginate['Topic']['conditions'] = array('Topic.forum_category_id' => $category['ForumCategory']['id']);
			$this->paginate['Topic']['contain'] = array('User.id', 'User.username', 'LastPost.created', 'FirstPost.content');

			$this->set('items', $this->paginate('Topic'));
			$this->set('category', $category);
			$this->set('document', array('xmlns:dc' => 'http://purl.org/dc/elements/1.1/'));
		} else {

			$this->redirect('/forum/categories/feed/id:' . $id . '.rss');
		}
	}

	/**
	 * Admin index!
	 *
	 * @access public
	 * @category Admin
	 */
	public function admin_index() {
		if (!empty($this->data)) {
			$this->ForumCategory->updateOrder($this->data);
			$this->message(__d('forum', 'The order of the forums have been updated!', true));
		}

		$this->set('title_for_layout', __d('forum', 'Manage Forums', true));
		$this->set('forums', $this->ForumCategory->Forum->getAdminIndex());
	}

	/**
	 * Add a top level forum.
	 *
	 * @access public
	 * @category Admin
	 */
	public function admin_add_forum() {
		if (!empty($this->data)) {
			if ($this->ForumCategory->Forum->save($this->data, true, array('title', 'status', 'orderNo', 'accessView', 'access_level_id'))) {
				$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
			}
		}

		$this->set('title_for_layout', __d('forum', 'Add Forum', true));
		$this->set('method', 'add');
		$this->set('levels', $this->ForumCategory->AccessLevel->getHigherLevels(null, true));
		$this->render('admin_form_forum');
	}

	/**
	 * Edit top level forum.
	 *
	 * @access public
	 * @category Admin
	 * @param int $id
	 */
	public function admin_edit_forum($id) {
		$forum = $this->ForumCategory->Forum->get($id);
		$this->ForumToolbar->verifyAccess(array('exists' => $forum));

		// Form Processing
		if (!empty($this->data)) {
			$this->ForumCategory->Forum->id = $id;

			if ($this->ForumCategory->Forum->save($this->data, true, array('title', 'status', 'orderNo', 'accessView', 'access_level_id'))) {
				$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
			}
		} else {
			$this->data = $forum;
		}

		$this->set('title_for_layout', __d('forum', 'Edit Forum', true));
		$this->set('id', $id);
		$this->set('method', 'edit');
		$this->set('levels', $this->ForumCategory->AccessLevel->getHigherLevels(null, true));
		$this->render('admin_form_forum');
	}

	/**
	 * Delete a forum.
	 *
	 * @access public
	 * @category Admin
	 * @param int $id
	 */
	public function admin_delete_forum($id) {
		$forum = $this->ForumCategory->Forum->get($id, array('id', 'title'));
		$this->ForumToolbar->verifyAccess(array('exists' => $forum));

		// Form Processing
		if (!empty($this->data['Forum']['forum_id'])) {
			$this->ForumCategory->moveAll($forum['Forum']['id'], $this->data['Forum']['forum_id']);
			$this->ForumCategory->Forum->delete($forum['Forum']['id'], true);

			$this->message(sprintf(__d('forum', 'The forum %s has been deleted, and all its forum categories have been moved!', true), '<strong>' . $forum['Forum']['title'] . '</strong>'));
			$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
		}

		$this->set('title_for_layout', __d('forum', 'Delete Forum', true));
		$this->set('id', $id);
		$this->set('forum', $forum);
		$this->set('levels', $this->ForumCategory->AccessLevel->getHigherLevels(null, true));
		$this->set('forums', $this->ForumCategory->Forum->getList($forum['Forum']['id']));
	}

	/**
	 * Add a forum category.
	 *
	 * @access public
	 * @category Admin
	 */
	public function admin_add_category($forum_id = null) {
		if (!empty($this->data)) {
			if (empty($this->data['ForumCategory']['parent_id'])) {
				$this->data['ForumCategory']['parent_id'] = 0;
			}

			if ($this->ForumCategory->save($this->data, true, array('title', 'description', 'forum_id', 'parent_id', 'status', 'orderNo', 'accessRead', 'accessPost', 'accessReply', 'accessPoll', 'settingPostCount', 'settingAutoLock', 'access_level_id'))) {
				$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
			}
		} elseif ($forum_id) {
			$this->data['ForumCategory']['forum_id'] = $forum_id;
		}

		$this->set('title_for_layout', __d('forum', 'Add Forum Category', true));
		$this->set('levels', $this->ForumCategory->AccessLevel->getHigherLevels(null, true));
		$this->set('forums', $this->ForumCategory->Forum->getList());
		$this->set('categories', $this->ForumCategory->getParents());
		$this->set('method', 'add');
		$this->render('admin_form_category');
	}

	/**
	 * Edit a forum category.
	 *
	 * @access public
	 * @category Admin
	 * @param int $id
	 */
	public function admin_edit_category($id) {
		$category = $this->ForumCategory->get($id);
		$this->ForumToolbar->verifyAccess(array('exists' => $category));

		// Form Processing
		if (!empty($this->data)) {
			$this->ForumCategory->id = $id;

			if (empty($this->data['ForumCategory']['parent_id'])) {
				$this->data['ForumCategory']['parent_id'] = 0;
			}

			if ($this->ForumCategory->save($this->data, true, array('title', 'description', 'forum_id', 'parent_id', 'status', 'orderNo', 'accessRead', 'accessPost', 'accessReply', 'accessPoll', 'settingPostCount', 'settingAutoLock', 'access_level_id'))) {
				$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
			}
		} else {
			$this->data = $category;
		}

		$this->set('title_for_layout', __d('forum', 'Edit Forum Category', true));
		$this->set('id', $id);
		$this->set('levels', $this->ForumCategory->AccessLevel->getHigherLevels(null, true));
		$this->set('forums', $this->ForumCategory->Forum->getList());
		$this->set('categories', $this->ForumCategory->getParents($id));
		$this->set('method', 'edit');
		$this->render('admin_form_category');
	}

	/**
	 * Delete a category.
	 *
	 * @access public
	 * @category Admin
	 * @param int $id
	 */
	public function admin_delete_category($id) {
		$category = $this->ForumCategory->get($id);
		$this->ForumToolbar->verifyAccess(array('exists' => $category));

		// Form Processing
		if (!empty($this->data['ForumCategory']['category_id'])) {
			$this->ForumCategory->Topic->moveAll($category['ForumCategory']['id'], $this->data['ForumCategory']['category_id']);
			$this->ForumCategory->moveAll($category['ForumCategory']['id'], 0, true);
			$this->ForumCategory->delete($category['ForumCategory']['id'], true);

			$this->message(sprintf(__d('forum', 'The forum category %s has been deleted, and all its sub-forums and topics have been moved!', true), '<strong>' . $category['ForumCategory']['title'] . '</strong>'));
			$this->redirect(array('controller' => 'categories', 'action' => 'index', 'admin' => true));
		}

		$this->set('title_for_layout', __d('forum', 'Delete Forum Category', true));
		$this->set('id', $id);
		$this->set('category', $category);
		$this->set('categories', $this->ForumCategory->getHierarchy(0, $this->Session->read('Forum.access'), 'read', $id));
	}

	/**
	 * Before filter.
	 *
	 * @access public
	 * @return void
	 */
	public function beforeFilter() {
		parent::beforeFilter();

		//$this->Auth->allow('index', 'view', 'feed');
		$this->Security->disabledFields = array('items');

		if (isset($this->params['admin'])) {
			$this->ForumToolbar->verifyAdmin();
			$this->layout = 'admin';
			$this->set('menuTab', 'forums');
		} else {
			$this->set('menuTab', '');
		}
	}

}
