<?php

/**
 * Cupcake - Topics Controller
 *
 * @author 		Miles Johnson - www.milesj.me
 * @copyright	Copyright 2006-2009, Miles Johnson, Inc.
 * @license 	http://www.opensource.org/licenses/mit-license.php - Licensed under The MIT License
 * @link		www.milesj.me/resources/script/forum-plugin
 */
class TopicsController extends ForumAppController {

     /**
      * Models.
      *
      * @access public
      * @var array
      */
     public $uses = array('Forum.Topic', 'Forum.TopicObserve');
     /**
      * Pagination.
      *
      * @access public
      * @var array
      */
     public $paginate = array(
	 'Post' => array(
	     'order' => 'Post.created ASC',
	     'contain' => array('User' => array('Access' => array('AccessLevel')), 'ModifiedByUser'=> array('fields'=> array('id', 'username')))
	 )
     );

     function observe($topic_id=null, $create=true) {
	  if (!$topic_id)
	       $this->redirect($this->referer());

	  $topic = $this->Topic->getTopicForViewing($topic_id, $this->Auth->user('id'));

	  // Access
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $topic,
	      'permission' => $topic['ForumCategory']['accessRead']
	  ));
	  // @TODO: #translate
	  if ($create) {
	       if ($this->TopicObserve->add($topic_id)) {
		    $this->message(__d('forum', 'Temat został dodany do obserwowanych', true));
	       } else
		    $this->alert(__d('forum', 'Temat został dodany do obserwowanych', true));
	  } else {
	       if ($this->TopicObserve->remove($topic_id)) {
		    $this->message(__d('forum', 'Temat został usunięty z obserwowanych', true));
	       } else
		    $this->alert(__d('forum', 'Temat nie został usunięty z obserwowanych', true));
	  }

	  $this->redirect($this->referer());
     }

     /**
      * Redirect.
      *
      * @access public
      */
     public function index() {
	  $this->ForumToolbar->goToPage();
     }

     /**
      * Post a new topic or poll.
      *
      * @access public
      * @param int $category_id
      * @param string $type
      */
     public function add($category_id, $type = '') {
	  $category = $this->Topic->ForumCategory->getCategory($category_id);
	  $user_id = $this->Auth->user('id');

	  if ($type == 'poll') {
	       $pageTitle = __d('forum', 'Post Poll', true);
	       $access = 'accessPoll';
	       $isPoll = true;
	  } else {
	       $pageTitle = __d('forum', 'Post Topic', true);
	       $access = 'accessPost';
	       $isPoll = false;
	  }

	  // Access
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $category,
	      'status' => $category['ForumCategory']['status'],
	      'permission' => $category['ForumCategory'][$access]
	  ));

	  // Form Processing
	  if (!empty($this->data)) {
	       $this->data['Topic']['user_id'] = $user_id;
	       $this->data['Topic']['userIP'] = $this->RequestHandler->getClientIp();

	       if ($topic_id = $this->Topic->addTopic($this->data, $this->ForumToolbar->settings, $this->Session->read('Forum.topics'), $isPoll)) {
		    if ($category['ForumCategory']['settingPostCount'] == 1) {
			 $this->Topic->User->increasePosts($user_id);
			 $this->Topic->User->increaseTopics($user_id);
		    }

		    $this->ForumToolbar->updateTopics($topic_id);
		    $this->ForumToolbar->categoryObserveEmail($category_id, $topic_id);
		    $this->ForumToolbar->goToPage($topic_id);
	       } else
		    $this->alert('Nie można dodać tematu!');
	  } else {
	       $this->data['Topic']['forum_category_id'] = $category_id;
	  }

	  $this->ForumToolbar->pageTitle($pageTitle);
	  $this->set('id', $category_id);
	  $this->set('type', $type);
	  $this->set('category', $category);
	  $this->set('pageTitle', $pageTitle);
	  $this->set('forums', $this->Topic->ForumCategory->getHierarchy($this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'), 'post'));
     }

     /**
      * Edit a topic.
      *
      * @access public
      * @param int $id
      */
     public function edit($id) {
	  $topic = $this->Topic->getTopicForEdit($id);
	  $user_id = $this->Auth->user('id');

	  // Access
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $topic,
	      'moderate' => $topic['Topic']['forum_category_id'],
	      'ownership' => $topic['Topic']['user_id']
	  ));

	  // Form Processing
	  if (!empty($this->data)) {
	       $this->data['Post']['modified_by_id'] = $this->Auth->user('id');
	       if ($this->Topic->saveAll($this->data, array('validate' => 'only'))) {
		    if ($this->Topic->editTopic($id, $this->data)) {
			 $this->ForumToolbar->goToPage($id);
		    }
	       }
	  } else {
	       $this->data = $topic;
	  }

	  $this->ForumToolbar->pageTitle(__d('forum', 'Edit Topic', true));
	  $this->set('id', $id);
	  $this->set('topic', $topic);
	  $this->set('forums', $this->Topic->ForumCategory->getHierarchy($this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'), 'post'));
	  $this->set('feedId', $id);
     }

     /**
      * RSS Feed.
      *
      * @access public
      * @param int $id
      */
     public function feed($id) {
	  if ($this->RequestHandler->isRss()) {
	       $topic = $this->Topic->get($id, null, array('FirstPost.content'));
	       $this->ForumToolbar->verifyAccess(array('exists' => $topic));

	       $this->paginate['Post']['limit'] = $this->ForumToolbar->settings['posts_per_page'];
	       $this->paginate['Post']['conditions']['Post.topic_id'] = $id;
	       $this->paginate['Post']['contain'] = array('User');
	       $this->paginate['Post']['order'] = 'Post.created DESC';

	       $this->set('items', $this->paginate('Post'));
	       $this->set('topic', $topic);
	       $this->set('document', array('xmlns:dc' => 'http://purl.org/dc/elements/1.1/'));
	  } else {
	       $this->redirect('/forum/topics/feed/' . $id . '.rss');
	  }
     }

     /**
      * Delete a topic.
      *
      * @access public
      * @param int $id
      */
     public function delete($id) {
	  $topic = $this->Topic->get($id, array('id', 'forum_category_id', 'user_id'));

	  // Access
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $topic,
	      'moderate' => $topic['Topic']['forum_category_id']
	  ));

	  // Delete All
	  $this->Topic->destroy($id);
	  $category = $this->Topic->ForumCategory->get($topic['Topic']['forum_category_id']);
	  $categoryTitle = $category['ForumCategory']['title'];

	  if ($category['ForumCategory']['settingPostCount'] == 1) {
	       $this->Topic->User->decreasePosts($topic['Topic']['user_id']);
	  }

	  $this->alert(
		  __d('forum', "Topic has been deleted!", true),
		  array('controller' => 'categories', 'action' => 'view', 'id' => $topic['Topic']['forum_category_id'], 'slug' => slug($categoryTitle))
		  );
     }

     /**
      * Report a topic.
      *
      * @access public
      * @param int $id
      */
     public function report($id) {
	  $this->loadModel('Forum.Report');

	  $topic = $this->Topic->get($id, array('id', 'title', 'slug'));
	  $user_id = $this->Auth->user('id');

	  // Access
	  $this->ForumToolbar->verifyAccess(array('exists' => $topic));

	  // Submit Report
	  if (!empty($this->data)) {
	       $this->data['Report']['user_id'] = $user_id;
	       $this->data['Report']['item_id'] = $id;
	       $this->data['Report']['itemType'] = 'topic';

	       if ($this->Report->save($this->data, true, array('item_id', 'itemType', 'user_id', 'comment'))) {
		    $this->message(
			    __d('forum', 'You have succesfully reported this topic! A moderator will review this topic and take the necessary action.', true),
			    array('action' => 'view', 'id' => $topic['Topic']['id'], 'slug' => slug($topic['Topic']['title']))
		    );
	       }else
		    $this->alert(__d('forum', "Can't report topic!", true));
	  }

	  $this->ForumToolbar->pageTitle(__d('forum', 'Report Topic', true));
	  $this->set('id', $id);
	  $this->set('topic', $topic);
	  $this->set('feedId', $id);
     }

     /**
      * Read a topic.
      *
      * @access public
      * @param string $slug
      */
     public function view() {

	  $id = !empty($this->params['id']) ? $this->params['id'] : false;
	  if (!$id)
	       $this->redirect($this->referer());

	  $user_id = $this->Auth->user('id');
	  $topic = $this->Topic->getTopicForViewing($id, $user_id);

	  // Access to category
	  $category = $this->Topic->ForumCategory->getCategoryForViewing($topic['Topic']['forum_category_id'], $this->ForumToolbar->getAccess(), $this->Session->read('Forum.access'));
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $category
	  ));

	  // Access to topic
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $topic,
	      'permission' => $topic['ForumCategory']['accessRead']
	  ));

	  // Update
	  $this->ForumToolbar->markAsRead($topic['Topic']['id']);
	  $this->Topic->increaseViews($topic['Topic']['id']);

	  // Paginate
	  $this->paginate['Post']['limit'] = $this->ForumToolbar->settings['posts_per_page'];
	  $this->paginate['Post']['conditions']['Post.topic_id'] = $topic['Topic']['id'];

	  // Poll Voting
	  if (!empty($this->data['Poll']['option'])) {
	       $this->Topic->Poll->vote($topic['Poll']['id'], $this->data['Poll']['option'], $user_id);
	       $this->redirect(array('plugin' => 'forum', 'controller' => 'topics', 'action' => 'view', 'id' => $id, 'slug' => slug($topic['Topic']['title'])));
	  }

	  $this->ForumToolbar->pageTitle($topic['ForumCategory']['title'], $topic['Topic']['title']);
	  $this->set('topic', $topic);
	  $this->set('posts', $this->paginate('Post'));
	  $this->set('feedId', $topic['Topic']['id']);

	  $this->set('observed', $this->TopicObserve->check($topic['Topic']['id']));
     }

     /**
      * Moderate a topic.
      *
      * @access public
      * @param int $id
      */
     public function moderate($id) {
	  $user_id = $this->Auth->user('id');
	  $topic = $this->Topic->getTopicForViewing($id, $user_id, 'id');

	  // Access
	  $this->ForumToolbar->verifyAccess(array(
	      'exists' => $topic,
	      'permission' => $topic['ForumCategory']['accessRead'],
	      'moderate' => $topic['Topic']['forum_category_id']
	  ));

	  // Processing
	  if ($this->RequestHandler->isPost()) {
	       if (!empty($this->data['Post']['items'])) {
		    $items = $this->data['Post']['items'];
		    $action = $this->data['Post']['action'];

		    foreach ($items as $post_id) {
			 if (is_numeric($post_id)) {
			      if ($action == 'delete') {
				   $this->Topic->Post->destroy($post_id);
				   $this->message(sprintf(__d('forum', 'A total of %d post(s) have been permanently deleted', true), count($items)));
			      }
			 }
		    }
	       }
	  }

	  // Paginate
	  $this->paginate['Post']['limit'] = $this->ForumToolbar->settings['posts_per_page'];
	  $this->paginate['Post']['conditions']['Post.topic_id'] = $id;

	  $this->ForumToolbar->pageTitle(__d('forum', 'Moderate', true), $topic['Topic']['title']);
	  $this->set('id', $id);
	  $this->set('topic', $topic);
	  $this->set('posts', $this->paginate('Post'));
     }

     /**
      * Before filter.
      *
      * @access public
      * @return void
      */
     public function beforeFilter() {
	  parent::beforeFilter();

	  //$this->Auth->allow('index', 'view', 'feed');
	  $this->Security->disabledFields = array('option', 'items');
	  $this->set('menuTab', '');
     }

}
