<?php

/*
  Copyright 2012 Scott Harwell

  Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions and limitations under the License.
 */

App::import('Vendor', 'Stats.gchart');


class GoogleChartHelper extends AppHelper {

	public $helpers = array('Html');
	/**
	 * Path to Google Chart JS Library
	 *
	 * @var string
	 */
	public $googleScriptPath = "https://www.google.com/jsapi";

	/**
	 * Library Loaded
	 *
	 * @var used to determine if Google JS lib has been sent to HTML helper
	 */
	protected $libraryLoaded = false;


	/**
	 * Create Charts
	 *
	 * @param array - nested arrays of charts and data array('chart' => array($data, $keys, $Model, $chartDiv, $otherOptions))
	 * @param String $variableId Allows one to set the js variable name for chart objects to allow reference for updates in other JS files
	 * @return void
	 */
	public function createJsChart($chart, $options = array()) {
		if (get_class($chart) === "GoogleChart") {
			$this->_setupChartJs();
			$this->_buildDataChartJs($chart, $options);
		}
	}

	/**
	 * Setup JS Needed for Charts
	 * 
	 * @return void
	 */
	protected function _setupChartJs() {
		if (!$this->libraryLoaded) {
			echo $this->Html->script(
					array($this->googleScriptPath), array('inline' => false)
			);
			$this->libraryLoaded = true;

			//JS to load
			$js = 'google.load("visualization", "1", {packages:["corechart","annotatedtimeline"]});';

			//create an array of charts to load more than one
			$js .= "var charts = new Array();";
			echo $this->Html->scriptBlock($js, array('inline' => false));
		}
	}

	/**
	 * Builds JS for a chart
	 *
	 * @param Google Chart object
	 * @param String $variableId Allows one to set the js variable name for chart objects to allow reference for updates in other JS files
	 * @return void
	 */
	protected function _buildChartJs(GoogleChart $chart, $variableId = null) {
		//get Column keys to match against rows
		$columnKeys = array_keys($chart->columns);

		//Make sure you are using jQuery
		$scriptOutput = "$(document).ready(function(){\n";

		//create a uuid for chart variables in case we have multiples
		$chartOptionsId = !empty($variableId) ? "options_{$variableId}" : uniqid("options_");

		//encode chart options
		$options = json_encode($chart->options);

		$scriptOutput .= "{$chartOptionsId} = chartOptions = {$options};\n";

		//create a uuid for chart variables in case we have multiples
		$chartDataId = !empty($variableId) ? "js_{$variableId}" : uniqid("js_");

		$scriptOutput .= "{$chartDataId} = chartData = new google.visualization.arrayToDataTable(";

		$scriptOutput .= "[\n[";

		$keys = array_keys($chart->columns);
		$last_key = end($keys);
		foreach ($chart->columns as $key => $column) {
			$scriptOutput .= "'{$column['label']}'";

			if ($key !== $last_key) {
				$scriptOutput .= ",";
			}
		}

		$scriptOutput .= "],\n";

		$keys = array_keys($chart->rows);
		$last_key = end($keys);
		foreach ($chart->rows as $key => $row) {
			$scriptOutput .= "[";

			$rKeys = array_keys($row);
			$last_row_key = end($rKeys);
			foreach ($row as $rKey => $val) {
				$jsVal = $val;
				if ($chart->columns[$columnKeys[$rKey]]['type'] === "string") {
					$jsVal = "'{$val}'";
				}
//				elseif($chart->columns[$columnKeys[$rKey]]['type'] === "date"){
//					$jsVal = "new Date(".str_replace('-',',',$val).")";
//				}
				$scriptOutput .= $jsVal;

				if ($rKey !== $last_row_key) {
					$scriptOutput .= ",";
				}
			}
			$scriptOutput .= "]";

			if ($key !== $last_key) {
				$scriptOutput .= ",\n";
			}
		}

		$scriptOutput .= "]);";

		$chartVarId = !empty($variableId) ? "chart_{$variableId}" : uniqid("chart_");

		$scriptOutput .= "{$chartVarId} = chart = new google.visualization.{$chart->type}(document.getElementById('{$chart->div}'))";
		$scriptOutput .= ".draw({$chartDataId}, {$chartOptionsId});";

		$scriptOutput .= "});";

		$this->Html->scriptBlock($scriptOutput, array('inline' => false, 'safe' => true));
	}

	/**
	 * Builds JS for a chart using Google data type: DataTable()
	 *
	 * @param Google Chart object
	 * @param String $variableId Allows one to set the js variable name for chart objects to allow reference for updates in other JS files
	 * @return void
	 */
	protected function _buildDataChartJs(GoogleChart $chart, $options = null) {
		//get Column keys to match against rows
		$columnKeys = array_keys($chart->columns);

		//Make sure you are using jQuery
		$scriptOutput = "$(document).ready(function(){\n";

		//create a uuid for chart variables in case we have multiples
		$chartOptionsId = !empty($variableId) ? "options_{$variableId}" : uniqid("options_");

		//encode chart options
		$options = json_encode(array_merge($chart->options, $options));

		$scriptOutput .= "{$chartOptionsId} = chartOptions = {$options};\n";

		//create a uuid for chart variables in case we have multiples
		$chartDataId = !empty($variableId) ? "js_{$variableId}" : uniqid("js_");

		$scriptOutput .= "{$chartDataId} = chartData = new google.visualization.DataTable();\n";

		$keys = array_keys($chart->columns);

		foreach ($chart->columns as $key => $column) {
			$scriptOutput .= "{$chartDataId}.addColumn('";
			$scriptOutput .= $column['type'];
			$scriptOutput .= "','{$column['label']}');\n";
		}

		$keys = array_keys($chart->rows);
		$last_key = end($keys);
		
		
		$scriptOutput .= "{$chartDataId}.addRows([\n";
		foreach ($chart->rows as $key => $row) { // wiersze

			$rKeys = array_keys($row);
			$last_row_key = end($rKeys);
			$scriptOutput .= "[";
			foreach ($row as $rKey => $val) { // komórki

				$jsVal = $val;
				if ($chart->columns[$columnKeys[$rKey]]['type'] === "string") {
					$jsVal = "'{$val}'";
				} elseif ($chart->columns[$columnKeys[$rKey]]['type'] === "date") {
					
					$date = explode('-', $val);
					$date[1] -= 1;
					
					$jsVal = "new Date(" . implode(',',$date) . ")";
				}

				$scriptOutput .= $jsVal;

				if ($rKey !== $last_row_key) {
					$scriptOutput .= ",";
				}
			}
			

			if ($key !== $last_key) {
				$scriptOutput .= "],\n";
			} else {
				$scriptOutput .= "]\n\n";
			}
		}

		$scriptOutput .= "]);";

		$chartVarId = !empty($variableId) ? "chart_{$variableId}" : uniqid("chart_");

		$scriptOutput .= "{$chartVarId} = chart = new google.visualization.{$chart->type}(document.getElementById('{$chart->div}'))";
		$scriptOutput .= ".draw({$chartDataId}, {$chartOptionsId});";

		$scriptOutput .= "});";

		$this->Html->scriptBlock($scriptOutput, array('inline' => false, 'safe' => true));
	}

}
