!function()
{
    var RendererCanvas = function(maze, opts)
    {
        opts = opts || {};
        this.maze = maze;
        this.canvasID = opts.canvasID;
        this.mazeLineWidth = opts.mazeLineWidth;
        this.mazeLineColor = opts.mazeLineColor;

        this.canvas = document.getElementById(this.canvasID);
        this.ctx = this.canvas.getContext("2d");
        this.width = parseInt(this.canvas.getAttribute("width"));
        this.height = parseInt(this.canvas.getAttribute("height"));
        this.blockSize = Math.min(Math.floor(this.width / maze.x), Math.floor(this.height / maze.y));

        var _this = this,
            drawLineVertical = function drawLineVertical(colsNb, rowsNb, row, col)
            {
                var width = parseInt(_this.canvas.getAttribute("width")),
                    height = parseInt(_this.canvas.getAttribute("height")),
                    startY = _this.blockSize * row,
                    endY = _this.blockSize * row,
                    startX = _this.blockSize * col,
                    endX = startX + _this.blockSize,
                    firstOffsetX = 4,
                    firstOffsetY = 4,
                    secondOffsetX = 0,
                    secondOffsetY = 0;

                if(startY === 0)
                {
                    startY = _this.mazeLineWidth / 2;
                    endY = _this.mazeLineWidth / 2;
                    firstOffsetX = 0;
                    firstOffsetY = 0;
                    secondOffsetX = -4;
                    secondOffsetY = -4;
                }
                else
                {
                    if(startY === height)
                    {
                        startY -= _this.mazeLineWidth / 2;
                        endY -= _this.mazeLineWidth / 2;
                    }
                    else
                    {
                        startX -= _this.mazeLineWidth / 2;
                        endX += _this.mazeLineWidth / 2;
                    }
                }

                _this.ctx.lineWidth = _this.mazeLineWidth;
                _this.ctx.strokeStyle = _this.mazeLineColor[0];
                _this.ctx.beginPath();
                _this.ctx.moveTo(startX - firstOffsetX, startY - firstOffsetY);
                _this.ctx.lineTo(endX - firstOffsetX, endY - firstOffsetY);
                _this.ctx.stroke();

                _this.ctx.lineWidth = _this.mazeLineWidth;
                _this.ctx.strokeStyle = _this.mazeLineColor[1];
                _this.ctx.beginPath();
                _this.ctx.moveTo(startX - secondOffsetX, startY - secondOffsetY);
                _this.ctx.lineTo(endX - secondOffsetX, endY - secondOffsetY);
                _this.ctx.stroke();
            },
            drawLineHorizontal = function drawLineHorizontal(colsNb, rowsNb, row, col)
            {
                var width = parseInt(_this.canvas.getAttribute("width")),
                    height = parseInt(_this.canvas.getAttribute("height")),
                    startY = _this.blockSize * row,
                    endY = startY + _this.blockSize,
                    startX = _this.blockSize * col,
                    endX = startX,
                    firstOffsetX = 4,
                    firstOffsetY = 4,
                    secondOffsetX = 0,
                    secondOffsetY = 0;

                if(startX === 0)
                {
                    startX = _this.mazeLineWidth / 2;
                    endX = _this.mazeLineWidth / 2;
                    firstOffsetX = 0;
                    firstOffsetY = 0;
                    secondOffsetX = -4;
                    secondOffsetY = -4;
                }
                else
                {
                    if(startX === height)
                    {
                        startX -= _this.mazeLineWidth / 2;
                        endX -= _this.mazeLineWidth / 2;
                    }
                    else
                    {
                        startY -= _this.mazeLineWidth / 2;
                        endY += _this.mazeLineWidth / 2;
                    }
                }

                _this.ctx.lineWidth = _this.mazeLineWidth;
                _this.ctx.strokeStyle = _this.mazeLineColor[0];
                _this.ctx.beginPath();
                _this.ctx.moveTo(startX - firstOffsetX, startY - firstOffsetY);
                _this.ctx.lineTo(endX - firstOffsetX, endY - firstOffsetY);
                _this.ctx.stroke();

                _this.ctx.lineWidth = _this.mazeLineWidth;
                _this.ctx.strokeStyle = _this.mazeLineColor[1];
                _this.ctx.beginPath();
                _this.ctx.moveTo(startX - secondOffsetX, startY - secondOffsetY);
                _this.ctx.lineTo(endX - secondOffsetX, endY - secondOffsetY);
                _this.ctx.stroke();
            };

        this.render = function()
        {
            var rowNb, colNb;

            this.ctx.clearRect(0, 0, this.width, this.height);

            for(rowNb = 0; rowNb < this.maze.maze.y; rowNb += 1)
            {
                for(colNb = 0; colNb < this.maze.maze.x; colNb += 1)
                {
                    if(!this.maze.maze.horizontal[rowNb][colNb])
                    {
                        if(colNb + 1 !== this.maze.maze.x) // skip border
                        {
                            drawLineHorizontal(this.maze.maze.x, this.maze.maze.y, rowNb, colNb + 1);
                        }
                    }
                    if(!this.maze.maze.vertical[rowNb][colNb])
                    {
                        if(rowNb + 1 !== this.maze.maze.y) // skip border
                        {
                            drawLineVertical(this.maze.maze.x, this.maze.maze.y, rowNb + 1, colNb);
                        }
                    }

                    // render border
                    if(rowNb === 0 && colNb > 0)
                    {
                        drawLineVertical(this.maze.maze.x, this.maze.maze.y, rowNb, colNb);
                    }
                    if(rowNb + 1 === this.maze.maze.y)
                    {
                        drawLineVertical(this.maze.maze.x, this.maze.maze.y, rowNb + 1, colNb);
                    }
                    if(colNb === 0)
                    {
                        drawLineHorizontal(this.maze.maze.x, this.maze.maze.y, rowNb, colNb);
                    }
                    if(colNb + 1 === this.maze.maze.x && rowNb + 1 !== this.maze.maze.y)
                    {
                        drawLineHorizontal(this.maze.maze.x, this.maze.maze.y, rowNb, colNb + 1);
                    }
                }
            }
        }
    };

    var Maze = function(x, y)
    {
        this.x = x;
        this.y = y;

        this.generate = function()
        {
            var n = x * y - 1,
                j, k,
                horizontal = [],
                vertical = [],
                here,
                path,
                unvisited = [],
                potential,
                neighbors,
                next;
            for(j = 0; j < x + 1; j++)
            {
                horizontal[j] = [];
            }
            for(j = 0; j < y + 1; j++)
            {
                vertical[j] = [];
                here = [Math.floor(Math.random() * x), Math.floor(Math.random() * y)];
                path = [here];
            }
            for(j = 0; j < x + 2; j++)
            {
                unvisited[j] = [];
                for(k = 0; k < y + 1; k++)
                {
                    unvisited[j].push(j > 0 && j < x + 1 && k > 0 && (j != here[0] + 1 || k != here[1] + 1));
                }
            }
            while(0 < n)
            {
                potential = [
                    [here[0] + 1, here[1]], [here[0], here[1] + 1],
                    [here[0] - 1, here[1]], [here[0], here[1] - 1]
                ];
                neighbors = [];
                for(j = 0; j < 4; j++)
                {
                    if(unvisited[potential[j][0] + 1][potential[j][1] + 1])
                    {
                        neighbors.push(potential[j]);
                    }
                }

                if(neighbors.length)
                {
                    n = n - 1;
                    next = neighbors[Math.floor(Math.random() * neighbors.length)];
                    unvisited[next[0] + 1][next[1] + 1] = false;
                    if(next[0] == here[0])
                    {
                        horizontal[next[0]][(next[1] + here[1] - 1) / 2] = true;
                    }
                    else
                    {
                        vertical[(next[0] + here[0] - 1) / 2][next[1]] = true;
                    }
                    path.push(here = next);
                }
                else
                {
                    here = path.pop();
                }
            }
            this.maze = {x: x, y: y, horizontal: horizontal, vertical: vertical};
            return this.maze;
        };

        this.setRenderer = function(val)
        {
            this.renderer = val;
            return this;
        };

        this.getRenderer = function()
        {
            return this.renderer;
        };

        this.render = function()
        {
            this.renderer.render(this.maze);
        };
    };

    var Game = function(maze, opts)
    {
        this.maze = maze;
        this.opts = opts || {};
        this.renderer = maze.getRenderer();
        this.ballSize = Math.floor(this.renderer.blockSize / 2 * 0.7);
        this.onFinish = opts.onFinish || function()
        {
        };
        this.playerImg = new Image();
        this.playerImg.onload = function()
        {
            _this.playerImgWidth = this.width;
            _this.playerImgHeight = this.height;
            this.style.display = "none";
            _this.render();
        };
        this.playerImg.src = this.opts.playerImg;

        var _this = this,
            renderBall = function renderBall(x, y)
            {
                var width = _this.playerImgWidth,
                    height = _this.playerImgHeight,
                    xPosition = Math.floor(x - (width / 2)),
                    yPosition = Math.floor(y - (height / 2));

                _this.renderer.ctx.drawImage(_this.playerImg, xPosition, yPosition);
            },
            calculateBallPosition = function calculateBallPosition(currentPosition)
            {
                var width = _this.renderer.width,
                    height = _this.renderer.height,
                    xLength = Math.floor(width / _this.maze.x),
                    yLength = Math.floor(height / _this.maze.y),
                    x = Math.floor(currentPosition[0] * xLength + xLength / 2),
                    y = Math.floor(currentPosition[1] * yLength + yLength / 2);

                return [x, y];
            },
            canMove = function canMove(direction, currentPosition)
            {
                var newPosition, toCheck;

                switch(direction)
                {
                    case "left":
                        newPosition = [currentPosition[0] - 1, currentPosition[1]];
                        break;

                    case "right":
                        newPosition = [currentPosition[0] + 1, currentPosition[1]];
                        break;

                    case "up":
                        newPosition = [currentPosition[0], currentPosition[1] - 1];
                        break;

                    case "down":
                        newPosition = [currentPosition[0], currentPosition[1] + 1];
                        break;
                }

                if(newPosition[0] < 0 || newPosition[0] > _this.maze.x)
                {
                    return false;
                }
                if(newPosition[1] < 0 || newPosition[1] > _this.maze.y)
                {
                    return false;
                }

                switch(direction)
                {
                    case "left":
                        toCheck = _this.maze.maze.horizontal[newPosition[1]][newPosition[0]];
                        break;

                    case "right":
                        toCheck = _this.maze.maze.horizontal[currentPosition[1]][currentPosition[0]];
                        break;

                    case "up":
                        toCheck = _this.maze.maze.vertical[newPosition[1]][newPosition[0]];
                        break;

                    case "down":
                        toCheck = _this.maze.maze.vertical[currentPosition[1]][currentPosition[0]];
                        break;
                }

                if(toCheck)
                {
                    return newPosition;
                }
                return false;
            };

        this.start = function()
        {
            this.currentPosition = [0, 0];
            this.finishPosition = [this.maze.x - 1, this.maze.y - 1];

            this.render();
        };

        this.render = function()
        {
            var ballPosition = calculateBallPosition(this.currentPosition);
            this.renderer.render();
            renderBall(ballPosition[0], ballPosition[1]);
        };

        this.move = function(direction)
        {
            var currentPosition = this.currentPosition,
                newPosition;

            if(newPosition = canMove(direction, currentPosition))
            {
                this.currentPosition = newPosition;
                this.render();

                if(newPosition[0] === this.finishPosition[0] && newPosition[1] === this.finishPosition[1])
                {
                    this.onFinish.call(this);
                }
            }
        }
    };

    window.Maze = {
        Maze: Maze,
        RendererCanvas: RendererCanvas,
        Game: Game
    };

}(window, document);
