module.exports = {
    models: {},
    sequelize: {},
    http: {},
    appKey: null,

    /**
     * Główna funkcja rozpoczynająca synchronizację notatek
     *
     * @param user_id
     * @param productcode
     * @param notes_timestamp
     * @param notes
     * @param groups
     * @param token
     * @param response
     */
    synchronizeNotes: function(user_id, productcode, notes_timestamp, notes, groups, token, response)
    {
        var models = this.models,
            http = this.http,
            appKey = this.appKey,
            latestNoteTimestamp = 0,
            notesWithErrorsDuringSynchronization = [],
            quantityOfProcessedNotes = 0,
            newNotesTimestamp = 0,
            idOfBookForSynchronization = 0,
            idsOfGroupsForUser = [],
            modifiedOrAddedNotesForGroup = [],
            modifiedOrAddedPrivateNotes = [],
            quantityOfNewNotes = 0,
            errors = JSON.parse(require('fs').readFileSync(__dirname + '/errors.json'));

        beginSynchronization();

        /**
         * Znajdź timestamp najnowszej z danych notatek
         */
        function findLatestNoteTimestamp() {
            notes.forEach(function(note) {
                if(latestNoteTimestamp < note.note_timestamp) {
                    latestNoteTimestamp = note.note_timestamp;
                }
            });
        }

        /**
         * Daj listę id grup użytkownika
         */
        function prepareGroupsIds() {
            groups.forEach(function(group) {
                idsOfGroupsForUser.push(group.id);
            });
        }

        /**
         * Sprawdź, czy grupa o podanym id jest wśród grup użytkownika
         *
         * @param group_id
         * @returns {boolean}
         */
        function isGroupInGroups(group_id) {
            var i = 0,
                exists = false;
            while(i < groups.length && !exists) {
                if(groups[i].id == group_id) {
                    exists = true;
                }
                i++;
            }
            return exists;
        }

        /**
         * Dodaj lub zmień stan notatki grupowej
         *
         * @param groupnote
         * @param notedetails
         * @param deleted
         */
        function insertModifiedOrAddedNoteForGroup(groupnote, notedetails, deleted) {
            if(!modifiedOrAddedNotesForGroup[groupnote.group_id]) {
                modifiedOrAddedNotesForGroup[groupnote.group_id] = [];
            }

            if(deleted) {
                modifiedOrAddedNotesForGroup[groupnote.group_id].push({
                    id: groupnote.note_id*1,
                    group_id: groupnote.group_id*1,
                    user_id: user_id*1,
                    page_id: groupnote.page_id,
                    task_id: groupnote.task_in_book_id,
                    note_timestamp: Math.floor(groupnote.note_timestamp.getTime()/1000),
                    deleted: true
                });
            } else {
                modifiedOrAddedNotesForGroup[groupnote.group_id].push({
                    id: groupnote.note_id*1,
                    group_id: groupnote.group_id*1,
                    user_id: user_id*1,
                    page_id: groupnote.page_id,
                    task_id: groupnote.task_in_book_id,
                    position_x: notedetails.position_x*1,
                    position_y: notedetails.position_y*1,
                    note: notedetails.note,
                    note_timestamp: Math.floor(groupnote.note_timestamp.getTime()/1000)
                });
            }
        }

        /**
         * Dodaj lub zmień stan notatki użytkownika
         *
         * @param usernote
         * @param notedetails
         * @param note
         * @param deleted
         */
        function insertModifiedOrAddedPrivateNote(usernote, notedetails, note, deleted) {
            if(deleted) {
                modifiedOrAddedPrivateNotes.push({
                    id: usernote.note_id*1,
                    group_id: note.group_id*1,
                    user_id: user_id*1,
                    page_id: usernote.page_id,
                    task_id: usernote.task_in_book_id,
                    note_timestamp: Math.floor(usernote.note_timestamp.getTime()/1000),
                    deleted: true
                });
            } else {
                modifiedOrAddedPrivateNotes.push({
                    id: usernote.note_id*1,
                    group_id: note.group_id*1,
                    user_id: user_id*1,
                    page_id: usernote.page_id,
                    task_id: usernote.task_in_book_id,
                    position_x: notedetails.position_x*1,
                    position_y: notedetails.position_y*1,
                    note: notedetails.note,
                    note_timestamp: Math.floor(usernote.note_timestamp.getTime()/1000)
                });
            }
        }

        /**
         * Właściwy start synchronizacji notatek
         */

        function beginSynchronization() {
            findLatestNoteTimestamp();
            prepareGroupsIds();

            findOrCreateBookAndContinueSynchronization();

            /**
             * Jeśli nie ma książki o takim productcode to ją stwórz
             */
            function findOrCreateBookAndContinueSynchronization() {
                models.Book.findOrCreate({productcode: productcode})
                    .success(function(book) {
                        idOfBookForSynchronization = book.id;
                        findAndEditOrCreateStateForBook();
                    })
                    .error(function(error) {
                        sendErrorResponseDirectly(12001, error);
                    });
            }

            /**
             * Jeśli dla danej książki i użytkownika nie ma wcześniejszego stanu to go stwórz
             * Funkcja edycji/tworzenia stanu inicjuje synchronizację notatek dla stanu
             */
            function findAndEditOrCreateStateForBook() {
                models.BookState.find({where: {user_id: user_id, book_id: idOfBookForSynchronization}})
                    .success(function(bookstate) {
                        if(bookstate != null) {
                            if(latestNoteTimestamp > Math.floor(bookstate.notes_timestamp.getTime()/1000)) {
                                newNotesTimestamp = latestNoteTimestamp;
                            } else {
                                newNotesTimestamp = Math.floor(bookstate.notes_timestamp.getTime()/1000);
                            }
                            editState(bookstate);
                        } else {
                            newNotesTimestamp = latestNoteTimestamp;
                            createNewState();
                        }
                    })
                    .error(function(error) {
                        sendErrorResponseDirectly(12000, error);
                    });
            }
        }

        /**
         * Stwórz nowy stan książki (user_id + book_id) i utwórz wszystkie przysłane notatki jako nowe
         */
        function createNewState() {
            if(notes.length > 0) {
                models.BookState.create({notes_timestamp: new Date(latestNoteTimestamp*1000), tasks_timestamp: 0, user_id: user_id, book_id: idOfBookForSynchronization})
                    .success(function(bookstate) {
                        addAllNewNotes();
                    })
                    .error(function(error) {
                        sendErrorResponseDirectly(12002, error);
                    });
            } else {
                checkEndOfSavingNotes(true);
            }
        }

        /**
         * Dodaj jako nowe wszystkie notatki, które nie mają id
         */
        function addAllNewNotes() {
            notes.forEach(function(note) {
                if(note.id == null) {
                    if(note.deleted) {
                        nextNoteProcessed();
                        errorNoteToResponse("8002", null, note);
                    } else {
                        createNewNote(note);
                    }
                } else {
                    nextNoteProcessed();
                    errorNoteToResponse("8005", null, note);
                }
            });
        }

        /**
         * Stwórz nową notatkę (NoteDetails, UserNote, [GroupNote])
         *
         * @param note
         */
        function createNewNote(note) {
            createNoteDetailsAndAssignItToUserAndGroup();

            /**
             * Stwórz rekord szczegółów notatki (współrzędne, treść) i przypisz go do rekordu notatki użytkownika (i grupowej)
             */
            function createNoteDetailsAndAssignItToUserAndGroup() {
                models.NoteDetails.create({position_x: note.position_x, position_y: note.position_y, note: note.note})
                    .success(function(notedetails) {
                        if(note.group_id && note.group_id != null) {
                            createGrupNoteAndUserNote(notedetails)
                        } else {
                            createUserNote(notedetails);
                        }
                    })
                    .error(function(error) {
                        nextNoteProcessed();
                        errorNoteToResponse("5020", error, note);
                    });
            }

            /**
             * Stwórz rekordy notatki użytkownika (i grupowej) i przypisz do nich id rekordu szczegółów notatki
             *
             * @param notedetails
             */
            function createGrupNoteAndUserNote(notedetails) {
                if(isGroupInGroups(note.group_id)) {
                    models.GroupNote.create({
                        page_id: note.page_id,
                        task_in_book_id: note.task_id,
                        note_timestamp: new Date(note.note_timestamp*1000),
                        note_id: notedetails.id,
                        book_id: idOfBookForSynchronization,
                        group_id: note.group_id,
                        deleted: false
                    })
                        .success(function(groupnote) {
                            insertModifiedOrAddedNoteForGroup(groupnote, notedetails, false);
                            createUserNote(notedetails);
                        })
                        .error(function(error) {
                            errorNoteToResponse("2020", error, note);
                            createUserNote(notedetails);
                        });
                } else {
                    errorNoteToResponse("8004", null, note);
                    createUserNote(notedetails);
                }
            }

            /**
             * Stwórz rekord notatki użytkownika
             *
             * @param notedetails
             */
            function createUserNote(notedetails) {
                models.UserNote.create({
                    page_id: note.page_id,
                    task_in_book_id: note.task_id,
                    note_timestamp: new Date(note.note_timestamp*1000),
                    note_id: notedetails.id,
                    book_id: idOfBookForSynchronization,
                    user_id: user_id,
                    deleted: false
                })
                    .success(function(usernote) {
                        insertModifiedOrAddedPrivateNote(usernote, notedetails, note, false);
                        quantityOfNewNotes++;
                        nextNoteProcessed();
                        checkEndOfSavingNotes(false);
                    })
                    .error(function(error) {
                        nextNoteProcessed();
                        errorNoteToResponse("3020", error, note);
                    });
            }
        }

        /**
         * Edytuj istniejący stan ksiazki i notatki do niego przypisane
         *
         * @param bookstate
         */
        function editState(bookstate) {
            bookstate.notes_timestamp = new Date(newNotesTimestamp*1000);
            bookstate.save()
                .success(function() {
                    if(notes.length > 0) {
                        notes.forEach(function(note) {
                            processNote(note);
                        });
                    } else {
                        checkEndOfSavingNotes(true);
                    }
                })
                .error(function(error) {
                    sendErrorResponseDirectly(12003, error);
                });

            /**
             * Przetwórz notatkę:
             * Stwórz nową, jeśli nie ma id lub edytuj istniejącą
             *
             * @param note
             */
            function processNote(note) {
                if(note.id == null || note.id == 0) {
                    createNewNoteIfNotDeleted(note);
                } else {
                    editNoteIfExistsAndIsNewest(note);
                }
            }

            /**
             * Dodaj nową notatkę jeśli nie ma flagi 'deleted'
             *
             * @param note
             */
            function createNewNoteIfNotDeleted(note) {
                if(note.deleted) {
                    nextNoteProcessed();
                    errorNoteToResponse("8002", null, note);
                } else {
                    createNewNote(note);
                }
            }

            /**
             * Edytuj istniejącą notatkę, ale tylko jeśli jej id rzeczywiście istnieje w bazie,
             * a jej timestamp jest nowszy niż ten z bazy
             *
             * @param note
             */
            function editNoteIfExistsAndIsNewest(note) {
                models.UserNote.find({where: {book_id: idOfBookForSynchronization, user_id: user_id, note_id: note.id}})
                    .success(function(usernote) {
                        if(usernote && usernote.id) {
                            if(usernote.note_timestamp < new Date(note.note_timestamp*1000)) {
                                editNote(note, usernote);
                            } else {
                                nextNoteProcessed();
                                checkEndOfSavingNotes(false);
                            }
                        } else {
                            nextNoteProcessed();
                            errorNoteToResponse("8003", null, note);
                        }
                    })
                    .error(function(error) {
                        nextNoteProcessed();
                        errorNoteToResponse("3030", error, note);
                    });
            }

            /**
             * Edytuj notatkę według przysłanych danych
             * Jeśli notatka ma flagę 'deleted' - usuń ją
             * W przeciwnym wypadku aktualizuj jej dane
             *
             * @param note
             * @param usernote
             */
            function editNote(note, usernote) {
                usernote.note_timestamp = new Date(note.note_timestamp*1000);
                if(note.deleted && note.deleted == true) {
                    deleteNote(usernote);
                } else {
                    editUserNoteAndGroupNote(usernote);
                }

                /**
                 * Dodaj notatkę do listy rzeczywiście zmienionych - lista używana do rozgłoszenia aktualizacji notatek
                 *
                 * @param usernote
                 * @param groupnote
                 * @param notedetails
                 */
                function insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails) {
                    var deleted = false;
                    if(!notedetails) {
                        notedetails = note;
                        notedetails.note_timestamp = new Date(notedetails.note_timestamp*1000);
                    }

                    if(groupnote) {
                        insertModifiedOrAddedNoteForGroup(groupnote, notedetails, deleted);
                        insertModifiedOrAddedPrivateNote(usernote, notedetails, note, deleted);
                    } else {
                        insertModifiedOrAddedPrivateNote(usernote, notedetails, note, deleted);
                    }
                }

                /**
                 * Edytuj szczegóły notatki jeśli się zmieniły
                 *
                 * @param usernote
                 * @param groupnote
                 */
                function editNoteDetails(usernote, groupnote) {
                    models.NoteDetails.find(note.id)
                        .success(function(notedetails) {
                            if(notedetails && notedetails.id) {
                                editNoteDetailsIfChanged(notedetails);
                            } else {
                                createNoteIfItWasDeletedPreviously(notedetails);
                            }
                        })
                        .error(function(error) {
                            nextNoteProcessed();
                            errorNoteToResponse("5030", error, note);
                        });

                    /**
                     * Jeśli notatka o podanym id została wcześniej usunięta, to przywróć ją i aktualizuj dane
                     *
                     * @param notedetails
                     */
                    function createNoteIfItWasDeletedPreviously(notedetails) {
                        models.UserNote.find({where: {user_id: user_id, book_id: idOfBookForSynchronization, note_id: note.id}})
                            .success(function(usernote) {
                                if(usernote && usernote.id) {
                                    createNoteDetailsForPreviouslyDeletedNote();
                                } else {
                                    insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                    nextNoteProcessed();
                                    checkEndOfSavingNotes(false);
                                }
                            })
                            .error(function(error) {
                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                nextNoteProcessed();
                                errorNoteToResponse("3030", error, note);
                            });
                    }

                    /**
                     * Stwórz rekord szczegółowych danych przywracanej notatki
                     */
                    function createNoteDetailsForPreviouslyDeletedNote() {
                        models.NoteDetails.create({id: note.id, position_x: note.position_x, position_y: note.position_y, note: note.note})
                            .success(function(notedetails) {
                                usernote.deleted = false;
                                usernote.note_id = notedetails.id;
                                usernote.save()
                                    .success(function() {
                                        if(note.group_id && note.group_id != null) {
                                            unsetGroupNoteDeleteFlagIfExists(notedetails);
                                        } else {
                                            insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                            nextNoteProcessed();
                                            checkEndOfSavingNotes(false);
                                        }
                                    })
                                    .error(function(error) {
                                        insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                        nextNoteProcessed();
                                        errorNoteToResponse("3010", error, note);
                                    });
                            })
                            .error(function(error) {
                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                nextNoteProcessed();
                                errorNoteToResponse("5020", error, note);
                            });
                    }

                    /**
                     * Przywróć notatkę grupową jeśli taka istnieje
                     *
                     * @param notedetails
                     */
                    function unsetGroupNoteDeleteFlagIfExists(notedetails) {
                        if(isGroupInGroups(note.group_id)) {
                            models.GroupNote.find({where: {book_id: idOfBookForSynchronization, note_id: note.id, group_id: note.group_id}})
                                .success(function(groupnote) {
                                    if(groupnote && groupnote.id) {
                                        groupnote.deleted = false;
                                        groupnote.note_id = notedetails.id;
                                        groupnote.save()
                                            .success(function() {
                                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                                nextNoteProcessed();
                                                checkEndOfSavingNotes(false);
                                            })
                                            .error(function(error) {
                                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                                nextNoteProcessed();
                                                errorNoteToResponse("2010", error, note);
                                            });
                                    } else {
                                        models.GroupNote.create({
                                            page_id: note.page_id,
                                            task_in_book_id: note.task_id,
                                            note_timestamp: new Date(note.note_timestamp*1000),
                                            note_id: notedetails.id,
                                            book_id: idOfBookForSynchronization,
                                            group_id: note.group_id,
                                            deleted: false
                                        })
                                            .success(function(groupnote) {
                                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                                nextNoteProcessed();
                                                checkEndOfSavingNotes(false);
                                            })
                                            .error(function(error) {
                                                insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                                nextNoteProcessed();
                                                errorNoteToResponse("2020", error, note);
                                            });
                                    }
                                })
                                .error(function(error) {
                                    insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                    nextNoteProcessed();
                                    errorNoteToResponse("2030", error, note);
                                });
                        } else {
                            insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                            nextNoteProcessed();
                            errorNoteToResponse("8004", null, note);
                        }
                    }

                    /**
                     * Aktualizuj dane szczegółów notatki jeśi się zmieniły
                     *
                     * @param notedetails
                     */
                    function editNoteDetailsIfChanged(notedetails) {
                        if(notedetails.position_x != note.position_x || notedetails.position_y != note.position_y || notedetails.note != note.note) {
                            notedetails.position_x = note.position_x;
                            notedetails.position_y = note.position_y;
                            notedetails.note = note.note;
                            notedetails.save()
                                .success(function() {
                                    insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                    nextNoteProcessed();
                                    checkEndOfSavingNotes(false);
                                })
                                .error(function(error) {
                                    insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                                    nextNoteProcessed();
                                    errorNoteToResponse("5010", error, note);
                                });
                        } else {
                            insertNotesToModifiedOrAddedList(usernote, groupnote, notedetails);
                            nextNoteProcessed();
                            checkEndOfSavingNotes(false);
                        }
                    }

                }

                /**
                 * Aktualizuj dane notatki użytkownika (i grupowej)
                 *
                 * @param usernote
                 */
                function editUserNoteAndGroupNote(usernote) {
                    usernote.task_in_book_id = note.task_id;
                    usernote.page_id = note.page_id;
                    usernote.save()
                        .success(function() {
                            editGroupNote(usernote);
                        })
                        .error(function(error) {
                            errorNoteToResponse("3010", error, note);
                        });
                }

                /**
                 * Aktualizuj notatkę grupową
                 *
                 * @param usernote
                 */
                function editGroupNote(usernote) {
                    models.GroupNote.find({where: {book_id: idOfBookForSynchronization, note_id: note.id, deleted: false}})
                        .success(function(groupnote) {
                            if(groupnote && groupnote.group_id) {
                                if(note.group_id == null) {
                                    setGroupNoteToDeletedAndEditNoteDetails(note, groupnote, usernote);
                                } else {
                                    groupnote.note_timestamp = new Date(note.note_timestamp*1000);
                                    if(groupnote.group_id == note.group_id) {
                                        groupnote.task_in_book_id = note.task_id;
                                        groupnote.page_id = note.page_id;
                                        groupnote.deleted = false;
                                        groupnote.save()
                                            .success(function() {
                                                editNoteDetails(usernote, groupnote);
                                            })
                                            .error(function(error) {
                                                errorNoteToResponse("2010", error, note);
                                                editNoteDetails(usernote, null);
                                            });
                                    } else {
                                        groupnote.deleted = true;
                                        groupnote.save()
                                            .success(function() {
                                                assignToAnotherGroupNoteAndEditNoteDetails(usernote);
                                            })
                                            .error(function(error) {
                                                errorNoteToResponse("2010", error, note);
                                                editNoteDetails(usernote, groupnote);
                                            });
                                    }
                                }
                            } else {
                                if(note.group_id == null) {
                                    editNoteDetails(usernote, null);
                                } else {
                                    assignToAnotherGroupNoteAndEditNoteDetails(usernote);
                                }
                            }

                            /**
                             * Przepisz notatkę do innej grupy
                             *
                             * @param usernote
                             */
                            function assignToAnotherGroupNoteAndEditNoteDetails(usernote) {
                                if(isGroupInGroups(note.group_id)) {
                                    models.GroupNote.create({
                                        page_id: note.page_id,
                                        book_id: idOfBookForSynchronization,
                                        group_id: note.group_id,
                                        task_in_book_id: note.task_id,
                                        note_id: note.id,
                                        note_timestamp: new Date(note.note_timestamp*1000),
                                        deleted: false
                                    })
                                        .success(function(groupnote) {
                                            editNoteDetails(usernote, groupnote);
                                        })
                                        .error(function(error) {
                                            errorNoteToResponse("2020", error, note);
                                            editNoteDetails(usernote, null);
                                        });
                                } else {
                                    errorNoteToResponse("8004", null, note);
                                    editNoteDetails(usernote, null);
                                }
                            }
                        })
                        .error(function(error) {
                            errorNoteToResponse("2030", error, note);
                            editNoteDetails(usernote, null);
                        });
                }

                /**
                 * Ustaw flagę 'deleted' notatki grupowej na true
                 *
                 * @param note
                 * @param groupnote
                 * @param usernote
                 */
                function setGroupNoteToDeletedAndEditNoteDetails(note, groupnote, usernote) {
                    groupnote.deleted = true;
                    groupnote.save()
                        .success(function() {
                        })
                        .error(function(error) {
                            errorNoteToResponse("2010", error, note);
                        });
                    editNoteDetails(usernote, groupnote);
                }

                /**
                 * Usuń szczegóły usuwanej notatki
                 *
                 * @param usernote
                 */
                function deleteNote(usernote) {
                    models.NoteDetails.find(note.id)
                        .success(function(notedetails) {
                            if(notedetails && notedetails.id) {
                                notedetails.destroy()
                                    .success(function() {
                                        deleteUserNoteAndGroupNote(usernote);
                                    })
                                    .error(function(error) {
                                        nextNoteProcessed();
                                        errorNoteToResponse("5040 ", error, note);
                                    });
                            } else {
                                deleteUserNoteAndGroupNote(usernote);
                            }
                        })
                        .error(function(error) {
                            nextNoteProcessed();
                            errorNoteToResponse("5030", error, note);
                        });
                }

                /**
                 * Ustaw flagę 'deleted' notatki użytkownika (i grupowej) na true
                 *
                 * @param usernote
                 */
                function deleteUserNoteAndGroupNote(usernote) {
                    usernote.deleted = true;
                    usernote.save()
                        .success(function() {
                            insertModifiedOrAddedPrivateNote(usernote, null, note, true);
                            models.GroupNote.find({where: {book_id: idOfBookForSynchronization, note_id: note.id, deleted: false}})
                                .success(function(groupnote) {
                                    if(groupnote && groupnote.id) {
                                        groupnote.note_timestamp = new Date(note.note_timestamp*1000);
                                        groupnote.deleted = true;
                                        groupnote.save()
                                            .success(function() {
                                                insertModifiedOrAddedNoteForGroup(groupnote, null, true);
                                                nextNoteProcessed();
                                                checkEndOfSavingNotes(false);
                                            })
                                            .error(function(error) {
                                                nextNoteProcessed();
                                                errorNoteToResponse("2010", error, note);
                                            });
                                    } else {
                                        nextNoteProcessed();
                                        checkEndOfSavingNotes(false);
                                    }
                                })
                                .error(function(error) {
                                    nextNoteProcessed();
                                    errorNoteToResponse("2030", error, note);
                                });
                        })
                        .error(function(error) {
                            nextNoteProcessed();
                            errorNoteToResponse("3010", error, note);
                        });
                }
            }
        }

        /**
         * Wyciągnij wszystkie notatki, które trzeba rozgłosić jako zaktualizowane
         */
        function getAllNewNotes() {
            var responseNotes = [];
            models.UserNote.findAll({where: {user_id: user_id, book_id: idOfBookForSynchronization, note_timestamp: {gt: new Date(notes_timestamp*1000)} }})
                .success(function(usernotes) {
                    models.GroupNote.findAll({where: {group_id: idsOfGroupsForUser, book_id: idOfBookForSynchronization, note_timestamp:{gt: new Date(notes_timestamp*1000)}}})
                        .success(function(groupnotes) {
                            var userNotesToPush = [],
                                groupNotesToPush = [],
                                users_ids = [];

                            if(groupnotes.length > 0) {
                                getGroupNotes();
                            } else {
                                if(usernotes.length > 0) {
                                    getUserNotes();
                                } else {
                                    pushNotesToResponseAndEndSynchronization();
                                }
                            }

                            /**
                             * Wyciągnij notatki grupowe do rozgłoszenia
                             */
                            function getGroupNotes() {
                                var processed = 0;
                                groupnotes.forEach(function(groupnote) {
                                    if(groupnote.deleted == false) {
                                        models.NoteDetails.find(groupnote.note_id)
                                            .success(function(note) {
                                                if(note && note.id) {
                                                    models.UserNote.find({where: {book_id: idOfBookForSynchronization, note_id: note.id}})
                                                        .success(function(usernote) {
                                                            if(usernote && usernote.user_id) {
                                                                var index = users_ids.indexOf(usernote.user_id);
                                                                if(index <= -1) {
                                                                    users_ids.push(usernote.user_id);
                                                                }
                                                                var groupNoteToPush = {
                                                                    id: groupnote.note_id,
                                                                    group_id: groupnote.group_id,
                                                                    user_id: usernote.user_id,
                                                                    page_id: groupnote.page_id,
                                                                    task_id: groupnote.task_in_book_id,
                                                                    position_x: note.position_x,
                                                                    position_y: note.position_y,
                                                                    note: note.note,
                                                                    note_timestamp: Math.floor(groupnote.note_timestamp.getTime()/1000)
                                                                };
                                                                groupNotesToPush.push(groupNoteToPush);
                                                                endIteration();
                                                            } else {
                                                                endIteration();
                                                            }
                                                        })
                                                        .error(function(error) {
                                                            endIteration();
                                                        });
                                                } else {
                                                    endIteration();
                                                }
                                            })
                                            .error(function(error) {
                                                endIteration();
                                            });
                                    } else {
                                        if(notes_timestamp > 0) {
                                            models.UserNote.find({where: {book_id: idOfBookForSynchronization, note_id: groupnote.note_id}})
                                                .success(function(usernote) {
                                                    if(usernote && usernote.user_id) {
                                                        var groupNoteToPush = {
                                                            id: groupnote.note_id,
                                                            group_id: groupnote.group_id,
                                                            user_id: usernote.user_id,
                                                            page_id: groupnote.page_id,
                                                            task_id: groupnote.task_in_book_id,
                                                            note_timestamp: Math.floor(groupnote.note_timestamp.getTime()/1000),
                                                            deleted: true
                                                        };
                                                        groupNotesToPush.push(groupNoteToPush);
                                                        endIteration();
                                                    } else {
                                                        endIteration();
                                                    }
                                                })
                                                .error(function(error) {
                                                    endIteration();
                                                });
                                        } else {
                                            endIteration();
                                        }
                                    }
                                });
                                function endIteration() {
                                    processed++;
                                    if(processed >= groupnotes.length) {
                                        if(usernotes.length > 0) {
                                            getUserNotes();
                                        } else {
                                            pushNotesToResponseAndEndSynchronization();
                                        }
                                    }
                                }
                            }

                            /**
                             * Wyciągnij notatki użytkownika do rozgłoszenia
                             */
                            function getUserNotes() {
                                var pos = 0;
                                usernotes.forEach(function(usernote) {
                                    if(!isNoteInArray(usernote, groupnotes)) {
                                        if(usernote.deleted == false) {
                                            models.NoteDetails.find(usernote.note_id)
                                                .success(function(note) {
                                                    if(note && note.id) {
                                                        var userNoteToPush = {
                                                            id: usernote.note_id,
                                                            user_id: user_id,
                                                            group_id: null,
                                                            page_id: usernote.page_id,
                                                            task_id: usernote.task_in_book_id,
                                                            position_x: note.position_x,
                                                            position_y: note.position_y,
                                                            note: note.note,
                                                            note_timestamp: Math.floor(usernote.note_timestamp.getTime()/1000)
                                                        };
                                                        userNotesToPush.push(userNoteToPush);

                                                        endIteration();
                                                    } else {
                                                        endIteration();
                                                    }
                                                })
                                                .error(function(error) {
                                                    endIteration();
                                                });
                                        } else {
                                            if(notes_timestamp > 0) {
                                                var userNoteToPush = {
                                                    id: usernote.note_id,
                                                    user_id: user_id,
                                                    group_id: null,
                                                    page_id: usernote.page_id,
                                                    task_id: usernote.task_in_book_id,
                                                    note_timestamp: Math.floor(usernote.note_timestamp.getTime()/1000),
                                                    deleted: true
                                                };
                                                userNotesToPush.push(userNoteToPush);
                                                endIteration();
                                            } else {
                                                endIteration();
                                            }

                                        }
                                    } else {
                                        endIteration();
                                    }
                                });
                                function endIteration() {
                                    pos++;
                                    if(pos >= usernotes.length) {
                                        pushNotesToResponseAndEndSynchronization();
                                    }
                                }
                            }

                            /**
                             * Przygotuj odpowiedź na zakończenie synchronizacji
                             */
                            function pushNotesToResponseAndEndSynchronization() {
                                var options = {
                                    host: 'sso.vm.pl',
                                    path: '/api',
                                    method: 'POST',
                                    headers: {
                                        'Content-Type': 'application/json'
                                    }
                                };

                                var requestForAuthorization = http.request(options,function(response) {
                                    response.on('data',function(data){

                                        data = JSON.parse(data);
                                        if(data.users && data.users.length > 0) {
                                            for(var i = 0; i < groupNotesToPush.length; i++) {
                                                var j = 0;
                                                while(j < data.users.length) {
                                                    if(groupNotesToPush[i].user_id == data.users[j].user_id) {
                                                        groupNotesToPush[i].user_name = data.users[j].name;
                                                        groupNotesToPush[i].user_surname = data.users[j].surname;
                                                        j = data.users.length;
                                                    }
                                                    j++;
                                                }
                                            }
                                            responseNotes = userNotesToPush.concat(groupNotesToPush);
                                            lastStepBeforeSendingSynchronizedNotesResponse();
                                        } else {
                                            responseNotes = userNotesToPush.concat(groupNotesToPush);
                                            lastStepBeforeSendingSynchronizedNotesResponse();
                                        }
                                    });
                                });

                                requestForAuthorization.on('error', function(error) {
                                    responseNotes = userNotesToPush.concat(groupNotesToPush);
                                    lastStepBeforeSendingSynchronizedNotesResponse();
                                });

                                requestForAuthorization.write(JSON.stringify({
                                    appKey: appKey,
                                    method: "user_get_names",
                                    params: {
                                        users: users_ids
                                    }
                                }));

                                requestForAuthorization.end();

                            }

                            /**
                             * Funkcja będący dodatkowym poziomem abstrakcji przed wysłaniem odpowiedzi -
                             * tutaj było/może być dodatkowe przetwarzanie odpowiedzi
                             */
                            function lastStepBeforeSendingSynchronizedNotesResponse() {
                                endSynchronization(responseNotes);
                            }

                            /**
                             * Czy notatka znajduje się w podanej tablicy
                             * @param note
                             * @param array
                             * @returns {boolean}
                             */
                            function isNoteInArray(note, array) {
                                if(array.length > 0) {
                                    var i = 0,
                                        found = false;
                                    while(i < array.length) {
                                        if(note.note_id == array[i].note_id) {
                                            i = array.length-1;
                                            found = true;
                                        }
                                        i++;
                                    }
                                    return found;
                                } else {
                                    return false;
                                }
                            }
                        })
                        .error(function(error) {
                            sendErrorResponseDirectly(12004, error);
                        });
                })
                .error(function(error) {
                    sendErrorResponseDirectly(12005, error);
                });
        }

        /**
         * Zakoncz synchronizacje - wyslij odpowiedz
         */
        function endSynchronization(responseNotes) {
            response.send({notes_timestamp: newNotesTimestamp, productcode: productcode, notes: responseNotes, error_notes: notesWithErrorsDuringSynchronization});
            var groups_ids = [];
            console.log(modifiedOrAddedNotesForGroup);
            modifiedOrAddedNotesForGroup.forEach(function(groupnote, key) {
                if(groupnote && groupnote[0]) {
                    groups_ids[key] = key;
                }
            });
            response.pushModifiedNotes(user_id, groups_ids, modifiedOrAddedPrivateNotes, modifiedOrAddedNotesForGroup, token, productcode, notes_timestamp);
        }


        /**
         * Wyślij ogólny błąd pakietu z notatkami
         *
         * @param errorCode
         * @param errorMessage
         */
        function sendErrorResponseDirectly(errorCode, errorMessage) {
            var errorMessageToSend = "";
            try {
                errorMessageToSend = errors[errorCode.toString()];
            } catch(e) {
                try {
                    errorMessageToSend = errors["default"];
                } catch (e) {
                    errorMessageToSend = "General default error message. Missing file with errors.";
                }
            }
            response.errors.push(errorCode + ' | ' + errorMessage);
            response.send({error: {code: errorCode, message: errorMessageToSend}});
        }

        /**
         * Sprawdź warunek stopu synchronizacji notatek i
         *
         * @param instantFinish - ustawienie true spowoduje zakończenie działania metody i wysłanie takiej odpowiedzi jaka jest zbudowana aktualnie
         */
        function checkEndOfSavingNotes(instantFinish) {
            if(quantityOfProcessedNotes >= notes.length || instantFinish) {
                incrementUserQuantityOfAddedNotesAndThenMake(getAllNewNotes);
            }
        }

        /**
         * Aktualizuj tabelkę z liczbą notatek dla każdego użytkownika
         *
         * @param nextFunction
         */
        function incrementUserQuantityOfAddedNotesAndThenMake(nextFunction) {
            models.UserNotesQuantity.findOrCreate({user_id: user_id})
                .success(function(notesQuantity) {
                    notesQuantity.notes_quantity += quantityOfNewNotes;
                    notesQuantity.save()
                        .success(function() {
                            nextFunction();
                        })
                        .error(function(error) {
                            nextFunction();
                        })
                })
                .error(function(error) {
                    nextFunction();
                });
        }

        function nextNoteProcessed() {
            quantityOfProcessedNotes++;
        }

        /**
         * Dodaj notatkę do listy błędnych notatek, które zostały dodane częściowo lub wcale
         *
         * @param errorCode
         * @param errorMessage
         * @param noteWithError
         * @param errorForGeneral
         */
        function errorNoteToResponse(errorCode, errorMessage, noteWithError, errorForGeneral) {
            response.errors.push(errorCode + ' | ' + errorMessage);

            if(!errorForGeneral) {
                var indexOfNoteInErrorNotes = -1,
                    i = 0;
                while(i < notesWithErrorsDuringSynchronization.length) {
                    if(notesWithErrorsDuringSynchronization[i].id == noteWithError.id) {
                        indexOfNoteInErrorNotes = i;
                        i = notesWithErrorsDuringSynchronization.length;
                    }
                    i++;
                }

                if(indexOfNoteInErrorNotes > -1) {
                    notesWithErrorsDuringSynchronization[indexOfNoteInErrorNotes].error.push(err_no);
                } else {
                    notesWithErrorsDuringSynchronization.push({id: noteWithError.id, page_id: noteWithError.page_id, task_id: noteWithError.task_id, position_x: noteWithError.position_x, position_y: noteWithError.position_y, note_timestamp: noteWithError.note_timestamp, error: [errorCode]});
                }

                checkEndOfSavingNotes(false);
            }
        }
    },

    /**
     * Pobierz liczbę notatek dla użytkownika
     *
     * @param user_ids
     * @param response
     */
    getUserNotesQuantity: function(user_ids, response) {
        var models = this.models,
            quantities = [];

        /**
         * Wyślij ogólny błąd pakietu z notatkami
         * @param errorCode
         * @param errorMessage
         */
        function sendErrorResponseDirectly(errorCode, errorMessage) {
            var errorMessageToSend = "";
            try {
                errorMessageToSend = errors[errorCode.toString()];
            } catch(e) {
                try {
                    errorMessageToSend = errors["default"];
                } catch (e) {
                    errorMessageToSend = "General default error message. Missing file with errors.";
                }
            }
            response.errors.push(errorCode + ' | ' + errorMessage);
            response.send({error: {code: errorCode, message: errorMessageToSend}});
        }

        if(!(user_ids instanceof Array)) {
            sendErrorResponseDirectly(12100, 'User_ids is not an Array.');
        } else {
            models.UserNotesQuantity.findAll({where:{user_id: user_ids}})
                .success(function(notesquantities) {
                    notesquantities.forEach(function(notesquantity) {
                        quantities.push({user_id: notesquantity.user_id, notes_quantity: notesquantity.notes_quantity});
                    });
                    if(user_ids.length > quantities.length) {
                        user_ids.forEach(function(user_id) {
                            if(!hasUserQuantity(user_id)) {
                                quantities.push({user_id: user_id, notes_quantity: 0});
                            }
                        });

                        function hasUserQuantity(user_id) {
                            var i = 0,
                                has = false;

                            while(i < quantities.length && !has) {
                                if(quantities[i].user_id == user_id) {
                                    has = true;
                                }
                                i++;
                            }

                            return has;
                        }
                    }

                    response.send(quantities);
                })
                .error(function(error) {
                    sendErrorResponseDirectly(12101, error);
                });
        }
    }
};