var express = require('express'),
    app = express(),
    http = require('http'),
    tasks = require('./tasks'),
    notes = require('./notes'),
    models = require('./models/index'),
    logs = require('./logs'),
    fs = require('fs'),
    parameters = JSON.parse(fs.readFileSync(__dirname + '/parameters.json')),
    errors = JSON.parse(fs.readFileSync(__dirname + '/errors.json')),
    server = http.createServer(app),
    io = require('socket.io').listen(server),
    clients = {},
    clientsByUserId = {},
    clientsLength = 0;

app.use(express.bodyParser());

app.use(function(err, req, res, next){
    sendError(10000, err, res);
});

app.get('/testNotes', function(req, res) {
    res.sendfile(__dirname + '/index.html');
});

function getGroupsForUserBySSO(data, reqResponse) {
    var options = createOptionsForHttpRequest(parameters.graBiznesowaHost, "/api/groups");

    var requestForGroups = http.request(options,function(response) {
        response.on('data',function(groupsData) {
            try {
                groupsData = JSON.parse(groupsData);
            } catch(e) {
                sendError(10001, e.name + " | " + e.message + '__( ' + groupsData + ' )__', reqResponse);
            }
            if(groupsData != null && groupsData.groups != null) {
                data.params.user_groups = groupsData.groups;
                callMethodOnNotes(data, reqResponse);
            } else {
                data.params.user_groups = [{id: 1}, {id: 2}];
                callMethodOnNotes(data, reqResponse);
            }
        });
    });

    requestForGroups.on('error', function(e) {
        sendError(10002, e, reqResponse);
    });

    var requestBody = createRequestBodyForHttpRequestForGetGroups("getGroups", data.token, data.user_id);

    if(requestBody.hasOwnProperty("error")) {
        sendError(10003, requestBody.error.name + " | " + requestBody.error.message, reqResponse);
    } else {
        requestForGroups.write(requestBody);
        requestForGroups.end();
    }
}

function authorizationBySSO(data, reqResponse) {
    var options = createOptionsForHttpRequest(parameters.ssoApiHost, "/api");

    var requestForAuthorization = http.request(options,function(response) {
        response.on('data',function(authorizationData) {
            try {
                authorizationData = JSON.parse(authorizationData);
            } catch(e) {
                sendError(10004, e.name + " | " + e.message + '__( ' + authorizationData + ' )__', reqResponse);
            }
            if(authorizationData && authorizationData.token_valid && authorizationData.token_valid == true) {
                getGroupsForUserBySSO(data, reqResponse);
            } else {
                sendError(10005, "", reqResponse);
            }
        });
    });

    requestForAuthorization.on('error', function(e) {
        sendError(10006, e, reqResponse);
    });

    var requestBody = createRequestBodyForHttpRequestForAuthorizationBySSO("user_check_token", {token: data.token, user_id: data.user_id});

    if(requestBody.hasOwnProperty("error")) {
        sendError(10007, requestBody.name + " | " + requestBody.message, reqResponse);
    } else {
        requestForAuthorization.write(requestBody);
        requestForAuthorization.end();
    }
}

function createOptionsForHttpRequest(host, restPath) {
    return {
        host: host,
        path: restPath,
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        }
    };
}

function createRequestBodyForHttpRequestForAuthorizationBySSO(method, params) {
    try {
        return JSON.stringify({
            appKey: parameters.appKey,
            method: method,
            params: params
        });
    } catch(e) {
        return {error: e};
    }
}

function createRequestBodyForHttpRequestForGetGroups(method, token, user_id) {
    try {
        return JSON.stringify({
            appKey: parameters.appKey,
            method: method,
            token: token,
            user_id: user_id
        });
    } catch(e) {
        return {error: e};
    }
}

function createRequestBodyForHttpRequestForGetUsers(method, token, user_id, params) {
    try {
        return JSON.stringify({
            appKey: parameters.appKey,
            method: method,
            token: token,
            user_id: user_id,
            params: params
        });
    } catch(e) {
        return {error: e};
    }
}

app.post("/notes", function(request, reqResponse) {
    var data = request.body;

    if(data && data.method && data.method == 'getUserNotesQuantity') {
        logs.beginLog(data, reqResponse, afterLoggingUserNotesQuantity);

        function afterLoggingUserNotesQuantity(log_id, data, reqResponse) {
            reqResponse = new LoggedResponseObject(log_id, data.method, reqResponse);

            callMethodOnNotes(data, reqResponse);
        }
    } else {
        if(!(data && data.method && data.params && data.user_id && data.token)) {
            sendError(10008, null, reqResponse);
        } else {
            logs.beginLog(data, reqResponse, afterLogging);

            function afterLogging(log_id, data, reqResponse) {
                reqResponse = new LoggedResponseObject(log_id, data.method, reqResponse);

                authorizationBySSO(data, reqResponse);
            }
        }
    }
});

function callMethodOnNotes(data, response) {
    switch(data.method) {
        case "synchronizeNotes":
            notes.synchronizeNotes(data.user_id, data.params.productcode, data.params.notes_timestamp, data.params.notes, data.params.user_groups, data.token, response);
            break;
        case "getUserNotesQuantity":
            notes.getUserNotesQuantity(data.params.user_ids, response);
            break;
        default:
            sendError(10009, null, response);
            break;
    }
}


function sendError(errorCode, errorMessageForDatabase, response) {
    var errorMessage = "";
    try {
        errorMessage = errors[errorCode.toString()];
    } catch(e) {
        try {
            errorMessage = errors["default"];
        } catch (e) {
            errorMessage = "General default error message. Missing file with errors.";
        }
    }
    response.errors.push(errorCode + ":"  + errorMessageForDatabase);
    response.send({error: {code: errorCode, message: errorMessage}});

}

function LoggedResponseObject(log_id, method, response){
    return {
        log_id: log_id,
        errors: [],
        send: function(dataToResponse) {
            logs.endLog(this.log_id, dataToResponse, this.errors);
            response.send(dataToResponse);
        },
        pushModifiedNotes: function(user_id, groups_ids, modifiedOrAddedPrivateNotes, modifiedOrAddedNotesForGroup, token, productcode, notes_timestamp) {
            logs.beginLog({
                    user_id: user_id, method: "newNotes", broadcast:
                        {forGroups: modifiedOrAddedNotesForGroup, forSelfUser: modifiedOrAddedPrivateNotes}
                },
                response, endLoggingBroadcast);

            function endLoggingBroadcast(log_id, data, responseObject) {
                logs.endLog(log_id, data, null);
                responseObject.pushModifiedNotes(user_id, groups_ids, modifiedOrAddedPrivateNotes, modifiedOrAddedNotesForGroup, token, productcode, notes_timestamp);
            }
        }
    }
}

function SocketResponseObject(socket_id, event_name){
    return {
        socket_id: socket_id,
        event_name: event_name,
        send: function(dataToResponse) {
            if(dataToResponse.hasOwnProperty('error')) {
                io.sockets.in(this.socket_id).emit(this.event_name + 'Error', dataToResponse);
            } else {
                io.sockets.in(this.socket_id).emit(this.event_name + 'Success', dataToResponse);
            }
        },
        pushModifiedNotes: function(user_id, groups_ids, modifiedOrAddedPrivateNotes, modifiedOrAddedNotesForGroup, token, productcode, notes_timestamp) {
            var current_socket_id = this.socket_id;
            pushNotesToGroups();
            pushNotesToSelfUser();

            function pushNotesToSelfUser() {
                if(clientsByUserId[user_id]) {
                    clientsByUserId[user_id].forEach(function(socket_id) {
                        if(socket_id != current_socket_id) {
                            io.sockets.in(socket_id).emit("newNotes", {notes: modifiedOrAddedPrivateNotes});
                        }
                    });
                }
            }

            function pushNotesToGroups() {
                groups_ids.forEach(function(group_id) {
                    var options = createOptionsForHttpRequest(parameters.graBiznesowaHost, "/api/groups"),
                        requestBody = createRequestBodyForHttpRequestForGetUsers("getUsers", token, user_id, {group_id: group_id}),
                        requestForGetUsers = http.request(options,function(response) {
                            response.on('data',function(usersData) {
                                try {
                                    usersData = JSON.parse(usersData);
                                } catch(e) {
                                    console.log("GetUsers json parse error: " + e);
                                }
                                if(usersData != null && usersData.users != null) {
                                    usersData.users.forEach(function(user) {
                                        if(user.id != user_id && modifiedOrAddedNotesForGroup[group_id] != null && clientsByUserId[user.id] && modifiedOrAddedNotesForGroup[group_id]) {
                                            clientsByUserId[user.id].forEach(function(socket_id) {
                                                io.sockets.in(socket_id).emit("newNotes", {notes_timestamp: notes_timestamp, productcode: productcode, notes: modifiedOrAddedNotesForGroup[group_id]});
                                            });
                                        }
                                    });
                                }
                            });
                        });

                    requestForGetUsers.on('error', function(e) {
                        console.log("GetUsers connect error: " + e);
                    });

                    requestForGetUsers.write(requestBody);
                    requestForGetUsers.end();
                });
            }
        }
    }
}

io.sockets.on('connection', function (socket) {

    console.log('User connected.');

    socket.on('handshake', function(data) {
        if(data && data.user_id && data.user_id != null) {
            socket.join(socket.id);
            if(clients[socket.id] == null) {
                clients[socket.id] = data.user_id;
                if(clientsByUserId[data.user_id] && clientsByUserId[data.user_id].length) {
                    clientsByUserId[data.user_id].push(socket.id);
                } else {
                    clientsByUserId[data.user_id] = [];
                    clientsByUserId[data.user_id].push(socket.id);
                }
                console.log(clients);
                console.log(clientsByUserId);
                clientsLength++;
                io.sockets.in(socket.id).emit('handshakeSuccess', {socket_id: socket.id});
            } else {
                sendSocketErrorWithoutSocketResponseObject(socket.id, 11003, 'handshakeError');
            }
        } else {
            sendSocketErrorWithoutSocketResponseObject(socket.id, 11000, 'handshakeError');
        }
    });

    socket.on('notes', function(data) {
        if(!data || data.socket_id == null || data.user_id == null || data.token == null || data.method == null) {
            sendSocketErrorWithoutSocketResponseObject(socket.id, 11002, 'notesError');
        } else {
            if(clients[data.socket_id] == data.user_id) {
                var response = new SocketResponseObject(data.socket_id, 'notes');

                logs.beginLog(data, response, afterLogging);

                function afterLogging(log_id, data, reqResponse) {
                    reqResponse = new LoggedResponseObject(log_id, data.method, reqResponse);

                    authorizationBySSO(data, reqResponse);
                }
            } else {
                console.log(clients);
                sendSocketErrorWithoutSocketResponseObject(socket.id, 11001, 'notesError');
            }
        }
    });

    socket.on('disconnect', function () {
        if(clients[socket.id] != null) {
            delete clientsByUserId[clients[socket.id]];
            delete clients[socket.id];
            clientsLength--;
        }
        console.log('User disconnected.');
    });

    function sendSocketErrorWithoutSocketResponseObject(socket_id, errorCode, errorEventName) {
        var errorMessage = "";
        try {
            errorMessage = errors[errorCode.toString()];
        } catch(e) {
            errorMessage = errors["default"];
        }
        io.sockets.socket(socket_id).emit(errorEventName, {error: {code: errorCode, message: errorMessage}});
    }

});

server.listen(parameters.port, function () {
    console.log('socket.io and server listening at port %s', parameters.port);
    clients = {};
    initLogs();
    initNotes();
});

function initTasks() {
    tasks.models = models;
    tasks.sequelize = models.sequelize;
}

function initNotes() {
    notes.models = models;
    notes.sequelize = models.sequelize;
    notes.http = http;
    notes.appKey = parameters.appKey;
}

function initLogs() {
    logs.models = models;
    logs.sequelize = models.sequelize;
}
