/**
 * @author Mat Groves http://matgroves.com/ @Doormat23
 */

/**
 * The [pixi.js](http://www.pixijs.com/) module/namespace.
 *
 * @module PIXI
 */
 
/**
 * Namespace-class for [pixi.js](http://www.pixijs.com/).
 *
 * Contains assorted static properties and enumerations.
 *
 * @class PIXI
 * @static
 */
var PIXI = PIXI || {};

/**
 * @property {Number} WEBGL_RENDERER
 * @protected
 * @static 
 */
PIXI.WEBGL_RENDERER = 0;
/**
 * @property {Number} CANVAS_RENDERER
 * @protected
 * @static
 */
PIXI.CANVAS_RENDERER = 1;

/**
 * Version of pixi that is loaded.
 * @property {String} VERSION
 * @static 
 */
PIXI.VERSION = "v2.2.0";

/**
 * Various blend modes supported by pixi. IMPORTANT - The WebGL renderer only supports the NORMAL, ADD, MULTIPLY and SCREEN blend modes.
 * @property {Object} blendModes
 * @property {Number} blendModes.NORMAL
 * @property {Number} blendModes.ADD
 * @property {Number} blendModes.MULTIPLY
 * @property {Number} blendModes.SCREEN
 * @property {Number} blendModes.OVERLAY
 * @property {Number} blendModes.DARKEN
 * @property {Number} blendModes.LIGHTEN
 * @property {Number} blendModes.COLOR_DODGE
 * @property {Number} blendModes.COLOR_BURN
 * @property {Number} blendModes.HARD_LIGHT
 * @property {Number} blendModes.SOFT_LIGHT
 * @property {Number} blendModes.DIFFERENCE
 * @property {Number} blendModes.EXCLUSION
 * @property {Number} blendModes.HUE
 * @property {Number} blendModes.SATURATION
 * @property {Number} blendModes.COLOR
 * @property {Number} blendModes.LUMINOSITY
 * @static
 */
PIXI.blendModes = {
    NORMAL:0,
    ADD:1,
    MULTIPLY:2,
    SCREEN:3,
    OVERLAY:4,
    DARKEN:5,
    LIGHTEN:6,
    COLOR_DODGE:7,
    COLOR_BURN:8,
    HARD_LIGHT:9,
    SOFT_LIGHT:10,
    DIFFERENCE:11,
    EXCLUSION:12,
    HUE:13,
    SATURATION:14,
    COLOR:15,
    LUMINOSITY:16
};

/**
 * The scale modes that are supported by pixi.
 *
 * The DEFAULT scale mode affects the default scaling mode of future operations.
 * It can be re-assigned to either LINEAR or NEAREST, depending upon suitability.
 *
 * @property {Object} scaleModes
 * @property {Number} scaleModes.DEFAULT=LINEAR
 * @property {Number} scaleModes.LINEAR Smooth scaling
 * @property {Number} scaleModes.NEAREST Pixelating scaling
 * @static
 */
PIXI.scaleModes = {
    DEFAULT:0,
    LINEAR:0,
    NEAREST:1
};

// used to create uids for various pixi objects..
PIXI._UID = 0;

if(typeof(Float32Array) != 'undefined')
{
    PIXI.Float32Array = Float32Array;
    PIXI.Uint16Array = Uint16Array;

    // Uint32Array and ArrayBuffer only used by WebGL renderer
    // We can suppose that if WebGL is supported then typed arrays are supported too
    // as they predate WebGL support for all browsers:
    // see typed arrays support: http://caniuse.com/#search=TypedArrays
    // see WebGL support: http://caniuse.com/#search=WebGL
    PIXI.Uint32Array = Uint32Array;
    PIXI.ArrayBuffer = ArrayBuffer;
}
else
{
    PIXI.Float32Array = Array;
    PIXI.Uint16Array = Array;
}

// interaction frequency 
PIXI.INTERACTION_FREQUENCY = 30;
PIXI.AUTO_PREVENT_DEFAULT = true;

/**
 * @property {Number} PI_2
 * @static
 */
PIXI.PI_2 = Math.PI * 2;

/**
 * @property {Number} RAD_TO_DEG
 * @static
 */
PIXI.RAD_TO_DEG = 180 / Math.PI;

/**
 * @property {Number} DEG_TO_RAD
 * @static
 */
PIXI.DEG_TO_RAD = Math.PI / 180;

/**
 * @property {String} RETINA_PREFIX
 * @protected
 * @static
 */
PIXI.RETINA_PREFIX = "@2x";
//PIXI.SCALE_PREFIX "@x%%";

/**
 * If true the default pixi startup (console) banner message will be suppressed.
 *
 * @property {Boolean} dontSayHello
 * @default false
 * @static
 */
PIXI.dontSayHello = false;

/**
 * The default render options if none are supplied to
 * {{#crossLink "WebGLRenderer"}}{{/crossLink}} or {{#crossLink "CanvasRenderer"}}{{/crossLink}}.
 *
 * @property {Object} defaultRenderOptions
 * @property {Object} defaultRenderOptions.view=null
 * @property {Boolean} defaultRenderOptions.transparent=false
 * @property {Boolean} defaultRenderOptions.antialias=false
 * @property {Boolean} defaultRenderOptions.preserveDrawingBuffer=false
 * @property {Number} defaultRenderOptions.resolution=1
 * @property {Boolean} defaultRenderOptions.clearBeforeRender=true
 * @property {Boolean} defaultRenderOptions.autoResize=false
 * @static
 */
PIXI.defaultRenderOptions = {
    view:null,
    transparent:false,
    antialias:false, 
    preserveDrawingBuffer:false,
    resolution:1,
    clearBeforeRender:true,
    autoResize:false
}

PIXI.sayHello = function (type) 
{
    if(PIXI.dontSayHello)return;

    if ( navigator.userAgent.toLowerCase().indexOf('chrome') > -1 )
    {
        var args = [
            '%c %c %c Pixi.js ' + PIXI.VERSION + ' - ' + type + '  %c ' + ' %c ' + ' http://www.pixijs.com/  %c %c ♥%c♥%c♥ ',
            'background: #ff66a5',
            'background: #ff66a5',
            'color: #ff66a5; background: #030307;',
            'background: #ff66a5',
            'background: #ffc3dc',
            'background: #ff66a5',
            'color: #ff2424; background: #fff',
            'color: #ff2424; background: #fff',
            'color: #ff2424; background: #fff'
        ];

        console.log.apply(console, args);
    }
    else if (window['console'])
    {
        console.log('Pixi.js ' + PIXI.VERSION + ' - http://www.pixijs.com/');
    }

    PIXI.dontSayHello = true;
};
